package junit.extensions.jfcunit;

import java.io.FileDescriptor;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.net.InetAddress;
import java.security.Permission;

import java.awt.Toolkit;

import junit.framework.TestCase;
import javax.swing.JFrame;
import javax.swing.JButton;
import java.awt.event.ActionListener;
import java.awt.event.ActionEvent;

/**
 * Extend this class to create new tests for Swing based interfaces. This is a
 * subclass of TestCase, and therefore provides all of the facilities that you
 * would normally expect from TestCase.
 *
 * An important point to realize about coding with Swing classes is that most
 * methods are single threaded. This means that once a component has been
 * shown, its methods should normally only be accessed by the AWT thread.
 * While JFCTestCase runs its tests, the AWT thread is temporarily blocked to
 * prevent multi-threading issues. The effect of this is that any method calls
 * that you make on Swing components during a test will have no impact on the
 * GUI, until the AWT Thread is restarted. This can occur in one of two ways:
 * (1) The test runs to completion (2) The method "awtSleep" defined within
 * this class is called.
 *
 * @author Matt Caswell
 * @author <a href="mailto:vraravam@thoughtworks.com">Vijay Aravamudhan : ThoughtWorks Inc.</a>
 * @author Kevin Wilson
 */
public class JFCTestCase
    extends TestCase {
    /**
     * Time to wait between lock attempts.
     */
    private static final long DEFAULTLOCKWAIT = 25L;

    /**
     * Default time to sleep.
     */
    private static final long DEFAULTSLEEP = 100L;

    /*
     * Kick Start the WindowMonitor to hopefully.
     * Get it started before any popups are displayed.
     */
    static {
        WindowMonitor.start();
    }

    /**
     * Lock instance.
     */
    private Object m_lock = new Object();

    /**
     * Exception thrown by testcase.
     */
    private Throwable m_err;

    /**
     * Assert Exit was called.
     */
    private transient boolean m_assertExit = false;

    /**
     * The test helper to be used in testing. This has been made protected,
     * since most users already have a super class in their test hierarchy
     * where it is protected. Make sure to actually instantiate this in your
     * setUp() method or at the beginning of the test methods.
     *
     * @depricated use getHelper() instead;
     */
    private TestHelper m_helper;

    /**
     * Continuation flag.
     */
    private boolean m_cont = false;

    /**
     * True if exited.
     */
    private boolean m_exited = false;

    /**
     * True if runBare has begun.
     * Used when setting the assertExit.
     */
    private boolean m_runBare = false;

    /**
     * Forcibly wait for full time.
     */
    private boolean m_forcedWait = false;

    /**
     * Run the AWTThread free from interruption.
     */
    private volatile boolean m_pausingAWT = false;

    /**
     * Waiting flag.
     */
    private volatile boolean m_waiting = false;

    /**
     * Lock Wait time.
     */
    private long m_lockWait = DEFAULTLOCKWAIT;

    /**
     * Time to sleep.
     */
    private long m_sleepTime = DEFAULTSLEEP;

    /**
     * Constructs a new JFCTestCase (default Constructor if using JUnit ver 3.8).
     */
    protected JFCTestCase() {
        super();
    }

    /**
     * Constructs a new JFCTestCase.
     *
     * @param name The name of the test.
     */
    public JFCTestCase(final String name) {
        super(name);
    }

    /**
     * Set the value of the assert exit property.
     * When true the System.exit() call will throw
     * an junit.extensions.jfcunit.ExitException to
     * interrupt the exit process, the application
     * will not exit. The test case will
     * assert that system exit was called.
     *
     * @param aValue true if System.exit() should be asserted,
     * to allow the test case to complete.
     */
    public final void setAssertExit(final boolean aValue) {
        if (!m_assertExit && aValue && m_runBare) {
            System.setSecurityManager(createNoExitSecurityManager());
        }
        m_assertExit = aValue;
    }

    /**
     * Get the current value of the assertExit property.
     * see setAssertExit for a details description of the
     * operation.
     *
     * @return true if System.exit() should be asserted.
     * to allow the test case to complete.
     */
    public final boolean getAssertExit() {
        return m_assertExit;
    }

    /**
     * Sets the helper.
     *
     * @param helper TestHelper to be used.
     */
    public final void setHelper(final TestHelper helper) {
        this.m_helper = helper;
    }

    /**
     * Get the test helper.
     *
     * @return The test helper.
     */
    public final TestHelper getHelper() {
        return m_helper;
    }

    /**
     * Sets the sleepTime.
     *
     * @param time New value for sleepTime
     */
    public final void setSleepTime(final long time) {
        m_sleepTime = time;
    }

    /**
     * Suspends the test for up to a maximum period of time, and allows the
     * AWT Thread to resume temporarily. If the AWT Thread has not been paused,
     * this api has no effect. The time period is what was set by the user by
     * calling 'setSleepTime()' or 'awtSleep(sleepTime)'. If the notify occurs
     * before this time has elapsed, the thread will continue.
     */
    public final void awtSleep() {
        release();

        try {
            acquire();
        } catch (ExitException exe) {
            setError(exe);
        } catch (InterruptedException ie) {
            ; // shouldn't occur
        }
    }

    /**
     * Suspends the test for up to the specified (maximum) period of time, and allows the AWT
     * Thread to resume temporarily. If the AWT Thread has not been paused, this api has
     * no effect. Note that the sleep time for subsequent calls to awtSleep()
     * will use the time specified in the current call. If the notify occurs before
     * this time has elapsed, the thread will continue.
     *
     * @param sleepTime The number of ms to sleep
     */
    public final void awtSleep(final long sleepTime) {
        setSleepTime(sleepTime);
        awtSleep();
    }

    /**
     * Flush all events currently in the AWTEventQueue.
     * The event queue is left running afterwards.
     */
    public final void flushAWT() {
        pauseAWT();
        resumeAWT();
    }

    /**
     * Pause the awt event queue until it is released by
     * releaseAWT() or the end of the test is reached.
     */
    public final void pauseAWT() {
        m_pausingAWT = true;
        awtSleep();
    }

    /**
     * Resets the sleepTime to the default value (DEFAULTSLEEP).
     */
    public final void resetSleepTime() {
        m_sleepTime = DEFAULTSLEEP;
    }

    /**
     * Resume the awt event queue.
     */
    public final void resumeAWT() {
        awtSleep();
        m_pausingAWT = false;
        awtSleep();
    }

    /**
     * Suspends the test for a period of time, and allows the
     * AWT Thread to resume during this period if it has been
     * paused.
     *
     * @param delay The minimum amount of time the test case thread
     *              should be delayed.
     */
    public final void sleep(final long delay) {
        long releaseTime = System.currentTimeMillis() + delay;
        releaseTime += delay;

        while (System.currentTimeMillis() < releaseTime) {
            release();

            try {
                Thread.currentThread().sleep(m_lockWait);
                acquire();
            } catch (InterruptedException ie) {
                ; // shouldn't occur
            }
        }
    }

    /**
     * Get the AWT state.
     *
     * @return  boolean specifying whether the AWT Thread is still running or not.
     */
    public boolean isAWTRunning() {
        return !m_pausingAWT;
    }

    /**
     * Sets up, executes and then tears down a test.
     *
     * @exception Throwable exceptions thrown by code.
     */
    public void runBare() throws Throwable {
        SecurityManager oldsm = System.getSecurityManager();
        if (m_assertExit) {
            System.setSecurityManager(createNoExitSecurityManager());
        }
        m_runBare = true;

        try {
            try {
                runCode(
                    new Runnable() {
                    public void run() {
                        try {
                            JFCTestCase.this.setUp();
                            flushAWT();
                        } catch (Throwable e) {
                            setError(e);
                        }
                    }
                });

                try {
                    runTest();
                } catch (ExitException exe) {
                    // might need it here due to the test being run
                    setError(exe);
                }
            } finally {
                runCode(
                    new Runnable() {
                    public void run() {
                        try {
                            resumeAWT();
                            JFCTestCase.this.tearDown();
                            flushAWT();
                        } catch (Throwable e) {
                            setError(e);
                        }
                    }
                });
            }
        } finally {
            // put back the original security manager
            System.setSecurityManager(oldsm);

            if (m_assertExit) {
                assertTrue("System.exit() was not called", m_exited);
            }
        }
    }

    /**
     * Sets the err.
     *
     * @param error New value for err.
     */
    protected final void setError(final Throwable error) {
        if (error != null) {
            setContinue(false);
        }

        m_err = error;
    }

    /**
     * Returns the error.
     *
     * @return error.
     */
    protected final Throwable getError() {
        return m_err;
    }

    /**
     * Sets the forcedWait.
     *
     * @param forced New value for forcedWait.
     */
    protected final void setForcedWait(final boolean forced) {
        m_forcedWait = forced;
    }

    /**
     * Set the duration between checking the lock.
     * @param duration in milliseconds. A value less
     * than or equal to zero will reset the lockwait to the
     * default value.
     */
    protected final void setLockWait(final long duration) {
        if (duration <= 0) {
            m_lockWait = DEFAULTLOCKWAIT;
        } else {
            m_lockWait = duration;
        }
    }

    /**
     * Get the current duration between checking locks.
     * @return duration in milliseconds.
     */
    protected final long getLockWait() {
        return m_lockWait;
    }

    /**
     * Default setUp which does nothing. Override this to set up the environment
     * for your test.
     *
     * @exception  Exception   An instance of java.lang.Exception can be thrown
     */
    protected void setUp() throws Exception {
        super.setUp();
    }

    /**
     * This method creates a replacement for the default system manager.
     * The goal is to intercept System.exit calls and make it throw an
     * exception instead so that a System.exit in a task does not
     * fully terminate the test run.
     *
     * @return  An instance of a SecurityManager that will "consume" a call to System.exit()
     */
    protected final SecurityManager createNoExitSecurityManager() {
        return new JFCSecurityManager(System.getSecurityManager());
    }

    /**
     * Checks of the current test case has any errors.
     *
     * @return true if the current test case has any errors.
     */
    protected final boolean hasError() {
        return ((m_err != null) && !(m_err instanceof ExitException));
    }

    /**
     * Resets the err to the default value (null).
     */
    protected final void resetError() {
        setError(null);
    }

    /**
     * Resets the forcedWait to the default value (false).
     */
    protected final void resetForcedWait() {
        m_forcedWait = false;
    }

    /**
     * Run the code of the test case.
     * Note: This method will wait till the thread running code.run() completes
     * The above behavior is overridden if the setForcedWait() has been called with 'true'
     * Then the behavior is that the thread will wait for the time specified when awtSleep()
     * was called
     *
     * @param code Code which is to be executed.
     * @exception Throwable exceptions thrown by code.
     */
    protected synchronized void runCode(final Runnable code) throws Throwable {
        m_waiting = false;
        setContinue(true);
        resetError();
        TestHelper.setCurrentTestCase(this);
        new Thread(
            new Runnable() {
            public void run() {
                synchronized (m_lock) {
                    m_waiting = true;

                    try {
                        acquire();
                    } catch (InterruptedException ie) {
                        // shouldn't occur
                    }

                    try {
                        code.run();
                    } catch (Throwable exe) {
                        // might need it here due to the method invocation
                        setError(exe);
                    }

                    setContinue(false);
                    release();
                }
            }
        }

        ,
            getName()).start();

        // here we have to wait for the code.run() to block
        while (!m_waiting) {
            try {
                Thread.currentThread().sleep(m_lockWait);
            } catch (ExitException exe) {
                // might need it here due to the method invocation
                setError(exe);
            } catch (InterruptedException ie) {
                ; // shouldn't occur
            }
        }

        boolean condition = false;
        boolean firstTime = true;
        long elapsedTime = 0L;

        do {
            if (firstTime && m_pausingAWT) {
                // flush out any other dialog boxes, etc which were opened by the call to code.run()
                Toolkit.getDefaultToolkit().getSystemEventQueue().invokeAndWait(
                    new Runnable() {
                    public void run() {
                        synchronized (m_lock) {
                            release();

                            try {
                                acquire();
                            } catch (InterruptedException ie) {
                                ; // shouldn't occur
                            }
                        }
                    }
                });

                if (m_forcedWait) {
                    firstTime = false;
                }
            }

            try {
                elapsedTime += m_lockWait;
                Thread.currentThread().sleep(m_lockWait);
            } catch (InterruptedException ie) {
                // shouldn't occur
            }

            if (m_forcedWait) {
                condition = (elapsedTime <= getSleepTime());
            } else {
                condition = getContinue();
            }
        }
        while (condition);

        if (hasError()) {
            throw getError();
        }
    }

    /**
     * Executes a test.
     *
     * @exception Throwable exceptions thrown by code.
     */
    protected void runTest() throws Throwable {
        final Method runMethod;

        try {
            runMethod = getClass().getMethod(
                getName(),
                new Class[0]);
        } catch (NoSuchMethodException e) {
            fail("Method \"" + getName() + "\" not found");

            return;
        }

        if ((runMethod != null) && !Modifier.isPublic(runMethod.getModifiers())) {
            fail("Method \"" + getName() + "\" should be public");

            return;
        }

        runCode(
            new Runnable() {
            public void run() {
                try {
                    runMethod.invoke(
                        JFCTestCase.this,
                        new Object[0]);
                    flushAWT();
                } catch (InvocationTargetException e) {
                    setError(e.getTargetException());
                } catch (Throwable e) {
                    setError(e);
                }
            }
        });
    }

    /**
     * Default tearDown which does nothing. Override this to tear down the
     * environment for your test.
     *
     * @exception  Exception   An instance of java.lang.Exception can be thrown
     */
    protected void tearDown() throws Exception {
        super.tearDown();
    }

    /**
     * Sets the cont.
     *
     * @param c New value for cont.
     */
    private void setContinue(final boolean c) {
        m_cont = c;
    }

    /**
     * Returns the cont.
     *
     * @return cont.
     */
    private boolean getContinue() {
        return m_cont;
    }

    /**
     * Returns the value of sleepTime.
     *
     * @return sleepTime.
     */
    private long getSleepTime() {
        return m_sleepTime;
    }

    /**
     * Acquire AWT Thread Control.
     *
     * @exception InterruptedException  Thrown when a thread is waiting, sleeping, or otherwise paused
     *                                  for a long time and another thread interrupts it using the
     *                                  <code>interrupt</code> method in class <code>Thread</code>.
     */
    private void acquire() throws InterruptedException {
        // when any events (usually window close events) try to call System.exit,
        // we need to trap the exception here
        synchronized (m_lock) {
            try {
                if (m_pausingAWT) {
                    m_lock.wait();
                } else {
                    m_lock.wait(m_lockWait);
                }
            } catch (ExitException exe) {
                ; // don't do anything here
            }
        }
    }

    /**
     * Release the AWT Thread Control.
     */
    private void release() {
        // when any events (usually window close events) try to call System.exit,
        // we need to trap the exception here
        synchronized (m_lock) {
            try {
                m_lock.notify();
            } catch (ExitException exe) {
                ; // don't do anything here
            }
        }
    }

    /**
     * Pause the test until the Option Pane is acknowleged.
     */
    protected void pause() {
        final boolean[] finished = new boolean[] {false};
        final JFrame frame = new JFrame();
        JButton button = new JButton("Continue");
        button.addActionListener(new ActionListener() {
            /**
             * Action was performed.
             * @param e ActionEvent which was generated by the
             * continue button.
             */
            public void actionPerformed(final ActionEvent e) {
                finished[0] = true;
                frame.setVisible(false);
                frame.dispose();
            }
        });
        frame.getContentPane().add(button);
        frame.pack();
        frame.setLocationRelativeTo(null);
        frame.setVisible(true);
        while (!finished[0]) {
            sleep(1000);
        }
    }


    /**
     * Security manager which proxies all calls to the
     * original security manager. If the checkExit is
     * called and the setAssertExit(true) then the
     * SecurityException is thrown.
     */
    class JFCSecurityManager
        extends SecurityManager {

        /**
         * Original Security manager.
         */
        private SecurityManager m_delegate;

        /**
         * Constructor.
         * @param delegate Delegate security manager.
         */
        public JFCSecurityManager(final SecurityManager delegate) {
            super();
            m_delegate = delegate;
        }

        /**
         * @param status Status to be used.
         * @see java.lang.SecurityManager#checkExit(int)
         */
        public void checkExit(final int status) {
            if (m_assertExit) {
                m_exited = true;
                setContinue(false);
                throw new ExitException(status);
            } else {
                m_delegate.checkExit(status);
            }
        }

        /**
         * @param perm Permission to be checked.
         * @see java.lang.SecurityManager#checkPermission(java.security.Permission)
         */
        public void checkPermission(final Permission perm) {
            if (m_delegate != null) {
                m_delegate.checkPermission(perm);
            }
        }

        /**
         * Throws a <code>SecurityException</code> if the calling thread is not
         * permitted to accept a socket connection from the specified host and port
         * number.
         *
         * @param host the host name of the socket connection.
         * @param port the port number of the socket connection.
         */
        public void checkAccept(final String host, final int port) {
            if (m_delegate != null) {
                m_delegate.checkAccept(host, port);
            }
        }

        /**
         * Throws a <code>SecurityException</code> if the calling thread is not
         * allowed to modify the thread argument.
         *
         * @param t the thread to be checked.
         */
        public void checkAccess(final Thread t) {
            if (m_delegate != null) {
                m_delegate.checkAccess(t);
            }
        }

        /**
         * Throws a <code>SecurityException</code> if the calling thread is not
         * allowed to modify the thread group argument.
         *
         * @param g the thread group to be checked.
         */
        public void checkAccess(final ThreadGroup g) {
            if (m_delegate != null) {
                m_delegate.checkAccess(g);
            }
        }

        /**
         * Throws a <code>SecurityException</code> if the calling thread is not
         * allowed to access the AWT event queue.
         */
        public void checkAwtEventQueueAccess() {
            if (m_delegate != null) {
                m_delegate.checkAwtEventQueueAccess();
            }
        }

        /**
         * Throws a <code>SecurityException</code> if the calling thread is not
         * allowed to open a socket connection to the specified host and port number.
         *
         * @param host the host name port to connect to.
         * @param port the protocol port to connect to.
         */
        public void checkConnect(final String host, final int port) {
            if (m_delegate != null) {
                m_delegate.checkConnect(host, port);
            }
        }

        /**
         * Throws a <code>SecurityException</code> if the specified security context
         * is not allowed to open a socket connection to the specified host and port
         * number.
         *
         * @param host the host name port to connect to.
         * @param port the protocol port to connect to.
         * @param context a system-dependent security context.
         */
        public void checkConnect(final String host, final int port,
                                 final Object context) {
            if (m_delegate != null) {
                m_delegate.checkConnect(host, port, context);
            }
        }

        /**
         * Throws a <code>SecurityException</code> if the calling thread is not
         * allowed to create a new class loader.
         */
        public void checkCreateClassLoader() {
            if (m_delegate != null) {
                m_delegate.checkCreateClassLoader();
            }
        }

        /**
         * Throws a <code>SecurityException</code> if the calling thread is not
         * allowed to delete the specified file.
         *
         * @param file the system-dependent filename.
         */
        public void checkDelete(final String file) {
            if (m_delegate != null) {
                m_delegate.checkDelete(file);
            }
        }

        /**
         * Throws a <code>SecurityException</code> if the calling thread is not
         * allowed to create a subprocess.
         *
         * @param cmd the specified system command.
         */
        public void checkExec(final String cmd) {
            if (m_delegate != null) {
                m_delegate.checkExec(cmd);
            }
        }

        /**
         * Throws a <code>SecurityException</code> if the calling thread is not
         * allowed to dynamic link the library code specified by the string argument
         * file.
         *
         * @param lib the name of the library.
         */
        public void checkLink(final String lib) {
            if (m_delegate != null) {
                m_delegate.checkLink(lib);
            }
        }

        /**
         * Throws a <code>SecurityException</code> if the calling thread is not
         * allowed to wait for a connection request on the specified local port
         * number.
         *
         * @param port the local port.
         */
        public void checkListen(final int port) {
            if (m_delegate != null) {
                m_delegate.checkListen(port);
            }
        }

        /**
         * Throws a <code>SecurityException</code> if the calling thread is not
         * allowed to access members.
         *
         * @param clazz the class that reflection is to be performed on.
         * @param which type of access, PUBLIC or DECLARED.
         */
        public void checkMemberAccess(final Class clazz, final int which) {
            if (m_delegate != null) {
                m_delegate.checkMemberAccess(clazz, which);
            }
        }

        /**
         * Throws a <code>SecurityException</code> if the calling thread is not
         * allowed to use (join/leave/send/receive) IP multicast.
         *
         * @param maddr Internet group address to be used.
         */
        public void checkMulticast(final InetAddress maddr) {
            if (m_delegate != null) {
                m_delegate.checkMulticast(maddr);
            }
        }

        /**
         * Throws a <code>SecurityException</code> if the calling thread is not
         * allowed to access the package specified by the argument.
         *
         * @param pkg the package name.
         */
        public void checkPackageAccess(final String pkg) {
            if (m_delegate != null) {
                m_delegate.checkPackageAccess(pkg);
            }
        }

        /**
         * Throws a <code>SecurityException</code> if the calling thread is not
         * allowed to define classes in the package specified by the argument.
         *
         * @param pkg the package name.
         */
        public void checkPackageDefinition(final String pkg) {
            if (m_delegate != null) {
                m_delegate.checkPackageDefinition(pkg);
            }
        }

        /**
         * Throws a <code>SecurityException</code> if the specified security context
         * is denied access to the resource specified by the given permission.
         *
         * @param perm the specified permission
         * @param context a system-dependent security context.
         */
        public void checkPermission(final Permission perm, final Object context) {
            if (m_delegate != null) {
                m_delegate.checkPermission(perm, context);
            }
        }

        /**
         * Throws a <code>SecurityException</code> if the calling thread is not
         * allowed to initiate a print job request.
         */
        public void checkPrintJobAccess() {
            if (m_delegate != null) {
                m_delegate.checkPrintJobAccess();
            }
        }

        /**
         * Throws a <code>SecurityException</code> if the calling thread is not
         * allowed to access or modify the system properties.
         */
        public void checkPropertiesAccess() {
            if (m_delegate != null) {
                m_delegate.checkPropertiesAccess();
            }
        }

        /**
         * Throws a <code>SecurityException</code> if the calling thread is not
         * allowed to access the system property with the specified <code>key</code>
         * name.
         *
         * @param key a system property key.
         */
        public void checkPropertyAccess(final String key) {
            if (m_delegate != null) {
                m_delegate.checkPropertyAccess(key);
            }
        }

        /**
         * Throws a <code>SecurityException</code> if the calling thread is not
         * allowed to read from the specified file descriptor.
         *
         * @param fd the system-dependent file descriptor.
         */
        public void checkRead(final FileDescriptor fd) {
            if (m_delegate != null) {
                m_delegate.checkRead(fd);
            }
        }

        /**
         * Throws a <code>SecurityException</code> if the calling thread is not
         * allowed to read the file specified by the string argument.
         *
         * @param file the system-dependent file name.
         */
        public void checkRead(final String file) {
            if (m_delegate != null) {
                m_delegate.checkRead(file);
            }
        }

        /**
         * Throws a <code>SecurityException</code> if the specified security context
         * is not allowed to read the file specified by the string argument.
         *
         * @param file the system-dependent filename.
         * @param context a system-dependent security context.
         */
        public void checkRead(final String file, final Object context) {
            if (m_delegate != null) {
                m_delegate.checkRead(file, context);
            }
        }

        /**
         * Determines whether the permission with the specified permission target
         * name should be granted or denied.
         *
         * @param target the target name of the <code>SecurityPermission</code>.
         */
        public void checkSecurityAccess(final String target) {
            if (m_delegate != null) {
                m_delegate.checkSecurityAccess(target);
            }
        }

        /**
         * Throws a <code>SecurityException</code> if the calling thread is not
         * allowed to set the socket factory used by <code>ServerSocket</code> or
         * <code>Socket</code>, or the stream handler factory used by
         * <code>URL</code>.
         */
        public void checkSetFactory() {
            if (m_delegate != null) {
                m_delegate.checkSetFactory();
            }
        }

        /**
         * Throws a <code>SecurityException</code> if the calling thread is not
         * allowed to access the system clipboard.
         */
        public void checkSystemClipboardAccess() {
            if (m_delegate != null) {
                m_delegate.checkSystemClipboardAccess();
            }
        }

        /**
         * Returns <code>false</code> if the calling thread is not trusted to bring
         * up the top-level window indicated by the <code>window</code> argument.
         *
         * @param window the new window that is being created.
         * @return <code>true</code> if the calling thread is trusted to put up
         *   top-level windows; <code>false</code> otherwise.
         */
        public boolean checkTopLevelWindow(final Object window) {
            if (m_delegate != null) {
                return m_delegate.checkTopLevelWindow(window);
            }
            return false;
        }

        /**
         * Throws a <code>SecurityException</code> if the calling thread is not
         * allowed to write to the specified file descriptor.
         *
         * @param fd the system-dependent file descriptor.
         */
        public void checkWrite(final FileDescriptor fd) {
            if (m_delegate != null) {
                m_delegate.checkWrite(fd);
            }
        }

        /**
         * Throws a <code>SecurityException</code> if the calling thread is not
         * allowed to write to the file specified by the string argument.
         *
         * @param file the system-dependent filename.
         */
        public void checkWrite(final String file) {
            if (m_delegate != null) {
                m_delegate.checkWrite(file);
            }
        }

        /**
         * Creates an object that encapsulates the current execution environment.
         *
         * @return an implementation-dependent object that encapsulates sufficient
         *   information about the current execution environment to perform some
         *   security checks later.
         */
        public Object getSecurityContext() {
            if (m_delegate != null) {
                return m_delegate.getSecurityContext();
            }
            return "";
        }

        /**
         * Returns the thread group into which to instantiate any new thread being
         * created at the time this is being called.
         *
         * @return ThreadGroup that new threads are instantiated into
         */
        public ThreadGroup getThreadGroup() {
            if (m_delegate != null) {
                return m_delegate.getThreadGroup();
            }
            return null;
        }
    }

}
