package junit.extensions.jfcunit;

import junit.extensions.jfcunit.keyboard.JFCKeyStroke;

import java.awt.AWTException;
import java.awt.Component;
import java.awt.Point;
import java.awt.Robot;
import java.awt.event.InputEvent;

import java.lang.reflect.Method;


/**
 * Class that provides facilities for locating components within a GUI. To use
 * create a new instance of RobotTestHelper in your setUp. Windows can only be
 * located once they have been shown once to the user.
 *
 * After calling the helper.enterClickAndLeave() the JFCTestCase.sleep(long delay)
 * method should be called to prevent subsequent clicks from being treated as a
 * double click. A sleep dalay of 300ms can be used as a guideline.
 *
 * @author <a href="mailto:vraravam@thoughtworks.com">Vijay Aravamudhan : ThoughtWorks Inc.</a>
 * @author Kevin Wilson
 */
public class RobotTestHelper extends TestHelper {
    /**
     * Last location.
     */
    private static Point s_last = new Point(-1, -1);

    /**
     * Legal buttons.
     */
    private static final int LEGAL_BUTTONS = (InputEvent.BUTTON1_MASK
        | InputEvent.BUTTON2_MASK | InputEvent.BUTTON3_MASK);

    /**
     * Robot instance.
     */
    private Robot m_robot = null;

    // Without this delay events tend to get lost somewhere between
    // the robot and the event queue.

    /**
     * Mandatory delay to get thing working.
     */
    private int m_delay = 1;

    /**
     * Construct a new RobotTestHelper.
     *
     * @exception  AWTException   An instance of java.awt.AWTException can be thrown
     *                            when initializing the Robot.
     */
    public RobotTestHelper() throws AWTException {
        super();
        m_robot = new Robot();
    }

    /**
     * Process a key press event on a component.
     *
     * @param   ultimate    Not used in the robot.
     * @param   stroke      The JFCKeyStroke to be performed.
     */
    protected void keyPressed(final Component ultimate,
        final JFCKeyStroke stroke) {
        m_robot.delay(m_delay);
        m_robot.keyPress(stroke.getKeyCode());
    }

    /**
     * Process a key release event on a component.
     *
     * @param   ultimate    Not used in the robot.
     * @param   stroke      The JFCKeyStroke to be performed.
     */
    protected void keyReleased(final Component ultimate,
        final JFCKeyStroke stroke) {
        m_robot.delay(m_delay);
        m_robot.keyRelease(stroke.getKeyCode());
    }

    /**
     * Process a mouse move event on a component.
     *
     * @param   ultimate    Not used in the robot.
     * @param   x           The x coordinate of the point where the mouse is being moved to.
     * @param   y           The y coordinate of the point where the mouse is being moved to.
     */
    protected void mouseMoved(final Component ultimate, final int x, final int y) {
        Point dest = new Point(x, y);

        while ((s_last.x != dest.x) || (s_last.y != dest.y)) {
            s_last = calcNextPoint(
                    s_last,
                    dest,
                    getStep());
            m_robot.delay(m_delay);
            m_robot.mouseMove(s_last.x, s_last.y);
        }
    }

    /**
     * Process a mouse press event on a component.
     *
     * @param   ultimate          Not used in the robot.
     * @param   modifiers         The modifiers associated with this mouse event.
     * @param   click             The number of clicks associated with this mouse event.
     * @param   isPopupTrigger    Whether this mouse event will generate a popup.
     */
    protected void mousePressed(final Component ultimate, final int modifiers,
        final int click, final boolean isPopupTrigger) {
        m_robot.delay(m_delay);

        if (modifiers != 0) {
            int buttons = (modifiers & LEGAL_BUTTONS);
            m_robot.mousePress(buttons);
        }
    }

    /**
     * Process a mouse release event on a component.
     *
     * @param   ultimate          Not used in the robot.
     * @param   modifiers         The modifiers associated with this mouse event.
     * @param   click             The number of clicks associated with this mouse event.
     * @param   isPopupTrigger    Whether this mouse event will generate a popup.
     */
    protected void mouseReleased(final Component ultimate, final int modifiers,
        final int click, final boolean isPopupTrigger) {
        m_robot.delay(m_delay);

        if (modifiers != 0) {
            int buttons = (modifiers & LEGAL_BUTTONS);
            m_robot.mouseRelease(buttons);
        }
    }

    /**
     * Simulate rotating the mouseWheel.
     * Only supported in Java 1.4.
     *
     * @param ultimate Component to fire the events on.
     * @param amount Amount to rotate the wheel. Ignored on the robot.
     *               The amount will be extracted from the OS Mouse configuration.
     * @param wheelRotation Rotation of the mouse wheel.
     */
    protected void mouseWheel(final Component ultimate, final int amount,
        final int wheelRotation) {
        m_robot.delay(m_delay);

        try {
            Method method = Robot.class.getMethod(
                    "mouseWheel",
                    new Class[] {int.class});
            method.invoke(
                m_robot,
                new Object[] {new Integer(wheelRotation)});
        } catch (Exception ex) {
            throw new RuntimeException("Mouse Wheel not supported:"
                + ex.toString());
        }
    }
}
