package junit.extensions.jfcunit;

import java.awt.AWTEvent;
import java.awt.Frame;
import java.awt.Toolkit;
import java.awt.Window;
import java.awt.event.AWTEventListener;
import java.awt.event.WindowEvent;

import java.util.ArrayList;
import java.lang.ref.WeakReference;


/**
 * The WindowMonitor class is used to monitor for windows
 * that might not show up in Frame.getFrames(). So we must watch for these
 * windows to be opened.
 *
 * @author Kevin Wilson
 * @author <a href="mailto:vraravam@thoughtworks.com">Vijay Aravamudhan : ThoughtWorks Inc.</a>
 */
public final class WindowMonitor implements AWTEventListener {
    /**
     * A handle to this class instance.
     */
    private static WindowMonitor s_singleton = null;

    /**
     * The dispatch thread used to process the event queue.
     */
    private static DispatchThread s_dt = null;

    /**
     * Flag used to shutdown the window monitor.
     */
    private static volatile boolean s_running = true;

    /**
     * Event queue item or null if no pending items.
     * The event queue is a linked list.
     */
    private static WindowEventItem s_windowEventQueue = null;

    /**
     * Lock for the event queue and dispatch thread.
     */
    private static final Object QLOCK = new Object();

    /**
     * All Non-Popup Windows which have been found.
     */
    private static final ArrayList WINDOWS = new ArrayList();

    /**
     * Constructor.
     *
     * Private use the static start() method.
     */
    private WindowMonitor() {
        super();

        if (s_dt == null) {
            // Survey the current windows open.
            Frame[] frames = Frame.getFrames();

            for (int i = 0; i < frames.length; i++) {
                populateWindows(frames[i]);
            }

            s_dt = new DispatchThread("WindowMonitor-DispatchThread");
            s_dt.setDaemon(true);
            s_dt.start();
            Toolkit.getDefaultToolkit().addAWTEventListener(this,
                AWTEvent.WINDOW_EVENT_MASK);
        }
    }

    /**
     * Get all of the windows which are open.
     *
     * @return Window[] containing all of the windows
     * which have been instantiated.
     */
    public static Window[] getWindows() {
      ArrayList v = new ArrayList();
      ArrayList toRemove = new ArrayList();
      for (int i = 0; i < WINDOWS.size(); i++) {
        WeakReference r = (WeakReference) WINDOWS.get(i);
        Window w = (Window) r.get();
        if (w == null) {
          toRemove.add(r);
        } else {
          v.add(w);
        }
      }
      WINDOWS.removeAll(toRemove);

      return (Window[]) v.toArray(new Window[0]);
    }

    /**
     * Start the monitor.
     */
    public static void start() {
        getSingleton();
    }

    /**
     * Stop the monitor.
     */
    public static void stop() {
        getSingleton().stopInternal();
    }

    /**
     * Event dispatch implementation.
     *
     * @param theEvent AWTEvent which is to be processed.
     */
    public void eventDispatched(final AWTEvent theEvent) {
        processEvent(theEvent);
    }

    /**
     * return the singleton instance.
     * If the singleton has not been created,
     * then create one.
     *
     * @return WindowMonitor singleton instance.
     */
    private static WindowMonitor getSingleton() {
        if (s_singleton == null) {
            s_singleton = new WindowMonitor();
        }

        return s_singleton;
    }

    /**
     * Flush the AWT Event Queue.
     */
    private static void flushAWT() {
        // Invoke the below to insure that we have flushed the queue
        // of all current pending events.
        if (Toolkit.getDefaultToolkit().getSystemEventQueue().isDispatchThread()) {
            return;
        }

        try {
            Toolkit.getDefaultToolkit().getSystemEventQueue().invokeAndWait(
                new Runnable() {
                    public void run() {
                    }
                });
        } catch (Exception e) {
            // don't do anything here
        }
    }

    /**
     * Populate frames.
     * @param w Frame to be traversed.
     */
    private static void populateWindows(final Window w) {
        WINDOWS.add(new WeakReference(w));

        Window[] windows = w.getOwnedWindows();

        for (int i = 0; i < windows.length; i++) {
            populateWindows(windows[i]);
        }
    }

    /**
     * Processes the events by adding the event onto the
     * event queue to be processed by the dispatch thread.
     *
     * @param theEvent The event to be processed.
     */
    private static void processEvent(final AWTEvent theEvent) {
        switch (theEvent.getID()) {
        case WindowEvent.WINDOW_OPENED:
        case WindowEvent.WINDOW_ACTIVATED:
        case WindowEvent.WINDOW_DEACTIVATED:
        case WindowEvent.WINDOW_CLOSING:
            queueWindowEvent((WindowEvent) theEvent);

            break;

        default:
            break;
        }
    }

    /**
     * Queue the event.
     *
     * @param we WindowEvent to be queued.
     */
    private static void queueWindowEvent(final WindowEvent we) {
        synchronized (QLOCK) {
            WindowEventItem qi = new WindowEventItem(we);

            if (s_windowEventQueue == null) {
                s_windowEventQueue = qi;
            } else {
                WindowEventItem q = s_windowEventQueue;

                while (true) {
                    if (q.getNext() != null) {
                        q = q.getNext();
                    } else {
                        break;
                    }
                }

                q.setNext(qi);
            }

            QLOCK.notifyAll();
        }
    }

    /**
     * Stop the window monitor.
     */
    private void stopInternal() {
        s_running = false;

        //dt.interrupt();
    }

    /**
     * Handle all Component events in a separate thread. The reason for
     * this is that WindowEvents tend to be used to do lots of processing
     * on the Window hierarchy. As a result, it can frequently result
     * in deadlock situations.
     */
    private class DispatchThread extends Thread {
        /**
         * Constructor.
         *
         * @param name Name of the thread.
         */
        public DispatchThread(final String name) {
            super(name);
            setDaemon(true);
        }

        /**
         * Run the Dispatch thread.
         * Process the events recieved and track the windows.
         */
        public void run() {
            WindowEvent we = null;

            while (s_running) {
                while (WindowMonitor.s_windowEventQueue == null) {
                    synchronized (WindowMonitor.QLOCK) {
                        try {
                            WindowMonitor.QLOCK.wait(500);
                        } catch (ExitException exe) {
                            // don't do anything, but still catch it
                        } catch (InterruptedException e) {
                            // don't do anything, but still catch it
                        }
                    }
                }

                synchronized (WindowMonitor.QLOCK) {
                    we                                   = WindowMonitor.s_windowEventQueue
                        .getEvent();
                    WindowMonitor.s_windowEventQueue     = WindowMonitor.s_windowEventQueue
                        .getNext();

                    switch (we.getID()) {
                    case WindowEvent.WINDOW_OPENED:
                    case WindowEvent.WINDOW_ACTIVATED:
                    case WindowEvent.WINDOW_DEACTIVATED:

                        if (!containsKey(we.getWindow())) {
                            WINDOWS.add(
                                new WeakReference(we.getWindow()));
                        }

                        break;

                    case WindowEvent.WINDOW_CLOSING:
                        remove(we.getWindow());
                        break;

                    default:
                        break;
                    }

                    try {
                        WindowMonitor.QLOCK.notifyAll();
                    } catch (ExitException exe) {
                        // don't do anything, but still catch it
                    }
                }
            }
        }
    }

    /**
     * WindowEventItem is the basic type that handles the
     * queue for queueWindowEvent and the DispatchThread.
     */
    private static class WindowEventItem {
        /**
         * Event to be processed by the dispatch thread.
         */
        private WindowEvent m_event;

        /**
         * Next event item to be processed. Null if no more.
         */
        private WindowEventItem m_next;

        /**
         * Constructor.
         *
         * @param evt Event to construct the queue item for.
         */
        WindowEventItem(final WindowEvent evt) {
            m_event     = evt;
            m_next      = null;
        }

        /**
         * Get the event.
         *
         * @return the event.
         */
        public WindowEvent getEvent() {
            return m_event;
        }

        /**
         * Set the next entry.
         *
         * @param n next item.
         */
        public void setNext(final WindowEventItem n) {
            m_next = n;
        }

        /**
         * Get the next entry.
         *
         * @return WindowEventItem next item.
         */
        public WindowEventItem getNext() {
            return m_next;
        }
    }

    /**
     * Check to see if the key is in the current
     * list.
     *
     * @param key Window to search for.
     * @return boolean true if the window is in the list.
     */
    private boolean containsKey(final Window key) {
      ArrayList toRemove = new ArrayList();
      boolean result = false;
      for (int i = 0; i < WINDOWS.size() && !result; i++) {
        WeakReference r = (WeakReference) WINDOWS.get(i);
        Window w = (Window) r.get();
        if (w == null) {
          toRemove.add(r);
        } else if (w == key) {
          result = true;
        }
      }
      WINDOWS.removeAll(toRemove);
      return result;
    }

    /**
     * Remove the window from the list.
     * @param key Window to be removed.
     * @return Object Window which was removed.
     */
    private Object remove(final Window key) {
      ArrayList toRemove = new ArrayList();
      Object result = null;
      for (int i = 0; i < WINDOWS.size() && result == null; i++) {
        WeakReference r = (WeakReference) WINDOWS.get(i);
        Window w = (Window) r.get();
        if (w == null) {
          toRemove.add(r);
        } else if (w == key) {
          toRemove.add(r);
          result = w;
        }
      }
      WINDOWS.removeAll(toRemove);
      return result;
    }
}
