package junit.extensions.jfcunit.eventdata;

import junit.extensions.jfcunit.JFCTestCase;
import junit.extensions.jfcunit.xml.JFCXMLConstants;

import java.applet.Applet;

import java.awt.AWTEvent;
import java.awt.Component;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.Window;
import java.awt.event.KeyEvent;

import javax.swing.JComponent;


/**
 * Abstract data container class that holds most of the
 * data necessary for jfcUnit to fire events.
 *
 * @author <a href="mailto:vraravam@thoughtworks.com">Vijay Aravamudhan : ThoughtWorks Inc.</a>
 */
public abstract class AbstractEventData implements EventDataConstants {
    /**
     * The {@link JFCTestCase} on whose thread <code>awtSleep()</code>
     * has to be invoked.
     */
    private JFCTestCase m_testCase;

    /**
     * The reference point to place the mouse. Either custom set or
     * derived from the screen location.
     */
    private Point m_referencePoint;

    /**
     * The screen location to place the mouse.
     */
    private Point m_screenLocation;

    /**
     * State of event data. If invalid then data has not been loaded
     * into the event yet.
     */
    private boolean m_valid;

    /**
     * The modifier key values that need to be passed onto the event.
     */
    private int m_modifiers;

    /**
     * The position to place the mouse within the component.
     */
    private int m_position;

    /**
     * The wait time in ms between each event.
     */
    private long m_sleepTime;

    /**
     * Set the attribute value.
     *
     * @param testCase The new value of the attribute
     */
    public final void setTestCase(final JFCTestCase testCase) {
        m_testCase = testCase;
    }

    /**
     * Get the attribute value.
     *
     * @return JFCTestCase    The value of the attribute
     */
    public final JFCTestCase getTestCase() {
        return m_testCase;
    }

    /**
     * The source {@link java.awt.Component} on which the event has
     * to be fired.
     *
     * @return The source {@link java.awt.Component}
     */
    public abstract Component getComponent();

    /**
     * Get the default position.
     * @return Default position.
     */
    public int getDefaultPosition() {
        return EventDataConstants.DEFAULT_POSITION;
    }

    /**
     * Get the default sleep time.
     * @return ms sleep time.
     */
    public long getDefaultSleepTime() {
        return EventDataConstants.DEFAULT_SLEEPTIME;
    }

    /**
     * Get the screen location for the event.
     * @return Point screen location for the event.
     */
    public final Point getLocationOnScreen() {
        return m_screenLocation;
    }

    /**
     * Returns true if the keyCode refers to a meta character.
     *
     * @param keyCode code to be checked.
     * @return boolean true if the keyCode is a meta key.
     */
    public final boolean isMetaChar(final int keyCode) {
        switch (keyCode) {
        case KeyEvent.VK_ALT:
        case KeyEvent.VK_ALT_GRAPH:
        case KeyEvent.VK_CAPS_LOCK:
        case KeyEvent.VK_CONTROL:
        case KeyEvent.VK_NUM_LOCK:
        case KeyEvent.VK_META:
        case KeyEvent.VK_SHIFT:
            return true;

        default:
            return false;
        }
    }

    /**
     * Set the attribute value.
     *
     * @param modifiers The new value of the attribute
     */
    public void setModifiers(final int modifiers) {
        m_modifiers = modifiers;
    }

    /**
     * Get the attribute value.
     *
     * @return int    The value of the attribute
     */
    public final int getModifiers() {
        return m_modifiers;
    }

    /**
     * Get the default modifiers.
     * @return default modifiers for the event.
     */
    public abstract int getDefaultModifiers();

    /**
     * Set the attribute value.
     * @param position The value of the attribute
     */
    public void setPosition(final int position) {
        m_position = position;
    }

    /**
     * Get the attribute value.
     * @return int The value of the attribute
     */
    public int getPosition() {
        return m_position;
    }

    /**
     * Set the attribute value.
     * @param referencePoint The value of the attribute
     */
    public void setReferencePoint(final Point referencePoint) {
        m_referencePoint = referencePoint;
    }

    /**
     * Get the attribute value.
     * @return Point The value of the attribute
     */
    public Point getReferencePoint() {
        return m_referencePoint;
    }

    /**
     * Return the root {@link java.awt.Container} of the
     * current {@link java.awt.Component}.
     *
     * @return The root {@link java.awt.Container}.
     */
    public Component getRoot() {
        return getRoot(getComponent());
    }

    /**
     * Return the root {@link java.awt.Container} of the
     * specified {@link java.awt.Component}.
     *
     * @param c The {@link java.awt.Component} to obtain the root for.
     * @return The root {@link java.awt.Container}.
     */
    public Component getRoot(final Component c) {
        Component applet = null;

        for (Component p = c; p != null; p = p.getParent()) {
            if (p instanceof Window || !p.isLightweight()) {
                return p;
            }

            if (p instanceof Applet) {
                applet = p;
            }
        }

        return applet;
    }

    /**
     * Set the attribute value.
     *
     * @param sleepTime The new value of the attribute
     */
    public void setSleepTime(final long sleepTime) {
        m_sleepTime = sleepTime;
    }

    /**
     * Get the attribute value.
     *
     * @return long    The value of the attribute
     */
    public long getSleepTime() {
        return m_sleepTime;
    }

    // The next set of API's deal with creating event data
    // Based upon a AWT Event for recording.

    /**
     * Consume the {@link java.awt.Event}.
     *
     * @param ae Event to be consumed.
     * @return boolean true if the event was consumed.
     */
    public abstract boolean consume(final AWTEvent ae);

    /**
     * Set valid state.
     *
     * @param valid true if the event has data configured.
     */
    public final void setValid(final boolean valid) {
        this.m_valid = valid;
    }

    /**
     * Get the valid state.
     *
     * @return boolean true if the event is valid.
     */
    public final boolean isValid() {
        return m_valid;
    }

    /**
     * Compare to event datas and deterime if they are equal.
     *
     * @param o Object to be compared.
     * @return true if the events are the same.
     */
    public boolean equals(final Object o) {
        if (o == null) {
            return false;
        }

        if (o.getClass() != getClass()) {
            return false;
        }

        // isValid
        AbstractEventData data = (AbstractEventData) o;

        return (data.isValid() == isValid())
        && (data.getComponent() == getComponent())
        && (data.getModifiers() == getModifiers())
        && (data.getPosition() == getPosition())
        && (data.getReferencePoint() == getReferencePoint());
    }

    /**
     * Returns the hashCode of the contained component.
     * @return int hashCode.
     */
    public int hashCode() {
        Component comp = getComponent();

        if (comp != null) {
            return comp.hashCode();
        }

        return 1;
    }

    /**
     * Populate the Element with the attributes to recreate this
     * event data.
     * @param e Element to be populated.
     */
    public void populate(final org.w3c.dom.Element e) {
        if (getPosition() != getDefaultPosition()) {
            String position;

            switch (getPosition()) {
            case NORTH:
                position = "north";

                break;

            case SOUTH:
                position = "south";

                break;

            case EAST:
                position = "east";

                break;

            case WEST:
                position = "west";

                break;

            case NORTH_WEST:
                position = "northwest";

                break;

            case NORTH_EAST:
                position = "northeast";

                break;

            case SOUTH_WEST:
                position = "southwest";

                break;

            case SOUTH_EAST:
                position = "southeast";

                break;

            case CENTER:
                position = "center";

                break;

            case CUSTOM:
                position = "custom";

                break;

            case PERCENT:
                position = "percent";

                break;

            case OFFSET:
                position = "offset";

                break;

            default:
                position = "" + getPosition();

                break;
            }

            e.setAttribute(JFCXMLConstants.POSITION, position);
        }

        if (getModifiers() != getDefaultModifiers()) {
            String txt = getModifierText();

            if ((txt != null) && (txt.length() > 0)) {
                e.setAttribute(JFCXMLConstants.MODIFIERS, txt);
            }
        }

        if (getSleepTime() != getDefaultSleepTime()) {
            e.setAttribute(JFCXMLConstants.SLEEPTIME, "" + getSleepTime());
        }

        Point p = getReferencePoint();

        if (p != null) {
            e.setAttribute(JFCXMLConstants.REFERENCE, "" + p.x + "," + p.y);
        }
    }

    /**
     * Prepare the component to receive the {@link java.awt.Event}
     * by calling <code>requestFocus()</code> on the {@link java.awt.Component}
     * and calculating the screen location.
     *
     * @return true if the component is ready to receive
     * the {@link java.awt.Event}.
     */
    public boolean prepareComponent() {
        if (!isValidForProcessing(getComponent())) {
            return false;
        }

        JFCTestCase testCase = getTestCase();

        if (testCase != null) {
            // try to clear the event queue
            testCase.flushAWT();
        }

        Rectangle compRect = getComponent().getBounds();

        if (testCase != null) {
            // try to clear the event queue
            testCase.flushAWT();
            testCase.pauseAWT();
        }

        Point p = calculatePoint(compRect);

        if (getComponent() instanceof JComponent) {
            JComponent c   = (JComponent) getComponent();
            Rectangle  vis = c.getVisibleRect();

            if (!vis.contains(p)) {
                Rectangle newView = new Rectangle(p.x - (int) (vis.width / 2),
                    p.y - (int) (vis.height / 2), vis.width, vis.height);
                c.scrollRectToVisible(newView);
            }
        }

        Point screen = getComponent().getLocationOnScreen();
        screen.translate(p.x, p.y);
        setLocationOnScreen(screen);

        return true;
    }

    /**
     * Get the String description of the abstract event.
     *
     * @return String description of the event.
     */
    public String toString() {
        StringBuffer buf = new StringBuffer(1000);
        buf.append(getClass().getName());

        if (!isValid()) {
            buf.append(" invalid");

            return buf.toString();
        }

        buf.append(" testCase: " + getTestCase());
        buf.append(" modifiers: " + getModifierText());
        buf.append(" sleepTime: " + m_sleepTime);
        buf.append(" position:" + POSITIONSTRINGS[m_position]);
        buf.append(" refPoint: " + m_referencePoint);
        buf.append(" screenLoc: " + m_screenLocation);
        buf.append(" component: " + getComponent());

        return buf.toString();
    }

    /**
     * Set the screen location for the event.
     * @param screenLocation screenLocation for event.
     */
    protected final void setLocationOnScreen(final Point screenLocation) {
        m_screenLocation = screenLocation;
    }

    /**
     * Checks whether a component is valid for processing using jfcUnit.
     *
     * @param comp   The {@link java.awt.Component} to be tested.
     * @return boolean specifying whether this component is not
     * null and is showing.
     */
    protected boolean isValidForProcessing(final Component comp) {
        return ((comp != null) && comp.isShowing());
    }

    /**
     * A utility method to calculate the {@link java.awt.Point}
     * at which the events are to be fired based on the position
     * and reference-point specified.
     *
     * @param rect  The {@link java.awt.Rectangle} containing the
     * source component.
     * @return The new {@link java.awt.Point} at which the mouse
     * events have to be fired.
     */
    protected final Point calculatePoint(final Rectangle rect) {
        // if the user has set a specific point or if this
        // calculation has already been done once, then just return.
        if (m_position == CUSTOM) {
            return new Point(rect.x + m_referencePoint.x,
                rect.y + m_referencePoint.y);
        }

        int x = 0;
        int y = 0;

        if (m_position == PERCENT) {
            x     = rect.x + (int) ((rect.width * m_referencePoint.x) / 100);
            y     = rect.y + (int) ((rect.height * m_referencePoint.y) / 100);
        } else {
            // Calculate the Y position
            if ((m_position == NORTH)
                    || (m_position == NORTH_WEST)
                    || (m_position == NORTH_EAST)) {
                y = rect.y;
            } else if ((m_position == SOUTH)
                    || (m_position == SOUTH_WEST)
                    || (m_position == SOUTH_EAST)) {
                y = rect.y + rect.height;
            } else {
                y = rect.y + (int) (rect.height / 2);
            }

            // Calculate the X position
            if ((m_position == WEST)
                    || (m_position == NORTH_WEST)
                    || (m_position == SOUTH_WEST)) {
                x = rect.x;
            } else if ((m_position == EAST)
                    || (m_position == NORTH_EAST)
                    || (m_position == SOUTH_EAST)) {
                x = rect.x + rect.width;
            } else {
                x = rect.x + (int) (rect.width / 2);
            }
        }

        return new Point(x, y);
    }

    /**
     * Check if this event can consume the {@link java.awt.AWTEvent}.
     *
     * @param ae Event to be consumed.
     * @return boolean true if the event can be consumed.
     */
    protected boolean canConsume(final AWTEvent ae) {
        if (!isValid()) {
            return true;
        }

        return (ae.getSource().equals(getComponent()));
    }

    /**
     * Get the modifier text for the current modifiers.
     * @return String with modifiers separated by
     * the +. Ex: Alt+Shift
     */
    abstract String getModifierText();
}
