package junit.extensions.jfcunit.eventdata;

import java.util.EventListener;
import javax.accessibility.Accessible;

import java.awt.AWTEvent;
import java.awt.Component;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.event.MouseEvent;
import javax.swing.AbstractListModel;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JList;
import javax.swing.ListModel;
import javax.swing.event.ListDataListener;
import javax.swing.plaf.ComboBoxUI;
import javax.swing.plaf.basic.BasicComboPopup;

import org.w3c.dom.Element;
import junit.extensions.jfcunit.JFCTestCase;
import junit.extensions.jfcunit.finder.ComponentFinder;
import junit.extensions.jfcunit.finder.Finder;
import junit.extensions.jfcunit.xml.JFCXMLConstants;


/**
 * Data container class that holds all the data necessary for jfcUnit to fire mouse events.
 * This class is specific to events on a {@link javax.swing.JComboBox}.
 *
 * @author <a href="mailto:vraravam@thoughtworks.com">Vijay Aravamudhan : ThoughtWorks Inc.</a>
 */
public class JComboBoxMouseEventData extends AbstractMouseEventData {
    /**
     * The {@link JComboBox} on which to trigger the event.
     */
    private JComboBox m_comboBox;

    /**
     * The list view associated with the {@link JComboBox}.
     */
    private JList m_listView;

    /**
         * The zero-based index of the specific element on which to trigger the event.
     */
    private int m_elementIndex;

    /**
     * Constructor.
     */
    public JComboBoxMouseEventData() {
        super();
        setValid(false);
    }

    /**
     * Constructor.
     *
     * @param testCase The {@link JFCTestCase} on whose thread <code>awtSleep()</code> has to be invoked.
     * @param comboBox The {@link JComboBox} on which to trigger the event.
     * @param element   The specific element on which to trigger the event.
     * @param numberOfClicks
     *                  Number of clicks in the {@link MouseEvent} (1 for single-click, 2 for double clicks)
     */
    public JComboBoxMouseEventData(final JFCTestCase testCase,
        final JComboBox comboBox, final Object element, final int numberOfClicks) {
        this(testCase, comboBox, element, numberOfClicks, DEFAULT_SLEEPTIME);
    }

    /**
     * Constructor.
     *
     * @param testCase  The {@link JFCTestCase} on whose thread <code>awtSleep()</code> has to be invoked.
     * @param comboBox  The {@link JComboBox} on which to trigger the event.
     * @param element   The specific element on which to trigger the event.
     * @param numberOfClicks
     *                   Number of clicks in the {@link MouseEvent} (1 for single-click, 2 for double clicks)
     * @param sleepTime
     *                   The wait time in ms between each event.
     */
    public JComboBoxMouseEventData(final JFCTestCase testCase,
        final JComboBox comboBox, final Object element,
        final int numberOfClicks, final long sleepTime) {
        this(testCase, comboBox, element, numberOfClicks,
            DEFAULT_MOUSE_MODIFIERS, DEFAULT_ISPOPUPTRIGGER, sleepTime);
    }

    /**
     * Constructor.
     *
     * @param testCase  The {@link JFCTestCase} on whose thread <code>awtSleep()</code> has to be invoked.
     * @param comboBox  The {@link JComboBox} on which to trigger the event.
     * @param element   The specific element on which to trigger the event.
     * @param numberOfClicks
     *                   Number of clicks in the {@link MouseEvent} (1 for single-click, 2 for double clicks)
     * @param modifiers The modifier key values that need to be passed onto the event.
     * @param isPopupTrigger
     *                   boolean specifying whether this event will show a popup.
     * @param sleepTime
     *                   The wait time in ms between each event.
     */
    public JComboBoxMouseEventData(final JFCTestCase testCase,
        final JComboBox comboBox, final Object element,
        final int numberOfClicks, final int modifiers,
        final boolean isPopupTrigger, final long sleepTime) {
        this(testCase, comboBox, element, numberOfClicks, modifiers,
            isPopupTrigger, sleepTime, DEFAULT_POSITION, null);
    }

    /**
     * Constructor.
     *
     * @param testCase  The {@link JFCTestCase} on whose thread <code>awtSleep()</code> has to be invoked.
     * @param comboBox  The {@link JComboBox} on which to trigger the event.
     * @param element   The specific element on which to trigger the event.
     * @param numberOfClicks
     *                   Number of clicks in the {@link MouseEvent} (1 for single-click, 2 for double clicks)
     * @param modifiers The modifier key values that need to be passed onto the event.
     * @param isPopupTrigger
     *                   boolean specifying whether this event will show a popup.
     * @param sleepTime
     *                   The wait time in ms between each event.
     * @param referencePoint
     *                   The CUSTOM mouse position within the cell.
     */
    public JComboBoxMouseEventData(final JFCTestCase testCase,
        final JComboBox comboBox, final Object element,
        final int numberOfClicks, final int modifiers,
        final boolean isPopupTrigger, final long sleepTime,
        final Point referencePoint) {
        this(testCase, comboBox, element, numberOfClicks, modifiers,
            isPopupTrigger, sleepTime, CUSTOM, referencePoint);
    }

    /**
     * Constructor.
     *
     * @param testCase  The {@link JFCTestCase} on whose thread <code>awtSleep()</code> has to be invoked.
     * @param comboBox  The {@link JComboBox} on which to trigger the event.
     * @param element   The specific element on which to trigger the event.
     * @param numberOfClicks
     *                   Number of clicks in the {@link MouseEvent} (1 for single-click, 2 for double clicks)
     * @param modifiers The modifier key values that need to be passed onto the event.
     * @param isPopupTrigger
     *                   boolean specifying whether this event will show a popup.
     * @param sleepTime
     *                   The wait time in ms between each event.
     * @param position  The relative mouse position within the cell.
     */
    public JComboBoxMouseEventData(final JFCTestCase testCase,
        final JComboBox comboBox, final Object element,
        final int numberOfClicks, final int modifiers,
        final boolean isPopupTrigger, final long sleepTime, final int position) {
        this(testCase, comboBox, element, numberOfClicks, modifiers,
            isPopupTrigger, sleepTime, position, null);
    }

    /**
     * Constructor.
     *
     * @param testCase  The {@link JFCTestCase} on whose thread <code>awtSleep()</code> has to be invoked.
     * @param comboBox  The {@link JComboBox} on which to trigger the event.
     * @param element   The specific element on which to trigger the event.
     * @param numberOfClicks
     *                   Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     * @param modifiers The modifier key values that need to be passed onto the event.
     * @param isPopupTrigger
     *                   boolean specifying whether this event will show a popup.
     * @param sleepTime
     *                   The wait time in ms between each event.
     * @param position  The relative mouse position within the cell.
     * @param referencePoint
     *                   If position is CUSTOM then the point is a offset from
     *                   the location of the {@link Component}. If the position is PERCENT
     *                   then the location is a percentage offset of the hight and width.
     *                   Otherwise, the referencePoint is unused.
     */
    public JComboBoxMouseEventData(final JFCTestCase testCase,
        final JComboBox comboBox, final Object element,
        final int numberOfClicks, final int modifiers,
        final boolean isPopupTrigger, final long sleepTime, final int position,
        final Point referencePoint) {
        this(testCase, comboBox,
            getIndexOf(comboBox, element), numberOfClicks, modifiers,
            isPopupTrigger, sleepTime, position, referencePoint);
    }

    /**
     * Constructor.
     *
     * @param testCase The {@link JFCTestCase} on whose thread <code>awtSleep()</code> has to be invoked.
     * @param comboBox The {@link JComboBox} on which to trigger the event.
     * @param elementIndex
     *                  The zero-based index of the specific element on which to trigger the event.
     * @param numberOfClicks
     *                  Number of clicks in the {@link MouseEvent} (1 for single-click, 2 for double clicks)
     */
    public JComboBoxMouseEventData(final JFCTestCase testCase,
        final JComboBox comboBox, final int elementIndex,
        final int numberOfClicks) {
        this(testCase, comboBox, elementIndex, numberOfClicks, DEFAULT_SLEEPTIME);
    }

    /**
     * Constructor.
     *
     * @param testCase  The {@link JFCTestCase} on whose thread <code>awtSleep()</code> has to be invoked.
     * @param comboBox  The {@link JComboBox} on which to trigger the event.
     * @param elementIndex
     *                   The zero-based index of the specific element on which to trigger the event.
     * @param numberOfClicks
     *                   Number of clicks in the {@link MouseEvent} (1 for single-click, 2 for double clicks)
     * @param sleepTime
     *                   The wait time in ms between each event.
     */
    public JComboBoxMouseEventData(final JFCTestCase testCase,
        final JComboBox comboBox, final int elementIndex,
        final int numberOfClicks, final long sleepTime) {
        this(testCase, comboBox, elementIndex, numberOfClicks,
            DEFAULT_MOUSE_MODIFIERS, DEFAULT_ISPOPUPTRIGGER, sleepTime);
    }

    /**
     * Constructor.
     *
     * @param testCase  The {@link JFCTestCase} on whose thread <code>awtSleep()</code> has to be invoked.
     * @param comboBox  The {@link JComboBox} on which to trigger the event.
     * @param elementIndex
     *                   The zero-based index of the specific element on which to trigger the event.
     * @param numberOfClicks
     *                   Number of clicks in the {@link MouseEvent} (1 for single-click, 2 for double clicks)
     * @param modifiers The modifier key values that need to be passed onto the event.
     * @param isPopupTrigger
     *                   boolean specifying whether this event will show a popup.
     * @param sleepTime
     *                   The wait time in ms between each event.
     */
    public JComboBoxMouseEventData(final JFCTestCase testCase,
        final JComboBox comboBox, final int elementIndex,
        final int numberOfClicks, final int modifiers,
        final boolean isPopupTrigger, final long sleepTime) {
        this(testCase, comboBox, elementIndex, numberOfClicks, modifiers,
            isPopupTrigger, sleepTime, DEFAULT_POSITION, null);
    }

    /**
     * Constructor.
     *
     * @param testCase  The {@link JFCTestCase} on whose thread <code>awtSleep()</code> has to be invoked.
     * @param comboBox  The {@link JComboBox} on which to trigger the event.
     * @param elementIndex
     *                   The zero-based index of the specific element on which to trigger the event.
     * @param numberOfClicks
     *                   Number of clicks in the {@link MouseEvent} (1 for single-click, 2 for double clicks)
     * @param modifiers The modifier key values that need to be passed onto the event.
     * @param isPopupTrigger
     *                   boolean specifying whether this event will show a popup.
     * @param sleepTime
     *                   The wait time in ms between each event.
     * @param referencePoint
     *                   The CUSTOM mouse position within the cell.
     */
    public JComboBoxMouseEventData(final JFCTestCase testCase,
        final JComboBox comboBox, final int elementIndex,
        final int numberOfClicks, final int modifiers,
        final boolean isPopupTrigger, final long sleepTime,
        final Point referencePoint) {
        this(testCase, comboBox, elementIndex, numberOfClicks, modifiers,
            isPopupTrigger, sleepTime, CUSTOM, referencePoint);
    }

    /**
     * Constructor.
     *
     * @param testCase  The {@link JFCTestCase} on whose thread <code>awtSleep()</code> has to be invoked.
     * @param comboBox  The {@link JComboBox} on which to trigger the event.
     * @param elementIndex
     *                   The zero-based index of the specific element on which to trigger the event.
     * @param numberOfClicks
     *                   Number of clicks in the {@link MouseEvent} (1 for single-click, 2 for double clicks)
     * @param modifiers The modifier key values that need to be passed onto the event.
     * @param isPopupTrigger
     *                   boolean specifying whether this event will show a popup.
     * @param sleepTime
     *                   The wait time in ms between each event.
     * @param position  The relative mouse position within the cell.
     */
    public JComboBoxMouseEventData(final JFCTestCase testCase,
        final JComboBox comboBox, final int elementIndex,
        final int numberOfClicks, final int modifiers,
        final boolean isPopupTrigger, final long sleepTime, final int position) {
        this(testCase, comboBox, elementIndex, numberOfClicks, modifiers,
            isPopupTrigger, sleepTime, position, null);
    }

    /**
     * Constructor.
     *
     * @param testCase  The {@link JFCTestCase} on whose thread <code>awtSleep()</code> has to be invoked.
     * @param comboBox  The {@link JComboBox} on which to trigger the event.
     * @param elementIndex
     *                   The zero-based index of the specific element on which to trigger the event.
     * @param numberOfClicks
     *                   Number of clicks in the {@link MouseEvent} (1 for single-click, 2 for double clicks)
     * @param modifiers The modifier key values that need to be passed onto the event.
     * @param isPopupTrigger
     *                   boolean specifying whether this event will show a popup.
     * @param sleepTime
     *                   The wait time in ms between each event.
     * @param position  The relative mouse position within the cell.
     * @param referencePoint
     *                   If position is CUSTOM then the point is a offset from
         *                   the location of the component. If the position is PERCENT
     *                   then the location is a percentage offset of the hight and width.
     *                   Otherwise, the referencePoint is unused.
     */
    public JComboBoxMouseEventData(final JFCTestCase testCase,
        final JComboBox comboBox, final int elementIndex,
        final int numberOfClicks, final int modifiers,
        final boolean isPopupTrigger, final long sleepTime, final int position,
        final Point referencePoint) {
        setTestCase(testCase);
        setSource(comboBox);
        setNumberOfClicks(numberOfClicks);
        setModifiers(modifiers);
        setPopupTrigger(isPopupTrigger);
        setElementIndex(elementIndex);
        setSleepTime(sleepTime);
        setPosition(position);
        setReferencePoint(referencePoint);
        setValid(true);
    }

    /**
     * Set the attribute value.
     *
     * @param comboBox The new value of the attribute.
     */
    public final void setSource(final JComboBox comboBox) {
        m_comboBox = comboBox;
    }

    /**
     * Get the attribute value.
     *
     * @return JComboBox    The value of the attribute.
     */
    public final JComboBox getSource() {
        return m_comboBox;
    }

    /**
     * The component on which the event has to be fired.
     *
     * @return The component.
     */
    public Component getComponent() {
        // by default, the component is the same as the source

        /** @todo this is a hack
         * but should work for now to get the
         * correct component to the recording. Should probably use a
         * new interface.
         */
        if (m_listView == null) {
            return getSource();
        }

        return m_listView;
    }

    /**
     * Set the attribute value.
     *
     * @param elementIndex The new value of the attribute.
     */
    public final void setElementIndex(final int elementIndex) {
        m_elementIndex = elementIndex;
    }

    /**
     * Get the attribute value.
     *
     * @return int    The value of the attribute.
     */
    public int getElementIndex() {
        return m_elementIndex;
    }

    /**
     * Returns true if the event can be consumed by
     * this instance of event data.
     *
     * @param ae Event to be consumed.
     * @return true if the event was consumed.
     */
    public boolean canConsume(final AWTEvent ae) {
        if (!super.canConsume(ae) || !(ae.getSource() instanceof JList)) {
            return false;
        }

        if (isValid()) {
            JList source = (JList) ae.getSource();
            int   index = source.locationToIndex(
                    new Point(
                        ((MouseEvent) ae).getX(),
                        ((MouseEvent) ae).getY()));

            if (index != getElementIndex()) {
                return false;
            }
        }

        return isComboBox((JList) ae.getSource());
    }

    /**
     * Consume the event.
     *
     * @param ae AWTEvent to be consumed.
     * @return boolean true if the event was consumed.
     */
    public boolean consume(final AWTEvent ae) {
        if (super.consume(ae)) {
            return true;
        }

        MouseEvent me     = (MouseEvent) ae;
        JList      source = (JList) me.getSource();
        setSource(getParentComboBox(source));
        setModifiers(me.getModifiers());
        setNumberOfClicks(me.getClickCount());
        setPopupTrigger(me.isPopupTrigger());

        Point p = new Point(
                me.getX(),
                me.getY());
        Point screen = source.getLocationOnScreen();
        screen.translate(p.x, p.y);
        setLocationOnScreen(screen);
        setSleepTime(getDefaultSleepTime());

        int index = source.locationToIndex(new Point(
                    me.getX(),
                    me.getY()));
        setElementIndex(index);

        setPosition(CENTER);
        setReferencePoint(null);

        setValid(true);

        return true;
    }

    /**
     * Compare to event datas and determine if they are equal.
     *
     * @param o Object to be compared.
     * @return true if the events are the same.
     */
    public boolean equals(final Object o) {
        if (!super.equals(o)) {
            return false;
        }

        JComboBoxMouseEventData data = (JComboBoxMouseEventData) o;

        return (data.getSource() == getSource())
        && (data.getElementIndex() == getElementIndex());
    }

    /**
     * Return a hashCode for this object.
     * @return int hashCode.
     */
    public int hashCode() {
        return super.hashCode() + m_elementIndex;
    }

    /**
     * Populate the JComboBoxMouseEventData XML element.
     * @param e element to be populated.
     */
    public void populate(final Element e) {
        super.populate(e);
        e.setAttribute(JFCXMLConstants.TYPE, "JComboBoxMouseEventData");
        e.setAttribute(JFCXMLConstants.INDEX, "" + getElementIndex());
    }

    /**
     * Prepare the {@link Component} to receive the event.
     *
     * @return true if the component is ready to receive the event.
     */
    public boolean prepareComponent() {
        if (!isValidForProcessing(getSource())) {
            return false;
        }

        JFCTestCase testCase = getTestCase();

        if (testCase != null) {
            // try to clear the event queue
            testCase.flushAWT();
        }

        Component toClick = m_comboBox;
        int clicks = 1;
        // If the combo box isEditable,
        // Then we must click in the button to
        // show the menu.
        if (m_comboBox.isEditable()) {
            Finder f = new ComponentFinder(JButton.class);
            f.setWait(0);
            Component comp = f.find(m_comboBox, 0);
            // If we have a button then use the button.
            if (comp != null) {
                MouseEventData data = new MouseEventData(testCase, comp);
                testCase.getHelper().enterClickAndLeave(data);
            } else {
                // We will try to assume that the
                // location to click is between the
                // editor and the combobox.
                Rectangle e = m_comboBox.getEditor().getEditorComponent().
                    getBounds();
                Rectangle cb = m_comboBox.getBounds();
                Rectangle button = calcButtonArea(e, cb);
                Point p = new Point((int) button.getCenterX(),
                                    (int) button.getCenterY());
                MouseEventData data = new MouseEventData(testCase, toClick);
                data.setPosition(MouseEventData.CUSTOM);
                data.setReferencePoint(p);
                testCase.getHelper().enterClickAndLeave(data);
            }
        } else {
            // Click on the combo box to open the menu.
            testCase.getHelper().enterClickAndLeave(
                new MouseEventData(testCase, toClick, clicks));
        }

        // If the popup is not open, then we are done.
        // Popup may not open if the combo box is not
        // enabled.
        if (!m_comboBox.isPopupVisible()) {
          return false;
        }

//        m_comboBox.showPopup();

        if (testCase != null) {
            // try to clear the event queue
            testCase.flushAWT();
        }

        BasicComboPopup bcp = (BasicComboPopup) m_comboBox.getAccessibleContext()
                                                          .getAccessibleChild(0 /*popup*/);

        if (testCase != null) {
            // try to clear the event queue
            testCase.flushAWT();
        }

        m_listView = bcp.getList();

        testCase.getHelper().enterClickAndLeave(
          new JListMouseEventData(testCase, m_listView, m_elementIndex, 1));



        return false;
    }

    /**
     * Calculate the difference between the two
     * rectangles. And use this as the button
     * location.
     *
     * @param comboBox Rectangle from combo box.
     * @param editor Rectangle from editor.
     * @return Rectangle Remaining rectangle.
     */
    private Rectangle calcButtonArea(final Rectangle comboBox, final Rectangle editor) {
        return new Rectangle(
            editor.x + editor.width, editor.y,
            comboBox.width - editor.width,
            editor.height
            );
    }

    /**
     * Get string description of event.
     *
     * @return String description of event.
     */
    public String toString() {
        if (!isValid()) {
            return super.toString();
        }

        StringBuffer buf = new StringBuffer(1000);
        buf.append(super.toString());
        buf.append(" index: " + getElementIndex());

        return buf.toString();
    }

    /**
     * Get the parent combo box of the list.
     *
     * @param list List to check.
     * @return true if combo box.
     */
    private boolean isComboBox(final JList list) {
        Accessible ap = list.getAccessibleContext().getAccessibleParent();

        while (ap != null) {
            if ((ap instanceof ComboBoxUI)
                    || (ap instanceof JComboBox)
                    || (ap.getClass().getName().indexOf("ComboBoxUI") != -1)) {
                return true;
            }

            ap = ap.getAccessibleContext().getAccessibleParent();
        }

        return false;
    }

    /**
     * Finds the index of the element passed in from the {@link JComboBox}.
     *
     * @param    comboBox    The JComboBox which holds the elements.
     * @param    element     The element whose index has to be found out.
     * @return   int   The zero-based index where the element occurs
     *                 (-1 is returned if not found).
     */
    private static int getIndexOf(final JComboBox comboBox, final Object element) {
        for (int i = 0; i < comboBox.getItemCount(); i++) {
            if ((comboBox.getItemAt(i) != null)
                    && comboBox.getItemAt(i).equals(element)) {
                return i;
            }
        }

        return -1;
    }

    /**
     * Get the parent combo box of the list.
     * This is difficult since the JList is normally a
     * inner class, and there is not a linkage to traverse
     * outside the class.
     *
     * Here we will traverse the Listener list to get to the
     * JComboBox as it should be listening to the list as well.
     *
     * @param list List to check
     * @return JComboBox of parent.
     */
    private JComboBox getParentComboBox(final JList list) {
        ListModel ldm = list.getModel();

        if (ldm instanceof AbstractListModel) {
            AbstractListModel cbm = (AbstractListModel) ldm;
            EventListener[]  listeners = cbm.getListeners(ListDataListener.class);

            for (int i = 0; i < listeners.length; i++) {
                if (listeners[i] instanceof JComboBox) {
                    return (JComboBox) listeners[i];
                }
            }
        }

        return null;
    }
}
