package junit.extensions.jfcunit.eventdata;

import junit.extensions.jfcunit.JFCTestCase;

import java.awt.AWTEvent;
import java.awt.Toolkit;
import java.awt.event.AWTEventListener;
import java.awt.event.InputEvent;
import java.awt.event.MouseEvent;

import java.util.ArrayList;
import java.util.EventListener;
import java.util.Iterator;

import javax.swing.UIManager;
import javax.swing.event.EventListenerList;


/**
 * This class provides a recording capabilities for AWTEvents.
 * {@link java.awt.AWTEvent}s are translated into their coresponding
 * Event Data types. An application may register a listener against
 * this class, to receive the event data.
 *
 * This class temporarily holds events until either a different
 * event type is received or a timer expires on the event.
 *
 * @author Kevin Wilson
 */
public final class JFCEventManager implements AWTEventListener,
    EventDataConstants {
    /**
     * Event Mapping Property.
     */
    public static final String EVENT_MAPPING_PROPERTY = "junit.extensions.jfcUnit.eventMapping";

    /**
    * Used to turn on debug info.
     */
    public static final String EVENT_DEBUG = "JFCEventManager.debug";

    static {
        if (UIManager.get(EVENT_MAPPING_PROPERTY) == null) {
            ArrayList mapping = new ArrayList();
            mapping.add(
                new String[] {
                    "junit.extensions.jfcunit.eventdata.JComboBoxMouseEventData",
                    "javax.swing.JList"
                });
            mapping.add(
                new String[] {
                    "junit.extensions.jfcunit.eventdata.JListMouseEventData",
                    "javax.swing.JList"
                });
            mapping.add(
                new String[] {
                    "junit.extensions.jfcunit.eventdata.JTableMouseEventData",
                    "javax.swing.JTable"
                });
            mapping.add(
                new String[] {
                    "junit.extensions.jfcunit.eventdata.JTableHeaderMouseEventData",
                    "javax.swing.table.JTableHeader"
                });
            mapping.add(
                new String[] {
                    "junit.extensions.jfcunit.eventdata.JTreeMouseEventData",
                    "javax.swing.JTree"
                });
            mapping.add(
                new String[] {
                    "junit.extensions.jfcunit.eventdata.JTabbedPaneMouseEventData",
                    "javax.swing.JTabbedPane"
                });
            mapping.add(
                new String[] {
                    "junit.extensions.jfcunit.eventdata.JTextComponentMouseEventData",
                    "javax.swing.text.JTextComponent"
                });
            mapping.add(
                new String[] {
                    "junit.extensions.jfcunit.eventdata.MouseWheelEventData",
                    "*"
                });
            mapping.add(
                new String[] {
                    "junit.extensions.jfcunit.eventdata.JMenuMouseEventData",
                    "*"
                });
            mapping.add(
                new String[] {
                    "junit.extensions.jfcunit.eventdata.MouseEventData", "*"
                });
            mapping.add(
                new String[] {
                    "junit.extensions.jfcunit.eventdata.StringEventData", "*"
                });
            mapping.add(
                new String[] {
                    "junit.extensions.jfcunit.eventdata.KeyEventData", "*"
                });

            UIManager.put(EVENT_MAPPING_PROPERTY, mapping);
        }
    }

    /**
     * This is a singleton class. There should never be more than
     * one recording session.
     */
    private static JFCEventManager s_singleton = null;

    /**
     * Debug flag set by UIManager property JFCEventManager.debug="true".
     */
    private static boolean s_debug = true;

    /**
     * AWTEvent inputs.
     */
    public static final int DEBUG_INPUT = 1;

    /**
     * EventData outputs.
     */
    public static final int DEBUG_OUTPUT = 2;

    /**
     * EventData creations.
     */
    public static final int DEBUG_CREATE = 4;

    /**
     * All debugging types.
     */
    public static final int DEBUG_ALL = 7;

    /**
     * Debugging type to be used.
     */
    private static int s_debugType = DEBUG_ALL;

    /**
     * Pending event held for consolidation.
     */
    private AbstractEventData m_pendingEvent = null;

    /**
     * Listener list.
     */
    private EventListenerList m_listenerList = new EventListenerList();

    /**
     * Timer {@link Thread} used to send the pending event.
     */
    private Thread m_timerThread;

    /**
     * Recording state.
     */
    private boolean m_recording = false;

    /**
     * Hold time for the timer thread before
     * firing the  pending event.
     */
    private long m_holdTime;

    /**
     * Time when the last event was recorded.
     */
    private volatile long m_lastEventTime = 0;

    /**
     * Private constructor.
     * The method <code>getEventManager()</code> should be used instead.
     *
     * @param holdTime duration to hold a pending event.
     */
    private JFCEventManager(final long holdTime) {
        setHoldTime(holdTime);
    }

    /**
     * Enable/Disable debug tracing of events.
     *
     * @param aValue true if debugging is to be
     * turned on.
     */
    public static void setDebug(final boolean aValue) {
        s_debug = aValue;

        if (s_debug) {
            UIManager.put(EVENT_DEBUG, "true");
        } else {
            UIManager.put(EVENT_DEBUG, "false");
        }
    }

    /**
     * Get the state of debugging.
     *
     * @return true if debugging is enabled.
     */
    public static boolean getDebug() {
        Boolean value = null;
        Object  prop = UIManager.get(EVENT_DEBUG);

        if (prop instanceof Boolean) {
            value = (Boolean) prop;
        } else if (prop instanceof String) {
            value = Boolean.valueOf((String) prop);
        }

        if (value == null) {
            s_debug = false;
        } else {
            s_debug = value.booleanValue();
        }

        return s_debug;
    }

    /**
     * Set the debugging type.
     *
     * @param type DEBUG_INPUT, DEBUG_OUTPUT and/or DEBUG_CREATE
     */
    public static void setDebugType(final int type) {
        s_debugType = type;
    }

    /**
     * Get the debugging type.
     *
     * @return sum of current debugging types.
     */
    public static int getDebugType() {
        return s_debugType;
    }

    /**
     * Returns a singleton instance of this class. This
     * method should be used instead of the constructor.
     * to attempt to consolidate events.
     *
     * @return JFCEventManager singleton instance.
     */
    public static JFCEventManager getEventManager() {
        return JFCEventManager.getEventManager(DEFAULT_HOLDTIME);
    }

    /**
     * Returns a singleton instance of this class. This
     * method should be used instead of the constructor.
     *
     * @param holdTime druration a event should be held
     *                  to attempt to consolidate events.
     *
     * @return JFCEventManager singleton instance.
     */
    public static JFCEventManager getEventManager(final long holdTime) {
        if (s_singleton == null) {
            s_singleton = new JFCEventManager(holdTime);
        }

        return s_singleton;
    }

    /**
     * Set the recording state.
     *
     * @param recording true if enabled. Otherwise false.
     */
    public static void setRecording(final boolean recording) {
        getEventManager().setRecordingImpl(recording);
    }

    /**
     * Set the maximum hold time for a event.
     *
     * @param holdTime maximum duration in millis to
     *  hold a event.
     */
    public void setHoldTime(final long holdTime) {
        m_holdTime = holdTime;
    }

    /**
     * Get the maximum hold time for a event.
     *
     * @return long maximum hold time.
     */
    public long getHoldTime() {
        return m_holdTime;
    }

    /**
     * Get the current recording state.
     *
     * @return boolean recording state. True if recording is
     * enabled.
     * Otherwise, false.
     */
    public boolean getRecording() {
        return m_recording;
    }

    /**
     * Add a listener.
     *
     * @param jl Listener to be added.
     */
    public void addJFCEventDataListener(final JFCEventDataListener jl) {
        m_listenerList.add(JFCEventDataListener.class, jl);
    }

    /**
     * Converts the event to a drag event if necessary.
     *
     * @param ae Event to be processed.
     * @return true if converted to drag event.
     */
    public boolean convertDrag(final AWTEvent ae) {
        if (!(m_pendingEvent instanceof AbstractMouseEventData)
                || !(ae instanceof MouseEvent)) {
            return false;
        }

        MouseEvent me = (MouseEvent) ae;

        if (me.getID() != MouseEvent.MOUSE_DRAGGED) {
            return false;
        }

        m_pendingEvent = new DragEventData((JFCTestCase) null,
                (AbstractMouseEventData) m_pendingEvent);

        return true;
    }

    /**
     * Create a event for the data given.
     *
     * @param ae Event to be processed.
     * @return true if the event is created.
     */
    public AbstractEventData createEvent(final AWTEvent ae) {
        Object    source = ae.getSource();

        ArrayList mapping = (ArrayList) UIManager.get(EVENT_MAPPING_PROPERTY);
        Iterator  iter    = mapping.iterator();

        while (iter.hasNext()) {
            String[] map    = (String[]) iter.next();
            Class    target = null;

            try {
                if (!map[1].equals("*")) {
                    target = Class.forName(map[1]);
                } else {
                    target = Object.class;
                }
            } catch (Exception e) {
                // don't do anything, but still catch it
            }

            if (target.isInstance(source)) {
                // Attempt to consume event.
                try {
                    if (s_debug && ((s_debugType & DEBUG_CREATE) > 0)) {
                        System.err.println(
                            "JFCEventManager.createEvent check class:" + map[0]
                            + " can handle " + ae);
                    }

                    Class             cls  = Class.forName(map[0]);
                    AbstractEventData data = (AbstractEventData) cls
                        .newInstance();

                    if (data.canConsume(ae)) {
                        return data;
                    }
                } catch (Exception e) {
                    System.err.println(
                        "Exception attempting to create instnce for:" + map[0]);
                    e.printStackTrace();
                }
            }
        }

        if (s_debug && ((s_debugType & DEBUG_CREATE) > 0)) {
            System.err.println(
                "JFCEventManager.createEvent No EventData structure for:" + ae);
        }

        return null;
    }

    /**
     * This method implements the {@link AWTEventListener} interface.
     * This method will be accessed for every {@link AWTEvent} which is
     * of the type: MOUSE_MOTION_EVENT, MOUSE_EVENT, KEY_EVENT, or
     * TEXT_EVENT.
     *
     * @param ae Event to be processed.
     */
    public void eventDispatched(final AWTEvent ae) {
        processEventData(ae);
    }

    /**
     * Remove all listeners.
     */
    public void removeAllJFCEventDataListeners() {
        Object[] listeners = m_listenerList.getListenerList();

        for (int i = 0; i < listeners.length; i += 2) {
            if (listeners[i] == JFCEventDataListener.class) {
                m_listenerList.remove(JFCEventDataListener.class,
                    (EventListener) listeners[i + 1]);
            }
        }
    }

    /**
     * Remove a listener.
     *
     * @param jl Listener to be removed.
     */
    public void removeJFCEventDataListener(final JFCEventDataListener jl) {
        m_listenerList.remove(JFCEventDataListener.class, jl);
    }

    /**
     * Fire event data to the listeners.
     */
    protected void fireEventData() {
        if ((m_pendingEvent != null) && m_pendingEvent.isValid()) {
            if (s_debug && ((s_debugType & DEBUG_OUTPUT) > 0)) {
                System.err.println("JFCEventManager.outputEvent:"
                    + m_pendingEvent);
            }

            Object[] listeners = m_listenerList.getListenerList();

            for (int i = listeners.length - 2; i >= 0; i -= 2) {
                if (listeners[i] == JFCEventDataListener.class) {
                    ((JFCEventDataListener) listeners[i + 1]).handleEvent(m_pendingEvent);
                }
            }
        }

        m_pendingEvent = null;
    }

    /**
     * This method converts the {@link AWTEvent} to the corresponding
     * AbstractEventData.
     *
     * @param ae AWTEvent to be processed.
     */
    protected synchronized void processEventData(final AWTEvent ae) {
        if (s_debug && ((s_debugType & DEBUG_INPUT) > 0)) {
            System.err.println("JFCEventManager.inputEvent:"
                + ((InputEvent) ae).getWhen() + " " + ae);
        }

        m_lastEventTime = System.currentTimeMillis();

        if (ae instanceof MouseEvent
                && (((MouseEvent) ae).getID() == MouseEvent.MOUSE_MOVED)) {
            return;
        }

        if ((m_pendingEvent != null) && !m_pendingEvent.canConsume(ae)) {
            if (!convertDrag(ae)) {
                fireEventData();
            }
        }

        if (m_pendingEvent == null) {
            m_pendingEvent = createEvent(ae);
        }

        if (m_pendingEvent != null) {
            m_pendingEvent.consume(ae);
            m_lastEventTime = 0;
        }
    }

    /**
     * Set the recording state.
     *
     * @param recording true if recording is to be enabled.
     * otherwise false.
     */
    private void setRecordingImpl(final boolean recording) {
        if (recording && !(this.m_recording)) {
            getDebug();

            long wheelMask = 0;

            try {
                wheelMask = AWTEvent.class.getField("MOUSE_WHEEL_EVENT_MASK")
                                          .getLong(null);
            } catch (Exception e) {
                wheelMask = 0;
            }

            Toolkit.getDefaultToolkit().addAWTEventListener(this,
                AWTEvent.MOUSE_EVENT_MASK | AWTEvent.MOUSE_MOTION_EVENT_MASK
                | AWTEvent.KEY_EVENT_MASK | AWTEvent.TEXT_EVENT_MASK
                | wheelMask);
            m_timerThread = new Thread(
                    new Runnable() {
                        public void run() {
                            while (m_recording) {
                                try {
                                    Thread.currentThread().sleep(m_holdTime);
                                } catch (InterruptedException ie) {
                                    // don't do anything, but still catch it
                                }

                                synchronized (JFCEventManager.this) {
                                    long time = System.currentTimeMillis();

                                    if (m_lastEventTime == 0) {
                                        m_lastEventTime = System
                                            .currentTimeMillis();
                                    }

                                    if ((m_pendingEvent != null)
                                            && ((time - m_lastEventTime) > m_holdTime)) {
                                        fireEventData();
                                    }
                                }
                            }

                            if (m_pendingEvent != null) {
                                fireEventData();
                            }
                        }
                    },
                    "JFCEventManager Timer");

            m_timerThread.start();
            this.m_recording = recording;
        } else if (!recording && (this.m_recording)) {
            Toolkit.getDefaultToolkit().removeAWTEventListener(this);
            this.m_recording = recording;
            m_timerThread.interrupt();

            //            try {
            //              timerThread.join();
            //            } catch (InterruptedException ie) {
            //                // don't do anything, but still catch it
            //            }
        }

        this.m_recording = recording;
    }
}
