package junit.extensions.jfcunit.eventdata;

import junit.extensions.jfcunit.JFCTestCase;
import junit.extensions.jfcunit.finder.ComponentFinder;
import junit.extensions.jfcunit.finder.Finder;
import junit.extensions.jfcunit.xml.JFCXMLConstants;

import org.w3c.dom.Element;

import java.awt.AWTEvent;
import java.awt.Component;
import java.awt.Container;
import java.awt.Point;
import java.awt.Rectangle;

import javax.swing.JButton;
import javax.swing.text.JTextComponent;


/**
 * Data container class that holds all the data necessary for jfcUnit to fire mouse events.
 *
 * @author <a href="mailto:vraravam@thoughtworks.com">Vijay Aravamudhan : ThoughtWorks Inc.</a>
 */
public class JSpinnerMouseEventData extends AbstractMouseEventData {
    /** Class for spinner component. */
    private static Class s_spinnerClass = null;

    /**
     * Use reflection to keep 1.2 and 1.3 jdk compiling.
     */
    static {
        try {
            s_spinnerClass = Class.forName("javax.swing.JSpinner");
        } catch (Exception e) {
            s_spinnerClass = null;
        }
    }

    /**
     * The Component on which to trigger the event.
     */
    private Component m_comp;

    /**
     * The Source on which to trigger the event.
     */
    private Component m_source;

    /**
     * Offset into the text component.
     */
    private int m_offset = INVALID_TEXT_OFFSET;

    /**
     * The subcomponent type to send click.
     */
    private int m_subComponent = INVALID_SUBCOMPONENT;

    /**
     * Default Constructor.
     */
    public JSpinnerMouseEventData() {
        super();
        setValid(false);
    }

    /**
     * Constructor - Defualt click cell.
     *
     * @param testCase The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param comp     The component on which to trigger the event.
     * @param subComponent Sub-component of the JSpinner.
     */
    public JSpinnerMouseEventData(final JFCTestCase testCase,
        final Component comp, final int subComponent) {
        this(testCase, comp, subComponent, DEFAULT_NUMBEROFCLICKS);
    }

    /**
     * Constructor.
     *
     * @param testCase The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param comp     The component on which to trigger the event.
     * @param subComponent Sub-component of the JSpinner.
     * @param numberOfClicks
     *                 Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     */
    public JSpinnerMouseEventData(final JFCTestCase testCase,
        final Component comp, final int subComponent, final int numberOfClicks) {
        this(testCase, comp, subComponent, numberOfClicks,
            DEFAULT_MOUSE_MODIFIERS);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param comp      The component on which to trigger the event.
     * @param subComponent Sub-component of the JSpinner.
     * @param sleepTime
     *                  The wait time in ms between each event.
     */
    public JSpinnerMouseEventData(final JFCTestCase testCase,
        final Component comp, final int subComponent, final long sleepTime) {
        this(testCase, comp, subComponent, DEFAULT_NUMBEROFCLICKS, sleepTime);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param comp      The component on which to trigger the event.
     * @param subComponent Sub-component of the JSpinner.
     * @param numberOfClicks
     *                  Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     * @param modifiers The modifier key values that need to be passed onto the event.
     */
    public JSpinnerMouseEventData(final JFCTestCase testCase,
        final Component comp, final int subComponent, final int numberOfClicks,
        final int modifiers) {
        this(testCase, comp, subComponent, numberOfClicks, modifiers,
            DEFAULT_ISPOPUPTRIGGER);
    }

    /**
     * Constructor.
     *
     * @param testCase The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param comp     The component on which to trigger the event.
     * @param subComponent Sub-component of the JSpinner.
     * @param numberOfClicks
     *                 Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     * @param isPopupTrigger
     *                 boolean specifying whether this event will show a popup.
     */
    public JSpinnerMouseEventData(final JFCTestCase testCase,
        final Component comp, final int subComponent, final int numberOfClicks,
        final boolean isPopupTrigger) {
        this(testCase, comp, subComponent, numberOfClicks,
            getDefaultModifiers(isPopupTrigger), isPopupTrigger);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param comp      The component on which to trigger the event.
     * @param subComponent Sub-component of the JSpinner.
     * @param numberOfClicks
     *                  Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     * @param sleepTime
     *                  The wait time in ms between each event.
     */
    public JSpinnerMouseEventData(final JFCTestCase testCase,
        final Component comp, final int subComponent, final int numberOfClicks,
        final long sleepTime) {
        this(testCase, comp, subComponent, numberOfClicks,
            DEFAULT_MOUSE_MODIFIERS, DEFAULT_ISPOPUPTRIGGER, sleepTime);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param comp      The component on which to trigger the event.
     * @param subComponent Sub-component of the JSpinner.
     * @param numberOfClicks
     *                  Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     * @param modifiers The modifier key values that need to be passed onto the event.
     * @param isPopupTrigger
     *                  boolean specifying whether this event will show a popup.
     */
    public JSpinnerMouseEventData(final JFCTestCase testCase,
        final Component comp, final int subComponent, final int numberOfClicks,
        final int modifiers, final boolean isPopupTrigger) {
        this(testCase, comp, subComponent, numberOfClicks, modifiers,
            isPopupTrigger, DEFAULT_SLEEPTIME);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param comp      The component on which to trigger the event.
     * @param subComponent Sub-component of the JSpinner.
     * @param numberOfClicks
     *                  Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     * @param isPopupTrigger
     *                  boolean specifying whether this event will show a popup.
     * @param sleepTime
     *                  The wait time in ms between each event.
     */
    public JSpinnerMouseEventData(final JFCTestCase testCase,
        final Component comp, final int subComponent, final int numberOfClicks,
        final boolean isPopupTrigger, final long sleepTime) {
        this(testCase, comp, subComponent, numberOfClicks,
            getDefaultModifiers(isPopupTrigger), isPopupTrigger, sleepTime);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param comp      The component on which to trigger the event.
     * @param subComponent Sub-component of the JSpinner.
     * @param numberOfClicks
     *                   Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     * @param modifiers The modifier key values that need to be passed onto the event.
     * @param isPopupTrigger
     *                   boolean specifying whether this event will show a popup.
     * @param sleepTime
     *                   The wait time in ms between each event.
     */
    public JSpinnerMouseEventData(final JFCTestCase testCase,
        final Component comp, final int subComponent, final int numberOfClicks,
        final int modifiers, final boolean isPopupTrigger, final long sleepTime) {
        this(testCase, comp, subComponent, numberOfClicks, modifiers,
            isPopupTrigger, sleepTime, DEFAULT_POSITION, null,
            INVALID_TEXT_OFFSET);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param comp      The component on which to trigger the event.
     * @param subComponent Sub-component of the JSpinner.
     * @param numberOfClicks
     *                   Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     * @param modifiers The modifier key values that need to be passed onto the event.
     * @param isPopupTrigger
     *                   boolean specifying whether this event will show a popup.
     * @param sleepTime
     *                   The wait time in ms between each event.
     * @param position  The relative mouse position within the cell.
     */
    public JSpinnerMouseEventData(final JFCTestCase testCase,
        final Component comp, final int subComponent, final int numberOfClicks,
        final int modifiers, final boolean isPopupTrigger,
        final long sleepTime, final int position) {
        this(testCase, comp, subComponent, numberOfClicks, modifiers,
            isPopupTrigger, sleepTime, position, null, INVALID_TEXT_OFFSET);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param comp      The component on which to trigger the event.
     * @param subComponent Sub-component of the JSpinner.
     * @param numberOfClicks
     *                   Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     * @param modifiers The modifier key values that need to be passed onto the event.
     * @param isPopupTrigger
     *                   boolean specifying whether this event will show a popup.
     * @param sleepTime
     *                   The wait time in ms between each event.
     * @param referencePoint     The CUSTOM mouse position within the cell.
     */
    public JSpinnerMouseEventData(final JFCTestCase testCase,
        final Component comp, final int subComponent, final int numberOfClicks,
        final int modifiers, final boolean isPopupTrigger,
        final long sleepTime, final Point referencePoint) {
        this(testCase, comp, subComponent, numberOfClicks, modifiers,
            isPopupTrigger, sleepTime, CUSTOM, referencePoint,
            INVALID_TEXT_OFFSET);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param comp      The component on which to trigger the event.
     * @param subComponent Sub-component of the JSpinner.
     * @param numberOfClicks
     *                   Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     * @param modifiers The modifier key values that need to be passed onto the event.
     * @param isPopupTrigger
     *                   boolean specifying whether this event will show a popup.
     * @param sleepTime
     *                   The wait time in ms between each event.
     * @param position  The relative mouse position within the cell.
     * @param referencePoint
     *                                   If _position is CUSTOM then the point is a offset from
     *                                   the location of the component. If the _position is PERCENT
     *                                   then the location is a percentage offset of the hight and width.
         *                                   Otherwise, the _referencePoint is unused.
     * @param offset    The offset into the text.
     */
    public JSpinnerMouseEventData(final JFCTestCase testCase,
        final Component comp, final int subComponent, final int numberOfClicks,
        final int modifiers, final boolean isPopupTrigger,
        final long sleepTime, final int position, final Point referencePoint,
        final int offset) {
        setTestCase(testCase);
        setSource(comp);
        setSubComponent(subComponent);
        setNumberOfClicks(numberOfClicks);
        setModifiers(modifiers);
        setPopupTrigger(isPopupTrigger);
        setSleepTime(sleepTime);
        setPosition(position);
        setReferencePoint(referencePoint);
        setOffset(offset);
        setValid(true);
    }

    /**
     * Set the attribute value.
     *
     * @param comp   The new value of the attribute.
     */
    public final void setSource(final Component comp) {
        m_source = comp;
    }

    /**
     * Get the attribute value.
     *
     * @return Component    The value of the attribute
     * NOTE: This is left as a Component so that the
     *       class may still compile under jdk1.3
     *       and below where JSpinner is not implemented.
     */
    public final Component getSource() {
        return m_source;
    }

    /**
     * Set the sub-component.
     * @param subComponent SubComponent
     *                      UP_ARROW_COMPONENT,
     *                      DOWN_ARROW_COMPONENT,
     *                      EDITOR_COMPONENT
     */
    public final void setSubComponent(final int subComponent) {
        m_subComponent = subComponent;
    }

    /**
     * Get the sub-component.
     * @return _subComponent Component constant.
     */
    public final int getSubComponent() {
        return m_subComponent;
    }

    /**
     * The component on which the event has to be fired.
     *
     * @param comp The component.
     */
    public void setComponent(final Component comp) {
        // by default, the component is the same as the source
        m_comp = comp;
    }

    /**
     * The component on which the event has to be fired.
     *
     * @return The component.
     */
    public Component getComponent() {
        // by default, the component is the same as the source
        return m_comp;
    }

    /**
     * Set the offset into the text.
     *
     * @param offset The offset into the text.
     */
    public final void setOffset(final int offset) {
        m_offset = offset;
    }

    /**
     * Get the offset into the text.
     *
     * @return int offset into the text.
     */
    public final int getOffset() {
        return m_offset;
    }

    /**
     * Returns true if the event can be consumed by
     * this instnace of event data. However, since
     * the JSpinner class is a composite of Buttons
     * and JFormattedTextField, no events can be issued,
     * on the JSpinner. Thus this call always returns false.
     *
     * @param ae Event to be consumed.
     * @return true if the event was consumed.
     */
    public boolean canConsume(final AWTEvent ae) {
        return false;
    }

    /**
     * Consume the event. This method always
     * returns false (see canConsume).
     *
     * @param ae AWTEvent to be consumed.
     * @return boolean true if the event was consumed.
     */
    public boolean consume(final AWTEvent ae) {
        return false;
    }

    /**
     * Compare to event data and deterime if they are equal.
     *
     * @param o Object to be compared.
     * @return true if the events are the same.
     */
    public boolean equals(final Object o) {
        if (!super.equals(o)) {
            return false;
        }

        JSpinnerMouseEventData data = (JSpinnerMouseEventData) o;

        return (data.getSource() == getSource())
            && (data.getSubComponent() == getSubComponent())
            && (data.getOffset() == getOffset());
    }

    /**
     * Get the hashCode.
     * @return int hashCode.
     */
    public int hashCode() {
        return super.hashCode() + this.m_offset;
    }

    /**
     * Populate the XML Element.
     * @param e element to be populated.
     */
    public void populate(final Element e) {
        super.populate(e);
        e.setAttribute(JFCXMLConstants.TYPE, "JSpinnerMouseEventData");

        if (getOffset() != INVALID_TEXT_OFFSET) {
            e.setAttribute(JFCXMLConstants.SUBCOMPONENT, JFCXMLConstants.EDITOR);
            e.setAttribute(JFCXMLConstants.INDEX, "" + getOffset());
        } else if (getSubComponent() == UP_ARROW_SUBCOMPONENT) {
            e.setAttribute(JFCXMLConstants.SUBCOMPONENT, JFCXMLConstants.UP);
        } else if (getSubComponent() == DOWN_ARROW_SUBCOMPONENT) {
            e.setAttribute(JFCXMLConstants.SUBCOMPONENT, JFCXMLConstants.DOWN);
        }
    }

    /**
     * Prepare the component for firing the mouse event.
     *
     * @return boolean true if the component is ready.
     */
    public boolean prepareComponent() {
        if (!isValidForProcessing(getSource()) || (s_spinnerClass == null)) {
            return false;
        }

        JFCTestCase testCase = getTestCase();

        if (testCase != null) {
            testCase.pauseAWT();
        }

        Component c      = null;
        Rectangle bounds = null;
        Finder    f      = new ComponentFinder(JButton.class);

        switch (getSubComponent()) {
        case DOWN_ARROW_SUBCOMPONENT:
            c = f.find((Container) getSource(), 1);
            bounds = c.getBounds();
            bounds.setLocation(c.getLocationOnScreen());
            setLocationOnScreen(calculatePoint(bounds));

            break;

        case UP_ARROW_SUBCOMPONENT:
            c = f.find((Container) getSource(), 0);
            bounds = c.getBounds();
            bounds.setLocation(c.getLocationOnScreen());
            setLocationOnScreen(calculatePoint(bounds));

            break;

        case EDITOR_SUBCOMPONENT:

            try {
                c = (Component) s_spinnerClass.getMethod(
                        "getEditor",
                        new Class[0]).invoke(
                        getSource(),
                        new Object[0]);

                Class defEditor = Class.forName(
                        "javax.swing.JSpinner$DefaultEditor");

                // If we have a DefaultEditor then we know how to get the
                // text field component.
                if (defEditor.isInstance(c)) {
                    c = (JTextComponent) defEditor.getMethod(
                            "getTextField",
                            new Class[0]).invoke(
                            c,
                            new Object[0]);
                }

                if ((getPosition() == OFFSET) && c instanceof JTextComponent) {
                    setPosition(EAST);

                    Point p = calculatePoint(
                            ((JTextComponent) c).modelToView(m_offset));
                    Point screen = c.getLocationOnScreen();
                    screen.translate(p.x, p.y);
                    setLocationOnScreen(screen);
                } else {
                    Point p      = calculatePoint(c.getBounds());
                    Point screen = c.getLocationOnScreen();
                    screen.translate(p.x, p.y);
                    setLocationOnScreen(screen);
                }
            } catch (Exception e) {
                e.printStackTrace();
            }

            break;

        default:
            throw new RuntimeException(
                "JSpinnerMouseEventData invalid sub component type");
        }

        setComponent(c);

        return true;
    }
}
