package junit.extensions.jfcunit.eventdata;

import junit.extensions.xml.IXMLTestCase;
import junit.extensions.xml.XMLException;

import org.w3c.dom.Element;

import javax.swing.JTabbedPane;


/**
 * This is a extension of the click and drag elements. Where the
 * type specified is JTabbedMouseEventData.
 *
 * <h3>Description</h3>
 * <p>
 *   Nothing is invoked directly by this tag handler. The click and drag elements
 *   use this tag handler to customize the data specificly for the type specified.
 * </p>
 *
 * <h3>Parameters</h3>
 * <table border="1" cellpadding="2" cellspacing="0">
 *   <tr>
 *     <td valign="top"><b>Attribute</b></td>
 *     <td valign="top"><b>Description</b></td>
 *     <td align="center" valign="top"><b>Required</b></td>
 *     <td valign="top"><b>Default</b></td>
 *     <td valign="top"><b>Values</b></td>
 *   </tr>
 *   <tr>
 *     <td valign="top">type</td>
 *     <td valign="top">For this element the type must be set to JTabbedPaneMouseEventData</td>
 *     <td valign="top" align="center">Yes</td>
 *     <td valign="top">N/A</td>
 *     <td valign="top">N/A</td>
 *   </tr>
 *   <tr>
 *     <td valign="top">refid</td>
 *     <td valign="top">Id of a object reference which has been previously found.</td>
 *     <td valign="top" align="center">Yes</td>
 *     <td valign="top">N/A</td>
 *     <td valign="top">N/A</td>
 *   </tr>
 *   <tr>
 *     <td valign="top">title</td>
 *     <td valign="top">The title of the tab.</td>
 *     <td valign="top" align="center">No</td>
 *     <td valign="top">null</td>
 *     <td valign="top">Text of the tab.</td>
 *   </tr>
 *   <tr>
 *     <td valign="top">index</td>
 *     <td valign="top">The index of the tab</td>
 *     <td valign="top" align="center">No</td>
 *     <td valign="top">-1</td>
 *     <td valign="top">Zero based index of the tab.</td>
 *   </tr>
 *   <tr>
 *     <td valign="top">clicks</td>
 *     <td valign="top">The number of clicks to be executed.</td>
 *     <td valign="top" align="center">No</td>
 *     <td valign="top">1</td>
 *     <td valign="top">Positive Integer</td>
 *   </tr>
 *   <tr>
 *     <td valign="top">modifiers</td>
 *     <td valign="top">The modifiers to be used for this event.</td>
 *     <td valign="top" align="center">No</td>
 *     <td valign="top">Button1 if popup is false otherwise Button3</td>
 *     <td valign="top">alt altgr control meta shift button1 button2 button3</td>
 *   </tr>
 *   <tr>
 *     <td valign="top">popup</td>
 *     <td valign="top">True if this is event is using the popup button.</td>
 *     <td valign="top" align="center">No</td>
 *     <td valign="top">false</td>
 *     <td valign="top">true false</td>
 *   </tr>
 *   <tr>
 *     <td valign="top">sleeptime</td>
 *     <td valign="top">Maximum sleep time</td>
 *     <td valign="top" align="center">No</td>
 *     <td valign="top">500</td>
 *     <td valign="top">Positive Integer representing the milli-seconds</td>
 *   </tr>
 *   <tr>
 *     <td valign="top">position</td>
 *     <td valign="top">Positioning of the mouse within the bounds of the component.</td>
 *     <td valign="top" align="center">No</td>
 *     <td valign="top">center</td>
 *     <td valign="top">center north northeast east southeast south southwest west northwest custom percentage</td>
 *   </tr>
 *   <tr>
 *     <td valign="top">reference</td>
 *     <td valign="top">Reference point or number for position calculation.</td>
 *     <td valign="top" align="center">No</td>
 *     <td valign="top">null</td>
 *     <td valign="top">integers &quot;x,y&quot;</td>
 *   </tr>
 *
 * </table>
 * <h3>Example</h3>
 * <blockquote><pre>
 * &lt;click
 *    type=&quot;JTabbedMouseEventData&quot;
 *    refid=&quot;TabbedPane&quot;
 *    title="First Tab"
 * /&gt;
 * </pre></blockquote>
 * <p>
 * The above clicks once on the center of the tab containing
 * the title "First Tab" in the JTabbedPane component identified by
 * TabbedPane.
 * </p>
 * @see junit.extensions.jfcunit.eventdata.JTabbedPaneMouseEventData
 * @author Kevin Wilson
 */
public class JTabbedPaneMouseEventDataTagHandler extends BaseEventDataTagHandler {
    /**
     * Constructor for JTabbedPaneMouseEventDataTagHandler.
     *
     * @param element     The element to be processed
     * @param testCase    The IXMLTestCase that uses this element
     */
    public JTabbedPaneMouseEventDataTagHandler(final Element element,
        final IXMLTestCase testCase) {
        super(element, testCase);
    }

    /**
     * @see junit.extensions.xml.elements.AbstractTagHandler#processElement()
     */
    public void processElement() throws XMLException {
        validateElement();

        setEventData(
            new JTabbedPaneMouseEventData(
                getJFCTestCase(),
                (JTabbedPane) getComponent(),
                getIndex(),
                getTitle(),
                getClicks(),
                getModifiers(),
                getPopupTrigger(),
                getSleepTime(),
                getPosition(),
                getReference()));
    }

    /**
     * @see junit.extensions.xml.elements.AbstractTagHandler#validateElement()
     */
    public void validateElement() throws XMLException {
        // do the default validations from the super class
        super.validateElement();

        checkOneRequiredAttribute(
            getElement(),
            new String[] {TITLE, INDEX});
    }

    /**
     * Returns the value of the INDEX attribute for this element. Defaults to -1 if not found.
     * @return int  The value of the parsed INDEX attribute, -1 if not found.
     */
    private int getIndex() {
        // default value in this class for index is -1
        return getInt(INDEX, -1);
    }

    /**
     * Returns the value of the TITLE attribute for this element.
     * @return String  The value of the TITLE attribute.
     */
    private String getTitle() {
        return getString(TITLE);
    }
}
