package junit.extensions.jfcunit.eventdata;

import junit.extensions.jfcunit.JFCTestCase;
import junit.extensions.jfcunit.xml.JFCXMLConstants;

import org.w3c.dom.Element;

import java.awt.AWTEvent;
import java.awt.Component;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.event.MouseEvent;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import javax.swing.JTree;
import javax.swing.tree.TreeModel;
import javax.swing.tree.TreeNode;
import javax.swing.tree.TreePath;


/**
 * Data container class that holds all the data necessary for jfcUnit to fire mouse events.
 * This class is specific to events on a JTree.
 *
 * @author <a href="mailto:vraravam@thoughtworks.com">Vijay Aravamudhan : ThoughtWorks Inc.</a>
 */
public class JTreeMouseEventData extends AbstractMouseEventData {
    /**
     * The JTree on which to trigger the event.
     */
    private JTree m_tree;

    /**
     * The String value of the specific node on which to trigger the event.
     */
    private String m_nodeValue;

    /**
     * The path of the specific node on which to trigger the event.
     */
    private TreePath m_treePath;

    /**
     * Constructor.
     */
    public JTreeMouseEventData() {
        super();
        setValid(false);
    }

    /**
     * Constructor.
     *
     * @param testCase The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param tree     The component on which to trigger the event.
         * @param treePath The path of the specific node on which to trigger the event.
     * @param numberOfClicks
     *                  Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     */
    public JTreeMouseEventData(final JFCTestCase testCase, final JTree tree,
        final TreePath treePath, final int numberOfClicks) {
        this(testCase, tree, treePath, numberOfClicks, DEFAULT_SLEEPTIME);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param tree      The component on which to trigger the event.
     * @param nodeValue The String value of the specific node on which to trigger the event.
     * @param numberOfClicks
     *                   Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     */
    public JTreeMouseEventData(final JFCTestCase testCase, final JTree tree,
        final String nodeValue, final int numberOfClicks) {
        this(testCase, tree, nodeValue, numberOfClicks, DEFAULT_SLEEPTIME);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param tree      The component on which to trigger the event.
         * @param treePath  The path of the specific node on which to trigger the event.
     * @param numberOfClicks
     *                   Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     * @param sleepTime
     *                   The wait time in ms between each event.
     */
    public JTreeMouseEventData(final JFCTestCase testCase, final JTree tree,
        final TreePath treePath, final int numberOfClicks, final long sleepTime) {
        this(testCase, tree, treePath, numberOfClicks, DEFAULT_ISPOPUPTRIGGER,
            sleepTime);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param tree      The component on which to trigger the event.
     * @param nodeValue The String value of the specific node on which to trigger the event.
     * @param numberOfClicks
     *                   Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     * @param sleepTime
     *                   The wait time in ms between each event.
     */
    public JTreeMouseEventData(final JFCTestCase testCase, final JTree tree,
        final String nodeValue, final int numberOfClicks, final long sleepTime) {
        this(testCase, tree, nodeValue, numberOfClicks, DEFAULT_ISPOPUPTRIGGER,
            sleepTime);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param tree      The component on which to trigger the event.
         * @param treePath  The path of the specific node on which to trigger the event.
     * @param numberOfClicks
     *                   Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     * @param isPopupTrigger
     *                   boolean specifying whether this event will show a popup.
     */
    public JTreeMouseEventData(final JFCTestCase testCase, final JTree tree,
        final TreePath treePath, final int numberOfClicks,
        final boolean isPopupTrigger) {
        this(testCase, tree, treePath, numberOfClicks, isPopupTrigger,
            DEFAULT_SLEEPTIME);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param tree      The component on which to trigger the event.
     * @param nodeValue The String value of the specific node on which to trigger the event.
     * @param numberOfClicks
     *                   Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     * @param isPopupTrigger
     *                   boolean specifying whether this event will show a popup.
     */
    public JTreeMouseEventData(final JFCTestCase testCase, final JTree tree,
        final String nodeValue, final int numberOfClicks,
        final boolean isPopupTrigger) {
        this(testCase, tree, nodeValue, numberOfClicks, isPopupTrigger,
            DEFAULT_SLEEPTIME);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param tree      The component on which to trigger the event.
     * @param treePath  The path of the specific node on which to trigger the event.
     * @param numberOfClicks
     *                   Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     * @param isPopupTrigger
     *                   boolean specifying whether this event will show a popup.
     * @param sleepTime
     *                   The wait time in ms between each event.
     */
    public JTreeMouseEventData(final JFCTestCase testCase, final JTree tree,
        final TreePath treePath, final int numberOfClicks,
        final boolean isPopupTrigger, final long sleepTime) {
        this(testCase, tree, treePath, numberOfClicks,
            getDefaultModifiers(isPopupTrigger), isPopupTrigger, sleepTime);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param tree      The component on which to trigger the event.
     * @param nodeValue The String value of the specific node on which to trigger the event.
     * @param numberOfClicks
     *                   Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     * @param isPopupTrigger
     *                   boolean specifying whether this event will show a popup.
     * @param sleepTime
     *                   The wait time in ms between each event.
     */
    public JTreeMouseEventData(final JFCTestCase testCase, final JTree tree,
        final String nodeValue, final int numberOfClicks,
        final boolean isPopupTrigger, final long sleepTime) {
        this(testCase, tree, nodeValue, numberOfClicks,
            getDefaultModifiers(isPopupTrigger), isPopupTrigger, sleepTime);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param tree      The component on which to trigger the event.
     * @param treePath  The path of the specific node on which to trigger the event.
     * @param numberOfClicks
     *                   Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     * @param modifiers The modifier key values that need to be passed onto the event.
     * @param isPopupTrigger
     *                   boolean specifying whether this event will show a popup.
     * @param sleepTime
     *                   The wait time in ms between each event.
     */
    public JTreeMouseEventData(final JFCTestCase testCase, final JTree tree,
        final TreePath treePath, final int numberOfClicks, final int modifiers,
        final boolean isPopupTrigger, final long sleepTime) {
        this(testCase, tree, treePath, numberOfClicks, modifiers,
            isPopupTrigger, sleepTime, DEFAULT_POSITION, null);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param tree      The component on which to trigger the event.
     * @param nodeValue The String value of the specific node on which to trigger the event.
     * @param numberOfClicks
     *                   Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     * @param modifiers The modifier key values that need to be passed onto the event.
     * @param isPopupTrigger
     *                   boolean specifying whether this event will show a popup.
     * @param sleepTime
     *                   The wait time in ms between each event.
     */
    public JTreeMouseEventData(final JFCTestCase testCase, final JTree tree,
        final String nodeValue, final int numberOfClicks, final int modifiers,
        final boolean isPopupTrigger, final long sleepTime) {
        this(testCase, tree, nodeValue, numberOfClicks, modifiers,
            isPopupTrigger, sleepTime, DEFAULT_POSITION, null);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param tree      The component on which to trigger the event.
         * @param treePath  The path of the specific node on which to trigger the event.
     * @param numberOfClicks
     *                   Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     * @param modifiers The modifier key values that need to be passed onto the event.
     * @param isPopupTrigger
     *                   boolean specifying whether this event will show a popup.
     * @param sleepTime
     *                   The wait time in ms between each event.
     * @param position  The relative mouse position within the cell.
     */
    public JTreeMouseEventData(final JFCTestCase testCase, final JTree tree,
        final TreePath treePath, final int numberOfClicks, final int modifiers,
        final boolean isPopupTrigger, final long sleepTime, final int position) {
        this(testCase, tree, treePath, numberOfClicks, modifiers,
            isPopupTrigger, sleepTime, position, null);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param tree      The component on which to trigger the event.
     * @param nodeValue The String value of the specific node on which to trigger the event.
     * @param numberOfClicks
     *                   Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     * @param modifiers The modifier key values that need to be passed onto the event.
     * @param isPopupTrigger
     *                   boolean specifying whether this event will show a popup.
     * @param sleepTime
     *                   The wait time in ms between each event.
     * @param position  The relative mouse position within the cell.
     */
    public JTreeMouseEventData(final JFCTestCase testCase, final JTree tree,
        final String nodeValue, final int numberOfClicks, final int modifiers,
        final boolean isPopupTrigger, final long sleepTime, final int position) {
        this(testCase, tree, nodeValue, numberOfClicks, modifiers,
            isPopupTrigger, sleepTime, position, null);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param tree      The component on which to trigger the event.
     * @param treePath  The path of the specific node on which to trigger the event.
     * @param numberOfClicks
     *                   Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     * @param modifiers The modifier key values that need to be passed onto the event.
     * @param isPopupTrigger
     *                   boolean specifying whether this event will show a popup.
     * @param sleepTime
     *                   The wait time in ms between each event.
     * @param referencePoint
     *                   The custom mouse position within the cell.
     */
    public JTreeMouseEventData(final JFCTestCase testCase, final JTree tree,
        final TreePath treePath, final int numberOfClicks, final int modifiers,
        final boolean isPopupTrigger, final long sleepTime,
        final Point referencePoint) {
        this(testCase, tree, treePath, numberOfClicks, modifiers,
            isPopupTrigger, sleepTime, CUSTOM, referencePoint);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param tree      The component on which to trigger the event.
     * @param nodeValue The String value of the specific node on which to trigger the event.
     * @param numberOfClicks
     *                   Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     * @param modifiers The modifier key values that need to be passed onto the event.
     * @param isPopupTrigger
     *                   boolean specifying whether this event will show a popup.
     * @param sleepTime
     *                   The wait time in ms between each event.
     * @param referencePoint
     *                        The CUSTOM mouse position within the cell.
     */
    public JTreeMouseEventData(final JFCTestCase testCase, final JTree tree,
        final String nodeValue, final int numberOfClicks, final int modifiers,
        final boolean isPopupTrigger, final long sleepTime,
        final Point referencePoint) {
        this(testCase, tree, nodeValue, numberOfClicks, modifiers,
            isPopupTrigger, sleepTime, CUSTOM, referencePoint);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param tree      The component on which to trigger the event.
         * @param treePath  The path of the specific node on which to trigger the event.
     * @param numberOfClicks
     *                   Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     * @param modifiers The modifier key values that need to be passed onto the event.
     * @param isPopupTrigger
     *                   boolean specifying whether this event will show a popup.
     * @param sleepTime
     *                   The wait time in ms between each event.
     * @param position  The relative mouse position within the cell.
     * @param referencePoint
     *                                   If position is CUSTOM then the point is a offset from
     *                                   the location of the component. If the position is PERCENT
     *                                   then the location is a percentage offset of the hight and width.
     *                                   Otherwise, the referencePoint is unused.
     */
    public JTreeMouseEventData(final JFCTestCase testCase, final JTree tree,
        final TreePath treePath, final int numberOfClicks, final int modifiers,
        final boolean isPopupTrigger, final long sleepTime, final int position,
        final Point referencePoint) {
        setTestCase(testCase);
        setSource(tree);
        setNumberOfClicks(numberOfClicks);
        setModifiers(modifiers);
        setPopupTrigger(isPopupTrigger);
        setTreePath(treePath);
        setSleepTime(sleepTime);
        setPosition(position);
        setReferencePoint(referencePoint);
        setValid(true);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param tree      The component on which to trigger the event.
     * @param nodeValue The String value of the specific node on which to trigger the event.
     * @param numberOfClicks
     *                   Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     * @param modifiers The modifier key values that need to be passed onto the event.
     * @param isPopupTrigger
     *                   boolean specifying whether this event will show a popup.
     * @param sleepTime
     *                   The wait time in ms between each event.
     * @param position  The relative mouse position within the cell.
     * @param referencePoint
     *                                   If position is CUSTOM then the point is a offset from
     *                                   the location of the component. If the position is PERCENT
     *                                   then the location is a percentage offset of the hight and width.
     *                                   Otherwise, the referencePoint is unused.
     */
    public JTreeMouseEventData(final JFCTestCase testCase, final JTree tree,
        final String nodeValue, final int numberOfClicks, final int modifiers,
        final boolean isPopupTrigger, final long sleepTime, final int position,
        final Point referencePoint) {
        setTestCase(testCase);
        setSource(tree);
        setNumberOfClicks(numberOfClicks);
        setModifiers(modifiers);
        setPopupTrigger(isPopupTrigger);
        setNodeValue(nodeValue);
        setSleepTime(sleepTime);
        setPosition(position);
        setReferencePoint(referencePoint);
        setValid(true);
    }

    /**
     * Set the attribute value.
     *
     * @param nodeValue The new value of the attribute.
     */
    public final void setNodeValue(final String nodeValue) {
        m_nodeValue = nodeValue;
    }

    /**
     * Get the attribute value.
     *
     * @return String    The value of the attribute.
     */
    public final String getNodeValue() {
        return m_nodeValue;
    }

    /**
     * Set the attribute value.
     *
     * @param tree  The new value of the attribute.
     */
    public final void setSource(final JTree tree) {
        m_tree = tree;
    }

    /**
     * Get the attribute value.
     *
     * @return JTree    The value of the attribute.
     */
    public final JTree getSource() {
        return m_tree;
    }

    /**
     * The component on which the event has to be fired.
     *
     * @return The component.
     */
    public Component getComponent() {
        // by default, the component is the same as the source
        return getSource();
    }

    /**
     * Set the attribute value.
     *
     * @param treePath The new value of the attribute.
     */
    public final void setTreePath(final TreePath treePath) {
        m_treePath = treePath;
    }

    /**
     * Get the attribute value.
     *
     * @return TreePath    The value of the attribute.
     */
    public final TreePath getTreePath() {
        return m_treePath;
    }

    /**
     * Returns true if the event can be consumed by
     * this instnace of event data.
     *
     * @param ae Event to be consumed.
     * @return true if the event was consumed.
     */
    public boolean canConsume(final AWTEvent ae) {
        if (!(ae.getSource() instanceof JTree)
                || !super.canConsume(ae)
                || !sameSource(ae)) {
            return false;
        }

        if (isValid()) {
            int row = ((JTree) ae.getSource()).getRowForLocation(
                    ((MouseEvent) ae).getX(),
                    ((MouseEvent) ae).getY());
            TreePath path = ((JTree) ae.getSource()).getPathForRow(row);

            if ((path != null) && !path.equals(getTreePath())) {
                return false;
            }
        }

        return true;
    }

    /**
     * Consume the event.
     * @param ae AWTEvent to be consumed.
     * @return boolean true if the event was consumed.
     */
    public boolean consume(final AWTEvent ae) {
        if (super.consume(ae)) {
            return true;
        }

        MouseEvent me     = (MouseEvent) ae;
        JTree      source = (JTree) me.getSource();
        int        row    = ((JTree) source).getRowForLocation(
                me.getX(),
                me.getY());
        TreePath path = ((JTree) source).getPathForRow(row);

        if (path == null) {
            return true;
        }

        setSource(source);
        setModifiers(me.getModifiers());
        setNumberOfClicks(me.getClickCount());
        setPopupTrigger(me.isPopupTrigger());
        setSleepTime(getDefaultSleepTime());

        Point p = new Point(
                me.getX(),
                me.getY());
        Point screen = source.getLocationOnScreen();
        screen.translate(p.x, p.y);
        setLocationOnScreen(screen);
        setTreePath(path);
        setNodeValue(path.getLastPathComponent().toString());

        setPosition(CENTER);
        setReferencePoint(null);

        setValid(true);

        return true;
    }

    /**
     * Compare to event datas and deterime if they are equal.
     *
     * @param o Object to be compared.
     * @return true if the events are the same.
     */
    public boolean equals(final Object o) {
        if (!super.equals(o)) {
            return false;
        }

        JTreeMouseEventData data = (JTreeMouseEventData) o;

        TreePath path     = getTreePath();
        TreePath dataPath = data.getTreePath();

        if ((path != null) && (dataPath != null) && !data.equals(m_treePath)) {
            return false;
        }

        if ((getNodeValue() != null)
                && (data.getNodeValue() != null)
                && !getNodeValue().equals(data.getNodeValue())) {
            return false;
        }

        return true;
    }

    /**
     * Get the hashCode.
     * @return int hashCode.
     */
    public int hashCode() {
        int nodeHash = 0;

        if (m_nodeValue != null) {
            nodeHash = m_nodeValue.hashCode();
        }

        return super.hashCode() + nodeHash;
    }

    /**
     * Populate the XML Element with this nodes attributes.
     * @param e element to be populated.
     */
    public void populate(final Element e) {
        super.populate(e);
        e.setAttribute(JFCXMLConstants.TYPE, "JTreeMouseEventData");

        e.setAttribute(
            JFCXMLConstants.NODEVALUE,
            getNodeValue());
    }

    /**
     * Prepare the component to receive the event.
     *
     * @return true if the component is ready to receive the event.
     */
    public boolean prepareComponent() {
        if (!isValidForProcessing(getSource())) {
            return false;
        }

        m_tree.updateUI();

        JFCTestCase testCase = getTestCase();

        if (testCase != null) {
            // try to clear the event queue
            testCase.flushAWT();
        }

        TreePath path = getTreePath();

        if (path == null) {
            path = new TreePath(buildPath(getNode(
                            m_tree,
                            getNodeValue())));
        }

        if (path == null) {
            return false;
        }

//        m_tree.setSelectionPath(path);

        if (testCase != null) {
            // try to clear the event queue
            testCase.pauseAWT();
        }
        m_tree.scrollPathToVisible(path);
        if (testCase != null) {
            testCase.flushAWT();
            testCase.pauseAWT();
        }

        Rectangle cellRect = m_tree.getPathBounds(path);

        if (testCase != null) {
            // try to clear the event queue
            testCase.flushAWT();
            testCase.pauseAWT();
        }

        Point p = calculatePoint(cellRect);

        if (!m_tree.getVisibleRect().contains(p)) {
            Rectangle vis     = m_tree.getVisibleRect();
            final Rectangle newView = new Rectangle(p.x - (int) (vis.width / 2),
                    p.y - (int) (vis.height / 2), vis.width, vis.height);
            m_tree.scrollRectToVisible(newView);
        }

        Point screen = m_tree.getLocationOnScreen();
        screen.translate(p.x, p.y);
        setLocationOnScreen(screen);

        return true;
    }

    /**
     * Get string description of event.
     * @return String description of event.
     */
    public String toString() {
        if (!isValid()) {
            return super.toString();
        }

        StringBuffer buf = new StringBuffer(1000);
        buf.append(super.toString());
        buf.append(" nodeValue: " + getNodeValue());
        buf.append(" treePath: " + getTreePath());

        return buf.toString();
    }

    /**
     * Finds the node in the specified tree by comparing
     * with the value passed in.
     *
     * @param tree    The tree containing the node
     * @param userObj The value of the node (usually the node label)
     * @return The node whose value matched the value passed in
     */
    private TreeNode getNode(final JTree tree, final String userObj) {
        if (userObj == null) {
            return null;
        }

        TreeModel model = tree.getModel();

        // if the userObj is the same as the userObject of the root, just return root
        TreeNode root = (TreeNode) model.getRoot();

        if (checkNodesUserObject(root, userObj)) {
            return root;
        }

        List nodeList = findNodeList(
                root,
                new ArrayList());
        TreeNode node = null;
        Iterator iter = nodeList.iterator();

        while (iter.hasNext()) {
            node = (TreeNode) iter.next();

            if (checkNodesUserObject(node, userObj)) {
                return node;
            }
        }

        // nothing matched - return null
        return null;
    }

    /**
     * Builds the parents of node up to and including the root node,
     * where the original node is the last element in the returned array.
     * The length of the returned array gives the node's depth in the
     * tree.
     *
     * @param node    The TreeNode whose path has to be built.
     * @return   The array of nodes in the tree above the specified node.
     */
    private Object[] buildPath(final TreeNode node) {
        ArrayList retVal = new ArrayList();

        if (node != null) {
            retVal.add(0, node);

            TreeNode parent = node.getParent();

            while (parent != null) {
                retVal.add(0, parent);
                parent = parent.getParent();
            }
        }

        retVal.trimToSize();

        return retVal.toArray();
    }

    /**
     * Checks to see if the node is not null and its
     * userObject is equal to the value passed in.
     *
     * @param node    Node
     * @param userObj Expected value
     * @return true if the values are the same
     */
    private boolean checkNodesUserObject(final TreeNode node,
        final String userObj) {
        if (node == null) {
            return false;
        }

        return ((node.toString() != null) && node.toString().equals(userObj));
    }

    /**
     * Calls itself recursively to build a list of all
     * the child nodes in the parent.
     *
     * @param parent  Node whose children have to be added to the list
     * @param outList List of all the nodes in the tree
     * @return List of all nodes
     */
    private List findNodeList(final TreeNode parent, final List outList) {
        TreeNode node;

        for (int idx = 0; idx < parent.getChildCount(); idx++) {
            node = (TreeNode) parent.getChildAt(idx);
            outList.add(node);
            findNodeList(node, outList);
        }

        return outList;
    }
}
