package junit.extensions.jfcunit.eventdata;

import junit.extensions.jfcunit.JFCTestCase;
import junit.extensions.jfcunit.xml.JFCXMLConstants;

import org.w3c.dom.Element;

import java.awt.AWTEvent;
import java.awt.Component;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.event.MouseEvent;
import javax.swing.JViewport;

/**
 * Data container class that holds all the data necessary for jfcUnit to fire mouse events.
 *
 * @author <a href="mailto:vraravam@thoughtworks.com">Vijay Aravamudhan : ThoughtWorks Inc.</a>
 */
public class MouseEventData
    extends AbstractMouseEventData {
    /**
     * Default value specifying whether the mouse event being fired
     * would trigger a popup or not.
     */
    public static final boolean DEFAULT_ISPOPUPTRIGGER = false;

    /**
     * The Component on which to trigger the event.
     */
    private Component m_comp;

    /**
     * Default Constructor.
     */
    public MouseEventData() {
        super();
        setValid(false);
    }

    /**
     * Constructor.
     *
     * @param testCase The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param comp     The component on which to trigger the event.
     */
    public MouseEventData(final JFCTestCase testCase, final Component comp) {
        this(testCase, comp, DEFAULT_NUMBEROFCLICKS);
    }

    /**
     * Constructor.
     *
     * @param testCase The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param comp     The component on which to trigger the event.
     * @param numberOfClicks
     *                 Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     */
    public MouseEventData(final JFCTestCase testCase, final Component comp,
                          final int numberOfClicks) {
        this(testCase, comp, numberOfClicks, DEFAULT_MOUSE_MODIFIERS);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param comp      The component on which to trigger the event.
     * @param sleepTime
     *                  The wait time in ms between each event.
     */
    public MouseEventData(final JFCTestCase testCase, final Component comp,
                          final long sleepTime) {
        this(testCase, comp, DEFAULT_NUMBEROFCLICKS, sleepTime);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param comp      The component on which to trigger the event.
     * @param numberOfClicks
     *                  Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     * @param modifiers The modifier key values that need to be passed onto the event.
     */
    public MouseEventData(final JFCTestCase testCase, final Component comp,
                          final int numberOfClicks, final int modifiers) {
        this(testCase, comp, numberOfClicks, modifiers, DEFAULT_ISPOPUPTRIGGER);
    }

    /**
     * Constructor.
     *
     * @param testCase The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param comp     The component on which to trigger the event.
     * @param numberOfClicks
     *                 Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     * @param isPopupTrigger
     *                 boolean specifying whether this event will show a popup.
     */
    public MouseEventData(final JFCTestCase testCase, final Component comp,
                          final int numberOfClicks,
                          final boolean isPopupTrigger) {
        this(testCase, comp, numberOfClicks,
             getDefaultModifiers(isPopupTrigger), isPopupTrigger);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param comp      The component on which to trigger the event.
     * @param numberOfClicks
     *                  Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     * @param sleepTime
     *                  The wait time in ms between each event.
     */
    public MouseEventData(final JFCTestCase testCase, final Component comp,
                          final int numberOfClicks, final long sleepTime) {
        this(testCase, comp, numberOfClicks, DEFAULT_MOUSE_MODIFIERS,
             DEFAULT_ISPOPUPTRIGGER, sleepTime);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param comp      The component on which to trigger the event.
     * @param numberOfClicks
     *                  Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     * @param modifiers The modifier key values that need to be passed onto the event.
     * @param isPopupTrigger
     *                  boolean specifying whether this event will show a popup.
     */
    public MouseEventData(final JFCTestCase testCase, final Component comp,
                          final int numberOfClicks, final int modifiers,
                          final boolean isPopupTrigger) {
        this(testCase, comp, numberOfClicks, modifiers, isPopupTrigger,
             DEFAULT_SLEEPTIME);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param comp      The component on which to trigger the event.
     * @param numberOfClicks
     *                  Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     * @param isPopupTrigger
     *                  boolean specifying whether this event will show a popup.
     * @param sleepTime
     *                  The wait time in ms between each event.
     */
    public MouseEventData(final JFCTestCase testCase, final Component comp,
                          final int numberOfClicks,
                          final boolean isPopupTrigger,
                          final long sleepTime) {
        this(testCase, comp, numberOfClicks,
             getDefaultModifiers(isPopupTrigger), isPopupTrigger, sleepTime);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param comp      The component on which to trigger the event.
     * @param numberOfClicks
     *                   Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     * @param modifiers The modifier key values that need to be passed onto the event.
     * @param isPopupTrigger
     *                   boolean specifying whether this event will show a popup.
     * @param sleepTime
     *                   The wait time in ms between each event.
     */
    public MouseEventData(final JFCTestCase testCase, final Component comp,
                          final int numberOfClicks, final int modifiers,
                          final boolean isPopupTrigger, final long sleepTime) {
        this(testCase, comp, numberOfClicks, modifiers, isPopupTrigger,
             sleepTime, DEFAULT_POSITION, null);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param comp      The component on which to trigger the event.
     * @param numberOfClicks
     *                   Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     * @param modifiers The modifier key values that need to be passed onto the event.
     * @param isPopupTrigger
     *                   boolean specifying whether this event will show a popup.
     * @param sleepTime
     *                   The wait time in ms between each event.
     * @param position  The relative mouse position within the cell.
     */
    public MouseEventData(final JFCTestCase testCase, final Component comp,
                          final int numberOfClicks, final int modifiers,
                          final boolean isPopupTrigger, final long sleepTime,
                          final int position) {
        this(testCase, comp, numberOfClicks, modifiers, isPopupTrigger,
             sleepTime, position, null);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param comp      The component on which to trigger the event.
     * @param numberOfClicks
     *                   Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     * @param modifiers The modifier key values that need to be passed onto the event.
     * @param isPopupTrigger
     *                   boolean specifying whether this event will show a popup.
     * @param sleepTime
     *                   The wait time in ms between each event.
     * @param referencePoint     The CUSTOM mouse position within the cell.
     */
    public MouseEventData(final JFCTestCase testCase, final Component comp,
                          final int numberOfClicks, final int modifiers,
                          final boolean isPopupTrigger, final long sleepTime,
                          final Point referencePoint) {
        this(testCase, comp, numberOfClicks, modifiers, isPopupTrigger,
             sleepTime, CUSTOM, referencePoint);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param comp      The component on which to trigger the event.
     * @param numberOfClicks
     *                   Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     * @param modifiers The modifier key values that need to be passed onto the event.
     * @param isPopupTrigger
     *                   boolean specifying whether this event will show a popup.
     * @param sleepTime
     *                   The wait time in ms between each event.
     * @param position  The relative mouse position within the cell.
     * @param referencePoint
     *                                   If _position is CUSTOM then the point is a offset from
     *                                   the location of the component. If the _position is PERCENT
     *                                   then the location is a percentage offset of the hight and width.
     *                                   Otherwise, the _referencePoint is unused.
     */
    public MouseEventData(final JFCTestCase testCase, final Component comp,
                          final int numberOfClicks, final int modifiers,
                          final boolean isPopupTrigger, final long sleepTime,
                          final int position,
                          final Point referencePoint) {
        setTestCase(testCase);
        setSource(comp);
        setNumberOfClicks(numberOfClicks);
        setModifiers(modifiers);
        setPopupTrigger(isPopupTrigger);
        setSleepTime(sleepTime);
        setPosition(position);
        setReferencePoint(referencePoint);
        setValid(true);
    }

    /**
     * The component on which the event has to be fired.
     *
     * @return The component.
     */
    public Component getComponent() {
        // by default, the component is the same as the source
        return getSource();
    }

    /**
     * Set the attribute value.
     *
     * @param comp   The new value of the attribute.
     */
    public void setSource(final Component comp) {
        m_comp = comp;
    }

    /**
     * Get the attribute value.
     *
     * @return Component    The value of the attribute.
     */
    public Component getSource() {
        return m_comp;
    }

    /**
     * Returns true if the event can be consumed by
     * this instnace of event data.
     *
     * @param ae Event to be consumed.
     * @return true if the event was consumed.
     */
    public boolean canConsume(final AWTEvent ae) {
        if (isValid()) {
            if (ae.getSource() != getSource()) {
                return false;
            }
        }

        return super.canConsume(ae);
    }

    /**
     * Consume the event.
     * @param ae AWTEvent to be consumed.
     * @return boolean true if the event was consumed.
     */
    public boolean consume(final AWTEvent ae) {
        if (super.consume(ae)) {
            return true;
        }

        MouseEvent me = (MouseEvent) ae;
        Component source = (Component) me.getSource();
        setSource(source);
        setModifiers(me.getModifiers());
        setNumberOfClicks(me.getClickCount());
        setPopupTrigger(me.isPopupTrigger());
        setSleepTime(getDefaultSleepTime());

        Point p = new Point(
            me.getX(),
            me.getY());

        try {
            Point screen = source.getLocationOnScreen();
            screen.translate(p.x, p.y);
            setLocationOnScreen(screen);

            setPosition(CUSTOM);
            setReferencePoint(p);
        } catch (java.awt.IllegalComponentStateException exception) {
            // Could not get the location on the screen. Ignore. The
            // component should be good enough.
        }

        setValid(true);

        return true;
    }

    /**
     * Compare to event data and deterime if they are equal.
     *
     * @param o Object to be compared.
     * @return true if the events are the same.
     */
    public boolean equals(final Object o) {
        return super.equals(o);
    }

    /**
     * Get the hashCode.
     * @return int hashCode.
     */
    public int hashCode() {
        return super.hashCode();
    }

    /**
     * Populate the XML Element with this objects data.
     * @param e XML Element to be populated.
     */
    public void populate(final Element e) {
        super.populate(e);
        e.setAttribute(JFCXMLConstants.TYPE, "MouseEventData");
    }

    /**
     * Prepare the component for firing the mouse event.
     *
     * @return boolean true if the component is ready.
     */
    public boolean prepareComponent() {
        if (!isValidForProcessing(m_comp)) {
            return false;
        }

        // Note: don't follow the scroll issue from the other MouseEventData sub-classes here
        // that is done only for specialized EventData classes.
        Rectangle bounds = m_comp.getBounds();
        insureVisibleLocation(calculatePoint(bounds));

        bounds.setLocation(m_comp.getLocationOnScreen());
        setLocationOnScreen(calculatePoint(bounds));

        return true;
    }

    /**
     * Insure the the point p is visible in the object.
     * If not then attempt to locate a viewport to
     * make the point visible.
     * @param p Point to be made visible.
     */
    public void insureVisibleLocation(final Point p) {
        // Need to get the parent viewport.
        Component parent = m_comp.getParent();
        while (parent != null
               && !(parent instanceof JViewport)) {
            parent = parent.getParent();
        }
        if (parent != null && parent instanceof JViewport) {
            JViewport port = (JViewport) parent;

            Rectangle r = port.getViewRect();
            r.x = p.x - (r.width / 2);
            r.y = p.y - (r.height / 2);

            port.scrollRectToVisible(r);
        }
    }
}
