package junit.extensions.jfcunit.eventdata;

import java.util.Arrays;
import java.util.Vector;

import java.awt.Component;
import java.awt.Container;
import javax.swing.JComponent;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JPopupMenu;
import javax.swing.JTree;
import javax.swing.tree.TreeModel;
import javax.swing.tree.TreePath;

import junit.extensions.jfcunit.finder.JMenuItemFinder;


/**
 * <p>Title: PathData</p>
 * <p>Description: This class encapsulates the data
 * which is to be transformed into a path for the given
 * GUI object type.</p>
 * <p>Copyright: Copyright (c) 2003</p>
 * <p>Company: JFCUnit Project</p>
 * @author Kevin Wilson
 * @version 1.0
 */
public class PathData {
    /**
     * Indexes of the strings representing the path.
     * Used only when there is a duplicate path element.
     */
    private int[] m_indexes;

    /**
     * Strings representing the path.
     */
    private String[] m_path;

    /**
     * PathData constructor.
     * @param size number of path elements
     */
    public PathData(final int size) {
        m_path        = new String[size];
        m_indexes     = new int[size];
    }

    /**
     * PathData constructor.
     * @param path Path to be represented.
     */
    public PathData(final String[] path) {
        this(path, null);
    }

    /**
     * PathData constructor.
     *
     * @param path String names of the items to be traversed.
     * @param indexes index array for paths containing more
     * than one identical string.
     */
    public PathData(final String[] path, final int[] indexes) {
        this.m_path = path;

        if (indexes == null) {
            this.m_indexes = new int[path.length];
            Arrays.fill(this.m_indexes, 0);
        } else {
            this.m_indexes = indexes;
        }

        if (this.m_indexes.length != path.length) {
            throw new java.lang.IllegalArgumentException(
                "Index length does not match string length");
        }
    }

    /**
     * PathData constructor.
     * @param mi MenuItem to buid the path for.
     */
    public PathData(final JMenuItem mi) {
        Vector strings = new Vector();

        // This loop builds up the name of the field from each of the
        // higher level menu's text labels.
        strings.insertElementAt(
            mi.getText(),
            0);

        Component parent = ((JPopupMenu) mi.getParent()).getInvoker();

        while (parent instanceof JMenuItem) {
            String text = ((JMenuItem) parent).getText();
            strings.insertElementAt(text, 0);

            if (parent instanceof JPopupMenu) {
                parent = ((JPopupMenu) parent.getParent()).getInvoker();
            } else if (parent instanceof JMenu) {
                parent = ((JMenu) parent).getParent();

                if (parent instanceof JPopupMenu) {
                    parent = ((JPopupMenu) parent).getInvoker();
                }
            }
        }

        this.m_path     = (String[]) strings.toArray(new String[0]);
        m_indexes       = new int[m_path.length];
        Arrays.fill(m_indexes, 0);
    }

    /**
     * Get the root object.
     * @param mi Menu item to retrieve the root menu for.
     * @return Root object.
     */
    public Object getRoot(final JMenuItem mi) {
        Component lastParent = null;
        Component parent = mi;

        while (parent instanceof JMenuItem) {
            lastParent     = parent;
            parent         = parent.getParent();

            if (parent instanceof JPopupMenu) {
                lastParent     = parent;
                parent         = ((JPopupMenu) parent).getInvoker();
            }
        }

        if (parent instanceof JMenuBar) {
            return parent;
        }

        return lastParent;
    }

    /**
     * set the path element at a given level.
     * @param i      Level of the path.
     * @param data   String of path element.
     * @param index  Index of path element.
     */
    public final void set(final int i, final String data, final int index) {
        m_path[i]        = data;
        m_indexes[i]     = index;
    }

    /**
     * Get the indexes for the JComponent.
     * @param menu menu to be traversed.
     * @return int[] indexes to be traversed.
     */
    public int[] getIndexes(final JComponent menu) {
        int[]           result  = new int[m_path.length];
        int             i       = 0;
        Container       parent  = menu;
        JMenuItemFinder mFinder = new JMenuItemFinder(m_path[i]);
        mFinder.setWait(0);

        Component c = mFinder.find(menu, m_indexes[i]);

        if (menu instanceof JMenuBar) {
            result[i++] = ((JMenuBar) menu).getComponentIndex(c);
        } else if (menu instanceof JPopupMenu) {
            result[i++] = ((JPopupMenu) menu).getComponentIndex(c);
        }

        for (; i < m_path.length; i++) {
            if (c instanceof JMenu) {
                JMenu m = (JMenu) c;
                mFinder = new JMenuItemFinder(m_path[i]);

                JPopupMenu pm = m.getPopupMenu();
                c = mFinder.find(pm, m_indexes[i]);

                // Locate the element index
                Component[] comps = pm.getComponents();
                result[i] = -1;
                for (int j = 0; j < comps.length && result[i] == -1; j++) {
                    if (comps[j] == c) {
                        result[i] = j;
                    }
                }
            }
        }

        return result;
    }

    /**
     * Generate a tree path for this path and the given tree.
     * @param tree JTree Tree to construct the path for.
     * @return TreePath TreePath to be used in firing events.
     */
    public TreePath getTreePath(final JTree tree) {
        if (tree == null) {
            throw new IllegalArgumentException("Tree cannot be null");
        }
        if (m_path.length == 0) {
            return null;
        }

        TreeModel model = tree.getModel();
        Object    node = model.getRoot();
        Vector    path = new Vector();

        for (int i = 0; i < this.m_path.length; i++) {
            String  text     = m_path[i];
            int     idx      = m_indexes[i];
            boolean found    = false;
            boolean root     = false;
            int     children;

            if (i == 0) {
                root         = true;
                children     = 1;
            } else {
                children = model.getChildCount(node);
            }

            for (int j = 0; (j < children) && !found; j++) {
                Object newnode;

                if (root) {
                    newnode = node;
                } else {
                    newnode = model.getChild(node, j);
                }

                if (text.equals(newnode.toString())) {
                    if (idx == 0) {
                        node = newnode;
                        path.add(node);
                        found = true;
                    } else {
                        idx--;
                    }
                }
            }

            if (!found) {
                return null;
            }
        }

        TreePath tp = new TreePath(path.toArray());

        return tp;
    }

    //  public int[] getIndexes(JPopupMenu menu) {
    //    int[] result = new int[path.length];
    //    int i = 0;
    //    Container parent = menu;
    //    JMenuItemFinder mFinder = new JMenuItemFinder(path[i]);
    //    Component c = TestHelper.findComponent(mFinder, menu, indexes[i]);
    //    result[i++] = menu.getComponentIndex(c);
    //
    //    for (; i < path.length; i++) {
    //      if (c instanceof JMenu) {
    //        JMenu m = (JMenu) c;
    //        mFinder = new JMenuItemFinder(path[i]);
    //        JPopupMenu pm = m.getPopupMenu();
    //        c = TestHelper.findComponent(mFinder, pm, indexes[i]);
    //        result[i] = pm.getComponentIndex(c);
    //      }
    //    }
    //    return result;
    //  }
}
