package junit.extensions.jfcunit.eventdata;

import junit.extensions.jfcunit.xml.JFCXMLConstants;

import junit.extensions.xml.IXMLTestCase;
import junit.extensions.xml.XMLException;
import junit.extensions.xml.elements.AbstractTagHandler;

import org.w3c.dom.Element;

import java.util.StringTokenizer;


/**
 * <p>Title: PathDataTagHandler </p>
 * <p>Description: Translates the attributes into a PathData element.</p>
 * <H3>Required</H3>
 * path - Path string
 * <H3>Optional Attributes</H3>
 * <pre>
 * id - used to save the value in the property cache.
 * indexdelimiter - delimiter character separating the text from the index.
 *     Default value "."
 * pathdelimiter - delimiter separating each path element
 *     Default value ":"
 * </pre>
 * <H3>Example</H3>
 * &lt;pathdata path="Level1.0:Level2.1:Level3.0"/&gt;
 *
 * <p>Copyright: Copyright (c) 2003</p>
 * <p>Company: JFCUnit Project</p>
 * @author Kevin Wilson
 * @version 1.0
 */
public class PathTagHandler extends AbstractTagHandler
    implements JFCXMLConstants {
    /**
     * Path data created.
     */
    private PathData m_pathData = null;

    /**
     * Constructor for the tag handler.
     * @param element Element to be processed.
     * @param testCase Parent testCase.
     */
    public PathTagHandler(final Element element, final IXMLTestCase testCase) {
        super(element, testCase);
    }

    /**
     * get the path data.
     * @return Path data.
     * @throws XMLException may be thrown if required
     * attributes are not present.
     */
    public PathData getPathData() throws XMLException {
        if (m_pathData == null) {
            processElement();
        }

        return m_pathData;
    }

    /**
     * Process the element.
     * @throws XMLException is thrown if the element cannot be understood.
     */
    public void processElement() throws XMLException {
        String indexDelimiter = super.getString(INDEXDELIMITER);

        if ((indexDelimiter == null) || (indexDelimiter.length() == 0)) {
            indexDelimiter = ".";
        }

        String pathDelimiter = super.getString(PATHDELIMITER);

        if ((pathDelimiter == null) || (pathDelimiter.length() == 0)) {
            pathDelimiter = ":";
        }

        String          path = getString(PATH);
        String          name = getString(ID);

        StringTokenizer tok = new StringTokenizer(path, pathDelimiter);
        m_pathData = new PathData(tok.countTokens());

        int i = 0;

        while (tok.hasMoreTokens()) {
            int    index = 0;
            String token = tok.nextToken();
            int    split = token.indexOf(indexDelimiter);

            if (split > 0) {
                index     = Integer.parseInt(token.substring(split + 1));
                token     = token.substring(0, split);
            }

            m_pathData.set(i++, token, index);
        }

        if (name != null) {
            getXMLTestCase().addProperty(name, m_pathData);
        }
    }

    /**
     * Validate the element has a path attribute.
     * @throws XMLException when PATH is not a valid attribute.
     */
    public void validateElement() throws XMLException {
        super.checkRequiredAttribute(PATH);
    }
}
