package junit.extensions.jfcunit.finder;

import java.awt.image.BufferedImage;
import java.awt.image.PixelGrabber;

import java.util.Arrays;

import javax.swing.Icon;


/**
 * Class for comparing that two {@link javax.swing.Icon}s match.
 *
 * @author <a href="mailto:vraravam@thoughtworks.com">Vijay Aravamudhan : ThoughtWorks Inc.</a>
 * @author Kevin Wilson
 */
public class IconMatcher {
    /**
     * The icon of the first component.
     */
    private Icon m_icon = null;

    /**
     * iconData read from the icon given.
     */
    private int[] m_iconData = null;

    /**
     * Height of the icon.
     */
    private int m_ich = 0;

    /**
     * width of the icon.
     */
    private int m_icw = 0;

    /**
     * Increment for performing spot checks across the icons
     * to speed performance of not equals comparisons.
     */
    private int m_incr = 1;

    /**
     * Size of the icon.
     */
    private int m_size = 0;

    /**
     * Constructor accepting all arguments needed to filter the component.
     *
     * @param icon   The desired pattern for the icon of the component.
     * @throws InterruptedException when the PixelGrabber.grabPixels()
     * is interrupted.
     */
    public IconMatcher(final Icon icon) throws InterruptedException {
        setIcon(icon);
    }

    /**
     * Set the Icon to be matched.
     * @param icon Icon icon to match.
     * @throws InterruptedException may be thrown when
     * converting the icon to an array.
     */
    public final void setIcon(final Icon icon) throws InterruptedException {
        m_icon = icon;

        if (icon != null) {
            m_iconData     = iconToArray(icon);
            m_icw          = icon.getIconWidth();
            m_ich          = icon.getIconHeight();
            m_size         = m_icw * m_ich;

            int min        = Math.min(m_icw, m_ich);

            if (min > 32) {
                m_incr = min / 8;
            } else {
                m_incr = min / 4;
            }
        }
    }

    /**
     * Get the icon to be matched.
     * @return Icon icon to be matched.
     */
    public final Icon getIcon() {
        return m_icon;
    }

    /**
     * Method that returns true if the given icon matches the one in this matcher.
     *
     * @param other   The other Icon to be compared
     * @return true if this Icon is a match or the icon which this
     *              matcher is created for is null.
     */
    public boolean matches(final Icon other) {
        // If the icon on this finder is null, then we treat the
        // finder as a don't care or always match.
        if (m_icon == null) {
            return true;
        }

        if (other == null) {
            return false;
        }

        if ((m_icw != other.getIconWidth()) || (m_ich != other.getIconHeight())) {
            return false;
        }

        int[] otherData = null;

        try {
            otherData = iconToArray(other);
        } catch (InterruptedException ie) {
            // Ignore
        }

        if ((otherData == null) || (m_iconData.length != otherData.length)) {
            return false;
        }

        // Spot check
        for (int i = 0; i < m_size; i += m_incr) {
            if (m_iconData[i] != otherData[i]) {
                return false;
            }
        }

        // If we make it this far then lets compare the whole array.
        return Arrays.equals(m_iconData, otherData);
    }

    /**
     * Paint the icon into an array of int.
     * @param icon Icon to be painted.
     * @return int[] containing the painted icon.
     * @throws InterruptedException when
     * PixelGrabber.grabPixels() is interrupted.
     */
    private int[] iconToArray(final Icon icon) throws InterruptedException {
        int width  = icon.getIconWidth();
        int height = icon.getIconHeight();

        if (width < 0) {
            width = 1;
        }

        if (height < 0) {
            height = 1;
        }

        int[]         array = new int[width * height];
        BufferedImage bf = new BufferedImage(width, height,
                BufferedImage.TYPE_INT_ARGB);

        icon.paintIcon(
            null,
            bf.getGraphics(),
            0,
            0);

        PixelGrabber grab = new PixelGrabber(bf, 0, 0, width, height, array, 0,
                width);
        grab.grabPixels();

        return array;
    }
}
