package junit.extensions.jfcunit.keyboard;

import java.util.HashMap;


/**
 * Abstract key mapping is used to translate key codes
 * or characters into key strokes. Any characters or
 * key codes which do not map are dropped.<br>
 * <br>
 * To create a new mapping:<br>
 * 1) extend this class<br>
 * 2) Define the Object[][]: See {@link DefaultKeyMapping} for example.<br>
 * 3) Create a default constructor that passes the code array
 *    to this class.<br>
 * 4) Install the class into the {@link junit.extensions.jfcunit.eventdata.AbstractKeyEventData} by calling
 *    <code>AbstractKeyEventData.setKeyMapping(yourMapping)</code><br>
 * <br>
 * Note: Not all key codes are valid on all systems with the {@link java.awt.Robot}.
 * Invalid keycodes or modifiers for the OS will throw exceptions.
 * Extensions of this class can be used to allow users to map their
 * keyboard mappings.
 *
 * @author <a href="mailto:kwilson227@sourceforge.net">Kevin Wilson</a>
 */
public abstract class AbstractKeyMapping implements KeyMapping {
    /**
     * The hash map is used to quickly access
     * the key strokes for characters.
     */
    private final HashMap m_mapChar = new HashMap();

    /**
     * The hash map is used to quickly access
     * the key strokes for codes.
     */
    private final HashMap m_mapCode = new HashMap();

    /**
     * Constructor that loads the two distinct mappings of the charMap
     * into a HashMap for quick access.
     *
     * @param charMap mapping to be loaded.
     */
    public AbstractKeyMapping(final Object[][] charMap) {
        for (int i = 0; i < charMap.length; i++) {
            Object         c     = charMap[i][0];
            JFCKeyStroke[] codes = new JFCKeyStroke[charMap[i].length - 1];

            for (int j = 1; j < charMap[i].length; j++) {
                codes[j - 1] = (JFCKeyStroke) charMap[i][j];
            }

            if (c instanceof Character) {
                m_mapChar.put(c, codes);
            }

            if (c instanceof Integer) {
                m_mapCode.put(c, codes);
            }
        }
    }

    /**
     * Get the key codes required to construct
     * the character.
     *
     * @param c Character to be constructed.
     * @return JFCKeyStroke[] containing the key strokes
     * used to create the character.
     */
    public JFCKeyStroke[] getKeyStrokes(final char c) {
        return getKeyStrokes(
            new Character(c),
            m_mapChar);
    }

    /**
     * Get the key strokes required to construct
     * the keyCode.
     *
     * @param keyCode Key code to be constructed.
     * @return {@link JFCKeyStroke}[] containing the key strokes
     * used to enter the key code.
     */
    public JFCKeyStroke[] getKeyStrokes(final int keyCode) {
        return getKeyStrokes(
            new Integer(keyCode),
            m_mapCode);
    }

    /**
     * Get the key strokes form the mapping specified.
     *
     * @param key Key object to be found.
     * @param map Map to look for the key in.
     * @return {@link JFCKeyStroke}[] array of key strokes copied from
     * the mapping.
     */
    private JFCKeyStroke[] getKeyStrokes(final Object key, final HashMap map) {
        JFCKeyStroke[] codes = (JFCKeyStroke[]) map.get(key);

        if (codes == null) {
            return new JFCKeyStroke[0];
        }

        JFCKeyStroke[] retCodes = new JFCKeyStroke[codes.length];

        for (int i = 0; i < codes.length; i++) {
            retCodes[i] = new JFCKeyStroke(codes[i]);
        }

        return retCodes;
    }
}
