package junit.extensions.jfcunit.keyboard;

import java.awt.event.KeyEvent;


/**
 * The JFCKeyStroke class holds all of the data pertaining to
 * a keystroke to be pressed/released by the TestHelper.
 *
 * @author Kevin Wilson
 */
public class JFCKeyStroke {
    /**
     * isTyped true if the key stroke is to emit a typed event.
     */
    private boolean m_isTyped;

    /**
     * Key Character.
     */
    private char m_keyChar;

    /**
     * Key Code.
     */
    private int m_keyCode;

    /**
     * Modifiers to be pressed for this event.
     */
    private int m_modifiers;

    /**
     * Constructor.
     *
     * @param c Character.
     * @param code Key code.
     * @param modifiers Modifiers to be pressed when sending the keystroke.
     * @param typed true if a typed event should be emitted.
     */
    public JFCKeyStroke(final char c, final int code, final int modifiers,
        final boolean typed) {
        setKeyChar(c);
        setKeyCode(code);
        setModifiers(modifiers);
        setTyped(typed);
    }

    public JFCKeyStroke(String keyDesc) {
        int modifiers = 0;
        boolean typed = true;
        String key = new String(keyDesc);
        if (key.indexOf("ctrl") != -1) {
            modifiers += KeyEvent.CTRL_MASK;
        }
        if (key.indexOf("altGr") != -1) {
            modifiers += KeyEvent.ALT_MASK;
            key = key.replaceFirst("altGr", " ");
            typed = false;
        }
        if (keyDesc.indexOf("alt") != -1) {
            modifiers += KeyEvent.ALT_MASK;
            typed = false;
        }
        if (key.indexOf("shift") != -1) {
            modifiers += KeyEvent.SHIFT_MASK;
        }
        setModifiers(modifiers);
        setTyped(typed);

        key = key.trim();
        char c = key.charAt(key.length() - 1);
        setKeyChar(c);
        setKeyCode(c);
    }

    /**
     * Copy Constructor.
     *
     * @param stroke Keystroke to be copied.
     */
    public JFCKeyStroke(final JFCKeyStroke stroke) {
        setKeyChar(stroke.getKeyChar());
        setKeyCode(stroke.getKeyCode());
        setModifiers(stroke.getModifiers());
        setTyped(stroke.isTyped());
    }

    /**
     * Get the key character.
     *
     * @return char key character.
     */
    public final char getKeyChar() {
        return m_keyChar;
    }

    /**
     * Get the key code.
     *
     * @return int key code
     */
    public final int getKeyCode() {
        return m_keyCode;
    }

    /**
     * Set the modifiers.
     *
     * @param modifiers Modifiers to be pressed for this keystroke.
     */
    public void setModifiers(final int modifiers) {
        this.m_modifiers = modifiers;
    }

    /**
     * Get the modifiers.
     *
     * @return int modifiers.
     */
    public final int getModifiers() {
        return m_modifiers;
    }

    /**
     * Is the keystroke to emit a typed event.
     *
     * @return true if the typed event is to be emitted.
     */
    public final boolean isTyped() {
        return m_isTyped;
    }

    /**
     * Test for equality.
     *
     * @param o Object to be tested for equality.
     * @return true if the object is equal to this object.
     */
    public boolean equals(final Object o) {
        if (o instanceof JFCKeyStroke) {
            JFCKeyStroke stroke = (JFCKeyStroke) o;

            return ((getKeyChar() == stroke.getKeyChar())
            && (getKeyCode() == stroke.getKeyCode())
            && (getModifiers() == stroke.getModifiers())
            && (m_isTyped == stroke.isTyped()));
        }

        return false;
    }

    /**
     * Get hashCode.
     * @return int hashCode.
     */
    public int hashCode() {
        return super.hashCode() + getKeyChar() + getKeyCode() + getModifiers();
    }

    /**
     * Create a string representation of the keystroke.
     *
     * @return String representation of the keystroke.
     */
    public String toString() {
        StringBuffer buf = new StringBuffer(100);

        if ((getKeyChar() < 26) && (getModifiers() == 2)) {
            buf.append("<control ");

            int x = getKeyChar();
            x += ('A' - 1);
            buf.append((char) x);
            buf.append(">");
        } else {
            buf.append("Char:" + getKeyChar());
            buf.append(" Code:" + getKeyCode());
            buf.append(" Mods:" + getModifiers());
            buf.append(" IsTyped:" + isTyped());
        }

        return buf.toString();
    }

    /**
     * Set the key character.
     *
     * @param c Character.
     */
    private void setKeyChar(final char c) {
        m_keyChar = c;
    }

    /**
     * Set the key code.
     *
     * @param code Key code
     */
    private void setKeyCode(final int code) {
        m_keyCode = code;
    }

    /**
     * Set the typed state.
     *
     * @param typed true if the keystroke is to emit a typed event.
     */
    private void setTyped(final boolean typed) {
        m_isTyped = typed;
    }
}
