package junit.extensions.jfcunit.xml;

import junit.extensions.jfcunit.TestHelper;
import junit.extensions.jfcunit.eventdata.AbstractEventData;
import junit.extensions.jfcunit.eventdata.AbstractKeyEventData;
import junit.extensions.jfcunit.eventdata.DragEventData;
import junit.extensions.jfcunit.eventdata.JFCEventDataListener;
import junit.extensions.jfcunit.eventdata.JFCEventManager;
import junit.extensions.jfcunit.eventdata.KeyEventData;
import junit.extensions.jfcunit.eventdata.MouseEventData;
import junit.extensions.jfcunit.eventdata.MouseWheelEventData;
import junit.extensions.jfcunit.finder.AbstractButtonFinder;
import junit.extensions.jfcunit.finder.ComponentFinder;
import junit.extensions.jfcunit.finder.Finder;
import junit.extensions.jfcunit.finder.JMenuItemFinder;
import junit.extensions.jfcunit.finder.NamedComponentFinder;
import junit.extensions.jfcunit.keyboard.JFCKeyStroke;
import junit.extensions.jfcunit.tools.Operator;

import junit.extensions.xml.IXMLTestCase;
import junit.extensions.xml.XMLException;
import junit.extensions.xml.elements.SaveTagHandler;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;

import java.awt.Component;
import java.awt.Container;
import java.awt.Point;
import java.awt.Window;

import javax.swing.AbstractButton;
import javax.swing.JDialog;
import javax.swing.JFrame;
import javax.swing.JMenu;
import javax.swing.JMenuItem;
import javax.swing.JPopupMenu;


/**
 * <p>Title: class XMLRecorder.</p>
 * <p>Description: </p>
 * <p>Copyright: Copyright (c) 2002</p>
 * <p>Company: Hewlett-Packard Company</p>
 * @author Kevin Wilson
 * @version 1.0
 *
 * Attributes:
 *    terminator=8 sets the terminator to ctrl-h
 *       improved parsing to be added later.
 */
public class XMLRecorder extends SaveTagHandler implements JFCEventDataListener,
    JFCXMLConstants {
    /**
     * Replay mode.
     */
    private static boolean s_replay = false;

    /**
     * Document of the recording.
     */
    private Document m_doc = null;

    /**
     * Root element of the recording.
     */
    private Element m_root = null;

    /**
     * Indent string for each node.
     */
    private String m_indent = null;

    /**
     * Terminator keystroke.
     */
    private JFCKeyStroke[] m_terminator = null;

    /**
     * Paused State.
     */
    private volatile boolean m_paused = false;

    /**
     * Index of next generated component.
     */
    private int m_compIndex = -1;

    /**
     * Constructor for the XML Recorder.
     * @param element element to be recorded.
     * @param testCase Test Case for the recording.
     */
    public XMLRecorder(final Element element, final IXMLTestCase testCase) {
        super(element, testCase);
        m_root     = (Element) element.getParentNode();
        m_doc      = element.getOwnerDocument();

        Node previousSibling = element.getPreviousSibling();
        int keyCode = super.getInt(element, TERMINATOR, 4);
        JFCKeyStroke[] ks = TestHelper.getKeyMapping().getKeyStrokes(keyCode);
        setTerminator(ks);

        if (previousSibling.getNodeType() == Node.TEXT_NODE) {
            m_indent = previousSibling.getNodeValue();
        }
    }

    /**
     * If the replay mode is set then the record elements are ignored.
     * @param value true if the record elements are to be ignored.
     */
    public static void setReplay(final boolean value) {
        s_replay = value;
    }

    /**
     * Set the terminator to be used to end the
     * recording session.
     * @param stroke Stroke to end the recording.
     */
    public void setTerminator(final JFCKeyStroke[] stroke) {
        m_terminator = stroke;
    }

    /**
     * Get the current record Terminator.
     * @return JFCKeyStroke which terminates the session.
     */
    public JFCKeyStroke[] getTerminator() {
        if (m_terminator == null) {
            m_terminator = getDefaultTerminator();
        }

        return m_terminator;
    }

    /**
     * Generate the finder node(s) to find the component.
     * @param comp Component to be found.
     * @return Name of the component.
     */
    public String generateFind(final Component comp) {
        String name = getXMLTestCase().getPropertyName(comp);

        if ((name != null) && (name.length() != 0)) {
            return name;
        }

        if (comp instanceof JMenuItem) {
            return generateMenuFind(comp);
        }

        Window w = null;

        if (!(comp instanceof Window)) {
            w = javax.swing.SwingUtilities.getWindowAncestor(comp);
        }

        String windowName = generateWindowFind(w);
        name = "Component" + getComponentIndex();

        String className  = comp.getClass().getName();
        String named      = comp.getName();
        String finderName = null;
        Finder finder     = null;
        String label      = null;

        if (named != null) {
            finderName     = "NamedComponentFinder";
            finder         = new NamedComponentFinder(
                    comp.getClass(),
                    named);
        } else if (comp instanceof AbstractButton
                && (((AbstractButton) comp).getText() != null)) {
            finderName     = "AbstractButtonFinder";
            label          = ((AbstractButton) comp).getText();
            finder         = new AbstractButtonFinder(label);
        } else {
            finderName     = "ComponentFinder";
            finder         = new ComponentFinder(comp.getClass());
        }

        int index = TestHelper.indexOf(finder, w, comp);

        if (index == -1) {
            windowName     = null;
            index          = TestHelper.indexOf(finder, null, comp);
        }

        if (index == -1) {
            Node comment = m_doc.createComment(
                    "Assuming zero index. Could not find index of: "
                    + comp.getClass());
            insertNode(comment);
            index = 0;
        }

        Element e = m_doc.createElement(FIND);
        e.setAttribute(FINDER, finderName);
        e.setAttribute(ID, name);

        if (label != null) {
            e.setAttribute(LABEL, label);
        }

        e.setAttribute(CLASS, className);

        if (named != null) {
            e.setAttribute(NAME, named);
        }

        e.setAttribute(
            INDEX,
            Integer.toString(index));
        e.setAttribute(
            OPERATION,
            Operator.Operation.toString(Operator.Operation.EQUALS));

        if (windowName != null) {
            e.setAttribute(CONTAINER, windowName);
        }

        insertNode(e);
        getXMLTestCase().addProperty(name, comp);

        return name;
    }

    /**
     * Handle the event data by generating XML.
     * @param evtData event date generated.
     */
    public void handleEvent(final AbstractEventData evtData) {
        if (evtData.getComponent() == null) {
            return;
        }

        if (evtData instanceof KeyEventData) {
            JFCKeyStroke[] terminator = getTerminator();
            JFCKeyStroke[] strokes = ((KeyEventData) evtData).getKeyStrokes();

            if (java.util.Arrays.equals(terminator, strokes)) {
                stop();
                System.err.println("Stop recording for test: "
                    + this.getTestCase().getName());

                return;
            }
        }

        if (evtData instanceof DragEventData) {
            DragEventData ded      = (DragEventData) evtData;
            Component     comp     = ded.getSource().getComponent();
            String        sourceId = generateFind(comp);
            Element       src      = m_doc.createElement(SOURCE);
            ded.getSource().populate(src);
            src.setAttribute(REFID, sourceId);

            Element e = m_doc.createElement(DRAG);
            e.appendChild(src);

            if (ded.getDest() != null) {
                Element dst    = m_doc.createElement(DESTINATION);
                String  destId = generateFind(ded.getDest().getComponent());
                ded.getDest().populate(dst);
                dst.setAttribute(REFID, destId);
                e.appendChild(dst);
            }

            if (ded.getPoints() != null) {
                java.awt.Point[] p = ded.getPoints();

                for (int i = 0; i < p.length; i++) {
                    Element pnt = m_doc.createElement(POINT);
                    String  ref = "" + p[i].x + "," + p[i].y;
                    pnt.setAttribute(REFERENCE, ref);
                    e.appendChild(pnt);
                }
            }

            insertNode(e);
        } else if (evtData instanceof MouseWheelEventData) {
            Element e      = m_doc.createElement("wheel");
            String  source = generateFind(evtData.getComponent());
            e.setAttribute(REFID, source);
            evtData.populate(e);
            insertNode(e);
        } else if (evtData instanceof AbstractKeyEventData) {
            String  source = generateFind(evtData.getComponent());
            Element e = m_doc.createElement(KEY);
            e.setAttribute(REFID, source);
            evtData.populate(e);
            insertNode(e);
        } else if (evtData instanceof AbstractEventData) {
            String  source = generateFind(evtData.getComponent());
            Element e = m_doc.createElement(CLICK);
            e.setAttribute(REFID, source);
            evtData.populate(e);
            insertNode(e);
        }
    }

    /**
     * @see junit.extensions.xml.elements.AbstractTagHandler#processElement().
     * @throws XMLException is thrown if the element cannot be understood.
     */
    public void processElement() throws XMLException {
        validateElement();

        if (s_replay) {
            return;
        }

        String term = getString(TERMINATOR);
        if (term != null && term.length() > 0) {
            setTerminator(new JFCKeyStroke[] {
                new JFCKeyStroke(term)
            });
        }
        record();
        pause();
        super.processElement();
    }

    /**
     * @see junit.extensions.xml.elements.AbstractTagHandler#validateElement().
     * @throws XMLException if the required elements are not present.
     */
    public void validateElement() throws XMLException {
        // do the default validations from the super class
        super.validateElement();

        // check the element tag name
        checkElementTagName("record");
    }

    /**
     * Get the default terminator.
     * @return JFCKeyStroke used to terminate the record session.
     */
    protected JFCKeyStroke[] getDefaultTerminator() {
        // Control D
        return TestHelper.getKeyMapping().getKeyStrokes(4);
    }

    /**
     * Get the component index.
     * @return int index of component.
     */
    private int getComponentIndex() {
        if (m_compIndex == -1) {
            int      max   = 0;
            String[] names = getXMLTestCase().getPropertyNames();

            for (int i = 0; i < names.length; i++) {
                int j;

                for (j = names[i].length() - 1;
                        Character.isDigit(names[i].charAt(j)) && (j > 0);
                        j--) {
                    // Nothing to do.
                }

                ;

                if (j < (names[i].length() - 1)) {
                    int number = Integer.parseInt(names[i].substring(j + 1));
                    max = Math.max(max, number);
                }
            }

            m_compIndex = max + 1;
        }

        return m_compIndex++;
    }

    /**
     * Generate the XML elements to locate a menu item.
     * @param comp Component to be located.
     * @return Name of the component.
     */
    private String generateMenuFind(final Component comp) {
        JMenuItem mi          = (JMenuItem) comp;
        String    label       = mi.getText();
        String    containerID = null;
        Container parent      = mi.getParent();

        if (parent instanceof JPopupMenu) {
            JPopupMenu menu    = (JPopupMenu) mi.getParent();
            Component  invoker = menu.getInvoker();

            if (invoker instanceof JMenuItem) {
                containerID = generateFind(invoker);

                MouseEventData data = new MouseEventData(null, invoker, 0);
                data.setNumberOfClicks(0);
                data.setModifiers(0);

                Element e = m_doc.createElement(CLICK);
                e.setAttribute(REFID, containerID);
                data.populate(e);
                insertNode(e);

                if (invoker instanceof JMenu) {
                    // Get the direction of the child menu.
                    Point popupLoc = ((JMenu) invoker).getPopupMenu()
                                      .getLocation();
                    Point invokerLoc = invoker.getLocation();
                    int   xPercent   = 110;

                    if (popupLoc.x < invokerLoc.x) {
                        xPercent = -10;
                    }

                    data.setPosition(AbstractEventData.PERCENT);
                    data.setReferencePoint(new Point(xPercent, 50));

                    Element moveRight = m_doc.createElement(CLICK);
                    moveRight.setAttribute(REFID, containerID);
                    data.populate(moveRight);
                    insertNode(moveRight);
                }

                Element sleep = m_doc.createElement("sleep");
                sleep.setAttribute(DURATION, "500");
                insertNode(sleep);
            }
        } else {
            Window w = javax.swing.SwingUtilities.getWindowAncestor(comp);
            containerID = generateWindowFind(w);
        }

        String name = "MenuItem" + getComponentIndex();

        String className  = comp.getClass().getName();
        String named      = comp.getName();
        String finderName = "JMenuItemFinder";
        Finder finder     = new JMenuItemFinder(label);
        finder.setOperation(Operator.Operation.EQUALS);

        int index = TestHelper.indexOf(finder, null, comp);

        // Kluge component may not be visible
        if (index == -1) {
            index = 0;
        }

        Element e = m_doc.createElement(FIND);

        if (containerID != null) {
            e.setAttribute(CONTAINER, containerID);
        }

        e.setAttribute(FINDER, finderName);
        e.setAttribute(ID, name);
        e.setAttribute(LABEL, label);
        e.setAttribute(
            INDEX,
            Integer.toString(index));
        e.setAttribute(
            OPERATION,
            Operator.Operation.toString(Operator.Operation.EQUALS));

        insertNode(e);
        getXMLTestCase().addProperty(name, comp);

        return name;
    }

    /**
     * Generate the XML Nodes to find the window given.
     * @param w Window to be located.
     * @return Name of the window.
     */
    private String generateWindowFind(final Component w) {
        String name   = getXMLTestCase().getPropertyName(w);
        String finder = null;
        String title  = null;

        if (name != null) {
            return name;
        }

        // The window was not previously found,
        // so lets generate the find.
        if (w instanceof JFrame) {
            name       = "JFrame" + getComponentIndex();
            title      = ((javax.swing.JFrame) w).getTitle();
            finder     = "FrameFinder";
        } else if (w instanceof JDialog) {
            name       = "JDialog" + getComponentIndex();
            title      = ((javax.swing.JDialog) w).getTitle();
            finder     = "DialogFinder";
        }

        if (title == null) {
            title = "";
        }

        if (name != null) {
            getXMLTestCase().addProperty(name, w);

            Element find = m_doc.createElement(FIND);
            find.setAttribute(FINDER, finder);
            find.setAttribute(TITLE, title);

            String operation = Operator.Operation.toString(Operator.Operation.EQUALS);
            find.setAttribute(OPERATION, operation);
            find.setAttribute(ID, name);
            find.setAttribute(INDEX, "0");
            insertNode(find);
        }

        return name;
    }

    /**
     * Insert the node into the tree.
     * @param e The node to be inserted.
     */
    private void insertNode(final Node e) {
        Node parent = getElement().getParentNode();
        parent.insertBefore(
            e,
            getElement());

        Node text = m_doc.createTextNode(m_indent);
        parent.insertBefore(
            text,
            getElement());
    }

    /**
     * Pause the recording.
     */
    private void pause() {
        m_paused = true;

        // stop the test case to allow the recording.
        while (m_paused) {
            try {
                Thread.yield();
                Thread.currentThread().sleep(500);
            } catch (java.lang.InterruptedException ie) {
                // Ignore
            }
        }
    }

    /**
     * Start the recording.
     */
    private void record() {
        JFCKeyStroke[] strokes = getTerminator();
        System.err.println("Start recording for test: "
            + this.getTestCase().getName());
        System.err.println("Press " + strokes[0].toString()
            + " to complete the recording.");

        Node comment = m_doc.createComment("started recording "
                + new java.util.Date().toString());
        this.insertNode(comment);
        JFCEventManager.setRecording(true);
        JFCEventManager.getEventManager().addJFCEventDataListener(this);
        ((JFCXMLTestCase) getTestCase()).flushAWT();
    }

    /**
     * Stop the recording.
     */
    private void stop() {
        JFCEventManager.getEventManager().removeJFCEventDataListener(this);
        JFCEventManager.setRecording(false);
        m_paused = false;
    }
}
