package junit.extensions.xml;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;

import org.xml.sax.SAXException;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;

import java.net.URL;

import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;


/**
 * Interface layer to w3c dom.
 *
 * @author Kevin Wilson
 */
public final class XMLUtil implements XMLConstants {
    /**
     * Private constructor: so that this class is not instantiated.
     */
    private XMLUtil() {
    }

    /**
     * A utility method that will get the attribute value,
     * and if not found, will return null.
     *
     * @param element    The element to be searched
     * @param name       The name of the attribute to be searched for
     * @return String    The string value of the attribute, null
     * if the value is not set or is an empty string
     */
    public static String getAttribute(final Element element, final String name) {
        String value = element.getAttribute(name);

        if ((value != null) && (value.length() > 0)) {
            return value;
        }

        return null;
    }

    /**
     * A utility method that will get the attribute value, and if not found, will return false.
     *
     * @param element    The element to be searched
     * @param name       The name of the attribute to be searched for
     * @return boolean   The boolean value of the attribute, false if the value is not set or is an
     * empty string
     */
    public static boolean getBooleanAttributeValue(final Element element,
        final String name) {
        String value = element.getAttribute(name);

        return new Boolean(value).booleanValue();
    }

    /**
     * The name of the Node. First, checks the attribute value,
     * and if that is null, uses the
     * <code>getNodeName()</code> method on the Node interface.
     *
     * @param node    The Node whose name is to be found.
     * @return String    The name of the node (priority is given to the
     * attribute called 'name')
     */
    public static String getName(final Node node) {
        String name = null;

        if (node instanceof Element) {
            name = getAttribute(((Element) node), NAME);
        }

        if (name != null) {
            return name;
        }

        return node.getNodeName();
    }

    /**
     * A utility method to build the path of the Node
     * (with '.' are the separators).
     *
     * @param node    The Node whose path is to be found.
     * @return String    The String representation of the path.
     */
    public static String getPath(final Node node) {
        StringBuffer buf = new StringBuffer(500);

        if ((node.getParentNode() != null)
                && node.getParentNode() instanceof Element) {
            buf.append(getPath(node.getParentNode()));
            buf.append(".");
        }

        buf.append(getName(node));

        return buf.toString();
    }

    /**
     * Resolve the path to the file.
     * @param file File path to resolve.
     * @return String absolute path or URL.
     */
    public static String getURLFromClassPath(final String file) {
        URL url = Thread.currentThread().getContextClassLoader().getResource(file);

        if (url != null) {
            return url.toString();
        }

        return new File(file).getAbsolutePath();
    }

    /**
     * Checks the element for an attribute with the given name.
     * @param element element to be checked.
     * @param name Name of the attribute.
     * @return true if the attribute exists.
     */
    public static boolean hasAttribute(final Element element, final String name) {
        return element.hasAttribute(name);
    }

    /**
     * This method is used to parse the xml file with the specified name.
     *
     * @param stream      The input stream of the xml file to be parsed.
     * @return Document   The root Document object.
     */
    public static Document parse(final InputStream stream) {
        DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();

        //factory.setValidating(true);
        //factory.setNamespaceAware(true);
        try {
            return factory.newDocumentBuilder().parse(stream);
        } catch (SAXException sxe) {
            // Error generated during parsing
            Exception x = sxe;

            if (sxe.getException() != null) {
                x = sxe.getException();
            }

            x.printStackTrace();
            throw new RuntimeException(x.toString());
        } catch (ParserConfigurationException pce) {
            // Parser with specified options can't be built
            pce.printStackTrace();
            throw new XMLException("Parser configuration exception.");
        } catch (IOException ioe) {
            // I/O error
            ioe.printStackTrace();
            throw new XMLException("IO Exception while parsing XML doc.");
        } finally {
            if (stream != null) {
                try {
                    stream.close();
                } catch (IOException ex) {
                    throw new XMLException(
                        "IO Exception while closing stream: " + ex);
                }
            }
        }
    }

    /**
     * A utility method to find a file from the context of the specified class and read its
     * contents. The difference between this method and the {@link #readFileFromClasspath(String)}
     * method is that this one uses the classpath, and then appends the package information of the
     * specified class and then looks for the specified file.
     *
     * @param clz             Class whos resource loader should be used.
     * @param fileName        The name of the file to be read.
     * @return InputStream    The file contents as an InputStream.
     */
    public static InputStream readFileFromClassContext(final Class clz,
        final String fileName) {
        return clz.getResourceAsStream(fileName);
    }

    /**
     * A utility method to find a file from the classpath and read its contents.
     *
     * @param fileName     The name of the file to be read.
     * @return InputStream    The file contents as an InputStream.
     */
    public static InputStream readFileFromClasspath(final String fileName) {
        InputStream is = Thread.currentThread().getContextClassLoader()
                               .getResourceAsStream(fileName);

        if (is == null) {
            // Try local directory
            try {
                return new FileInputStream(fileName);
            } catch (FileNotFoundException ex) {
                throw new RuntimeException("Could not open input file:"
                    + fileName);
            }
        }

        return is;
    }

    /**
     * Write the XML Document with the given encoding to a file.
     * @param encoding Encoding to use when writing the file.
     * @param strm Stream to write the document.
     * @param doc XML Document to be written.
     * @throws UnsupportedEncodingException thrown if the implementation
     * of the encoding cannot be found.
     */
    public static void writeFile(final String encoding,
        final OutputStream strm, final Document doc)
        throws UnsupportedEncodingException {
        XMLWriter wr = new XMLWriter(false);
        wr.setOutput(strm, encoding);
        wr.write(doc);
    }
}
