package junit.extensions.xml.elements;

import junit.extensions.xml.IXMLTestCase;
import junit.extensions.xml.XMLException;

import junit.framework.Assert;

import org.w3c.dom.Element;


/**
 * This class will handle the processing of &lt;assertequal&gt; nodes.
 * Insures that the actual and expected objects are equal.
 * <H3>Summary</H3>
 * &lt;assertequal [message="message text"] actualrefid="id"|actualobj="value"
 *  expectedrefid="id"|expectedobj="value"/&gt;
 *
 * <H3>One of the following attributes are required:</H3>
 *   actualrefid  id of the object to be compared.<br>
 *   actualobj value of the object to be compared.
 *
 * <H3>One of the following attributes are required:</H3>
 *   expectedrefid id of the object to be compared.<br>
 *   expectedobj value of the object to be compared.
 *
 * <H3>Optional Attributes:</H3>
 *   message Optional message text to be displayed in assertion.
 *
 * @author <a href="mailto:vraravam@thoughtworks.com">Vijay Aravamudhan : ThoughtWorks Inc.</a>
 */
public class AssertEqualsTagHandler extends AbstractAssertTagHandler {
    /**
     * Constructor for AssertEqualsTagHandler.
     *
     * @param element     The element to be processed
     * @param testCase    The IXMLTestCase that uses this element
     */
    public AssertEqualsTagHandler(final Element element,
        final IXMLTestCase testCase) {
        super(element, testCase);
    }

    /**
     * Obtain the expected and actual objects. Then
     * test if one of the objects is a string. If so
     * the call toString() to normalize both objects
     * to a string. Assert that the values of the
     * actual and the expected are equal.
     *
     * @throws XMLException upon assertion failure.
     */
    public void processElement() throws XMLException {
        validateElement();

        Object expected = getExpectedObject();
        Object actual = getActualObject();

        if (expected instanceof String || actual instanceof String) {
            if (actual != null) {
                actual = actual.toString();
            }

            if (expected != null) {
                expected = expected.toString();
            }
        }

        Assert.assertEquals(
            getMessage(),
            expected,
            actual);
    }

    /**
     * Insure that the element contains a Actual and expected
     * attribute.
     * @throws XMLException Thrown if a required attribute is missing.
     */
    public void validateElement() throws XMLException {
        // do the default validations from the super class
        super.validateElement();

        // reqd attribute: at least one of expected or expectedrefid
        checkExpected();
        checkActual();
    }
}
