package junit.extensions.xml.elements;

import junit.extensions.xml.IXMLTestCase;
import junit.extensions.xml.XMLException;

import junit.framework.Assert;

import org.w3c.dom.Element;


/**
 * Return the tokens of a string one at a time.
 *
 * <H2>Title:</H2>
 * TokenizeTagHandler
 * <H2>Description:</H2>
 *
 * The TokenizeTagHandler provides a mechanism for breaking
 * up a string into pieces.  The refid points to the string
 * that will be broken up, each call to tokenize returns
 * the token, and the remainder of the token string that
 * can be used again in a subsequent call.
 *
 * <H2>Attributes:</H2>
 * <pre>
 * id        - the string token to be returned [required]
 * refid     - the string of tokens to separate [required]
 * delimiter - the token separator [required]
 * </pre>
 *
 *
 * <H2>Example:</H2>
 *    If array contained "File>New" and your delimiter was ">".
 *    As in:
 * <pre>
 * <code>
 *    &lt;property name="array" value="File>New"/&gt;
 *    &lt;tokenize id="token" refid="array" delimiter="&gt;"/&gt;
 *    &lt;!-- Now array="New" token="File" --&gt;
 *    &lt;tokenize id="token" refid="array" delimiter="&gt;"/&gt;
 *    &lt;!-- Now array="" token="New" --&gt;
 * </code>
 * </pre>
 *    The first call would return "File" in the token property
 *    and update the array to "New".
 *    The second call would return "New" and update the
 array to: ""
 *    Any subsequent call would return ""<br>
 *
 * Another example is to return the values of various fields of
 * which some may be null.
 * <code>
 *    &lt;property name="array" value="userid:*:Guest:::"/&gt;
 *    &lt;tokenize id="token" refid="array" delimiter=":"/&gt;
 *    &lt;!-- Now array="*:Guest:::" token="userid" --&gt;
 *    &lt;tokenize id="token" refid="array" delimiter=":"/&gt;
 *    &lt;!-- Now array="Guest:::" token="*" --&gt;
 *    &lt;tokenize id="token" refid="array" delimiter=":"/&gt;
 *    &lt;!-- Now array="::" token="Guest" --&gt;
 *    &lt;tokenize id="token" refid="array" delimiter=":"/&gt;
 *    &lt;!-- Now array=":" token="" --&gt;
 *    &lt;tokenize id="token" refid="array" delimiter=":"/&gt;
 *    &lt;!-- Now array="" token="" --&gt;
 * </code>
 * <p>Copyright: Copyright (c) 2004</p>
 * <p>Company: JFCUnit Sourceforge project</p>
 *
 */
public class TokenizeTagHandler extends AbstractTagHandler {
    /**
     * Constructor.
     * @param element Element to be created.
     * @param testcase IXMLTestCase parent test case.
     */
    public TokenizeTagHandler(final Element element, final IXMLTestCase testcase) {
        super(element, testcase);
    }

    /**
     * Handle the XML processing of the tag.
     * @throws XMLException may be thrown.
     */
    public void processElement() throws XMLException {
        validateElement();

        String tokenlist = (String) getXMLTestCase().getProperty(
                getString(REFID));

        if (getXMLTestCase().getDebug()) {
            System.out.println("tokenlist:" + tokenlist);
        }

        String delimiter = getString(DELIMITER);

        Assert.assertNotNull("Error: Unable to obtain delimiter : ", delimiter);

        int index = tokenlist.indexOf(delimiter);

        // Last token.
        if (index == -1) {
            // set the token value to the ID property
            ((IXMLTestCase) getTestCase()).addProperty(
                getString(ID),
                tokenlist);

            // clear the refid  (remove the property to make null)
            ((IXMLTestCase) getTestCase()).removeProperty(getString(REFID));
        } else {
            // More tokens so break off the first.
            // set the token value to the ID property
            String first  = tokenlist.substring(0, index);
            String others = tokenlist.substring(index + delimiter.length());
            ((IXMLTestCase) getTestCase()).addProperty(
                getString(ID),
                first);

            // update the refid with the remainder of the tokenizer
            ((IXMLTestCase) getTestCase()).addProperty(
                getString(REFID),
                others);
        }
    }

    /**
     * Make sure the appropriate tag and attributes are
     * used.
     * @throws XMLException upon failure to validate.
     */
    public void validateElement() throws XMLException {
        super.validateElement();

        // check the element tag name
        // checkElementTagName("selectMenuOnToolbar");
        checkRequiredAttribute(ID);

        checkRequiredAttribute(DELIMITER);

        checkRequiredAttribute(REFID);
    }
}
