package junit.extensions.jfcunit.eventdata;

import junit.extensions.jfcunit.JFCTestCase;
import junit.extensions.jfcunit.TestHelper;
import junit.extensions.jfcunit.keyboard.JFCKeyStroke;
import junit.extensions.jfcunit.keyboard.KeyMapping;

import org.w3c.dom.Element;

import java.awt.AWTEvent;
import java.awt.Component;
import java.awt.event.KeyEvent;


/**
 * Data container class that holds all the data necessary for jfcUnit to fire key events.
 *
 * @author <a href="mailto:vraravam@thoughtworks.com">Vijay Aravamudhan : ThoughtWorks Inc.</a>
 */
public class KeyEventData extends AbstractKeyEventData {
    /**
     * The component to be fired upon.
     */
    private Component m_comp = null;

    /**
     * The key code to be fired.
     */
    private int m_keyCode = 0;

    /**
     * Default Constructor.
     */
    public KeyEventData() {
        super();
    }

    /**
     * Constructor.
     *
     * @param testCase The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param comp     The component on which to trigger the event.
     * @param keyCode  The key to be sent to the component
     */
    public KeyEventData(final JFCTestCase testCase, final Component comp,
        final int keyCode) {
        this(testCase, comp, keyCode, DEFAULT_KEY_MODIFIERS, DEFAULT_SLEEPTIME);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param comp      The component on which to trigger the event.
     * @param keyCode           The key to be sent to the component
     * @param sleepTime The wait time in ms between each event.
     */
    public KeyEventData(final JFCTestCase testCase, final Component comp,
        final int keyCode, final long sleepTime) {
        this(testCase, comp, keyCode, DEFAULT_KEY_MODIFIERS, sleepTime);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param comp      The component on which to trigger the event.
     * @param keyCode          The key to be sent to the component
     * @param modifiers The modifier key values that need to be passed onto the event.
     * @param sleepTime
     *                   The wait time in ms between each event.
     */
    public KeyEventData(final JFCTestCase testCase, final Component comp,
        final int keyCode, final int modifiers, final long sleepTime) {
        setTestCase(testCase);
        setSource(comp);
        setModifiers(modifiers);
        setKeyCode(keyCode);
        setSleepTime(sleepTime);
        setValid(true);
    }

    /**
     * Set the key code to be sent.
     *
     * @param keyCode Key code.
     */
    public final void setKeyCode(final int keyCode) {
        m_keyCode = keyCode;
        setupKeyStrokes();
    }

    /**
     * Get the key code to be sent.
     *
     * @return int key code to be sent.
     */
    public final int getKeyCode() {
        return m_keyCode;
    }

    /**
     * Set the attribute value.
     *
     * @param comp   The new value of the attribute
     */
    public final void setSource(final Component comp) {
        m_comp = comp;
    }

    /**
     * Get the attribute value.
     *
     * @return Component    The value of the attribute.
     */
    public final Component getSource() {
        return m_comp;
    }

    /**
     * The component on which the event has to be fired.
     *
     * @return The component.
     */
    public Component getComponent() {
        // by default, the component is the same as the source
        return getSource();
    }

    /**
     * Check if this event can consume the AWTEvent.
     *
     * @param ae Event to be consumed.
     * @return boolean true if the event can be consumed.
     */
    public boolean canConsume(final AWTEvent ae) {
        if (!(ae instanceof KeyEvent)) {
            return false;
        }

        if (isValid()) {
            return false;
        }

        char c = ((KeyEvent) ae).getKeyChar();

        return super.canConsume(ae)
        && ((c == KeyEvent.CHAR_UNDEFINED) || Character.isISOControl(c));
    }

    /**
     * Consume the event.
     *
     * @param ae Event to be consumed.
     * @return boolean true if the event was consumed.
     */
    public boolean consume(final AWTEvent ae) {
        if (super.consume(ae)) {
            return true;
        }

        setSource((Component) ae.getSource());
        setModifiers(((KeyEvent) ae).getModifiers());

        char c = ((KeyEvent) ae).getKeyChar();

        if (Character.isISOControl(c)) {
            setKeyCode((int) c);
        } else {
            setKeyCode(((KeyEvent) ae).getKeyCode());
        }

        setSleepTime(getDefaultSleepTime());

        setValid(true);

        return true;
    }

    /**
     * Populate the element with the key code.
     * @param e Element to be populated with the data for this event.
     */
    public void populate(final Element e) {
        super.populate(e);
        e.setAttribute("code", "" + m_keyCode);
    }

    /**
     * Generate a description of the event.
     *
     * @return String description of the event.
     */
    public String toString() {
        return "KeyEventData:" + m_keyCode + " on " + getSource() + " "
        + getModifiers() + " " + getSleepTime();
    }

    /**
     * Setup the keystrokes for the event.
     */
    protected void setupKeyStrokes() {
        KeyMapping km = TestHelper.getKeyMapping();
        clearKeyStrokes();

        JFCKeyStroke[] strokes = km.getKeyStrokes(m_keyCode);
        addKeyStrokes(strokes);
        applyModifier(getModifiers());
    }
}
