package junit.extensions.jfcunit.finder;

import java.util.ArrayList;
import java.util.List;

import java.awt.Container;
import java.awt.Frame;
import java.awt.Window;
import javax.swing.JDialog;
import javax.swing.SwingUtilities;
import junit.extensions.jfcunit.WindowMonitor;

/**
 * Class for checking if the ({@link java.awt.Window}) component being searched for has been found.
 * This class can be used to check only windows that have titles (Frame/Dialog and their sub-classes)
 * The pattern syntax can be found at the Jakarta RegExp API Documentation in {@link org.apache.regexp.RE}.
 *
 * @author <a href="mailto:vraravam@thoughtworks.com">Vijay Aravamudhan : ThoughtWorks Inc.</a>
 */
public abstract class AbstractWindowFinder
    extends Finder {
    /**
     * The title of the component.
     */
    private String m_title;

    /**
     * A boolean specifying whether the filtration is case insensitive.
     */
    private boolean m_caseIndependent = false;

    /**
     * Constructor accepting all arguments needed to filter component.
     *
     * @param str    The desired pattern for the title of the component.
     */
    public AbstractWindowFinder(final String str) {
        this(str, false);
    }

    /**
     * Constructor accepting all arguments needed to filter component.
     *
     * @param title                The desired pattern for the title of the component.
     * @param caseIndependent    Whether the match should be case independent (true) or not (false)
     */
    public AbstractWindowFinder(final String title, final boolean caseIndependent) {
        m_title = title;
        m_caseIndependent = caseIndependent;
        recreatePatternMatcher(m_title, caseIndependent);
    }

    /**
     * Set the title for the finder.
     * @param title String title to be set.
     */
    public final void setTitle(final String title) {
        m_title = title;
        recreatePatternMatcher(m_title, m_caseIndependent);
    }

    /**
     * Set the finder into a case independent mode.
     * @param ignoreCase true if case should be ignored.
     */
    public void setCaseIndependent(final boolean ignoreCase) {
        super.setCaseIndependent(ignoreCase);
        m_caseIndependent = ignoreCase;
        recreatePatternMatcher(m_title, m_caseIndependent);
    }

    /**
     * Returns the caseIndependent.
     * @return boolean
     */
    public boolean isCaseIndependent() {
        return m_caseIndependent;
    }

    /**
     * Returns the title.
     * @return String
     */
    public final String getTitle() {
        return m_title;
    }

    /**
     * Returns a set of all the Windows that are currently visible and the title
     * contains the given titlematch string.
     *
     * @param pRet     The list of already filtered and accepted windows.
     * @param windows The array of windows to filter and add.
     * @param finder  The FrameFinder which is used to filter using a title match
     * @return Set of Window objects.
     */
    public static List getWindows(final List pRet, final Container[] windows,
                                   final Finder finder) {
        List ret = pRet;

        if (ret == null) {
            ret = new ArrayList();
        }

        if ((windows == null) || (finder == null)) {
            return ret;
        }

        Window window;

        for (int i = 0; i < windows.length; i++) {
            Container c = windows[i];
            if (c instanceof Window) {
                window = (Window) windows[i];
            } else {
                window = SwingUtilities.getWindowAncestor(c);
            }
            if (!finder.testComponent(window)) {
                getWindows(
                    ret,
                    window.getOwnedWindows(),
                    finder);
                continue;
            }

            String title = "";

            if (window instanceof Frame) {
                title = ((Frame) window).getTitle();
            } else if (window instanceof JDialog) {
                title = ((JDialog) window).getTitle();
            }

            // if this test is run through any of the TestRunner classes which
            // brings up a GUI window, we need to skip that frame
            if ("JUnit".equalsIgnoreCase(title)) {
                continue;
            }

            if (!ret.contains(window)) {
                ret.add(window);
            }

            // add any windows owned by the current 'frame' object
            getWindows(
                ret,
                window.getOwnedWindows(),
                finder);
        }

        return ret;
    }

    /**
     * Find the components matching this finder.
     * Windows searched will be owned by the containers given.
     *
     * @param owners Owner windows which should be searched.
     * @return Set contain the windows matching the finder.
     */
    public List findAll(final Container[] owners) {
        List retSet = new ArrayList();
        int wait = Math.max(
            0,
            getWait());
        long date = System.currentTimeMillis() + (wait * 1000);

        do {
            if (owners == null) {
                retSet = getWindows(
                    retSet,
                    WindowMonitor.getWindows(),
                    this);
            } else {
                retSet = getWindows(
                    retSet,
                    owners,
                    this);
            }

            if (retSet.size() == 0) {
                pause(date);
            }
        }
        while ((retSet.size() == 0) && (System.currentTimeMillis() < date));

        return retSet;
    }

    /**
     * This method checks whether the winTitle and the title strings are equal.
     * Note: If the title string is null, we need to return all showing windows,
     * so 'true' is returned.
     *
     * @param winTitle The title of the window being checked
     * @return True if either the filter string is null or if the
     *         filter string is a substring of the window title
     */
    protected final boolean checkIfTitleMatches(final String winTitle) {
        boolean value = ((m_title == null)
                         || ((winTitle != null) && evaluate(winTitle, m_title)));

        if (getDebug()) {
            System.err.println("Comparing window title(" + value + "): "
                               + winTitle + " match " + m_title);
        }

        return value;
    }
}
