package junit.extensions.jfcunit.tools;

import org.apache.regexp.RE;
import org.apache.regexp.RESyntaxException;


/**
 * Abstract class for defining call back classes to test whether a {@link java.awt.Component}
 * that is being searched for has been found.
 *
 * @author kevin Wilson
 */
public final class Operator {
    /**
     * This is the lock for synchronizing checks of the cache.
     */
    private static Object s_lock = new Object();

    /**
     * Pattern which was cached.
     */
    private static String s_cachePattern = null;

    /**
     * Cached regular expression.
     */
    private static RE s_cacheRE = null;

    /**
     * Constructor should be private.
     */
    private Operator() {
    }

    /**
     * Evaluate the current operation.
     * @param leftside String to be compared.
     * @param rightside String or Regular expression.
     * @param operation See the Operation subclass for values.
     * @param ignoreCase Ignore the case of the values.
     * @return result of operation
     */
    public static boolean evaluate(final String leftside,
        final String rightside, final int operation, final boolean ignoreCase) {
        String left  = leftside;
        String right = rightside;

        if (left == null) {
            // If left and right are null then return true.
            // otherwise return false.
            return right == null;
        }

        if (ignoreCase) {
            if (left != null) {
                left = left.toUpperCase();
            }

            if (right != null) {
                right = right.toUpperCase();
            }
        }

        if (operation == Operation.CONTAINS) {
            return (left.indexOf(right) >= 0);
        } else if (operation == Operation.ENDSWITH) {
            return (left.endsWith(right));
        } else if (operation == Operation.STARTSWITH) {
            return (left.startsWith(right));
        } else if (operation == Operation.EQUALS) {
            return (left.equals(right));
        } else if (operation == Operation.MATCH) {
            return matchPattern(left, right);
        }

        throw new java.lang.IllegalStateException(
            "Invalid operation for finder:" + operation);
    }

    /**
     * Get the regular expression implementing the pattern.
     * @param pattern String representation of RE.
     * @return RE Class representation of RE.
     */
    private static RE getPattern(final String pattern) {
        synchronized (s_lock) {
            if ((pattern != null) && pattern.equals(s_cachePattern)) {
                return s_cacheRE;
            }

            try {
                s_cachePattern = pattern;

                if (pattern == null) {
                    s_cacheRE = new RE("");
                } else {
                    s_cacheRE = new RE(pattern);
                }
            } catch (RESyntaxException ex) {
                s_cacheRE = null;
            }
        }

        return s_cacheRE;
    }

    /**
     * This method is used to filter components' attributes based on a pattern specified by the user.
     * For example, to search for all windows with title matching 'testWindow*'.
     * Note: If the patternString is null, we need to avoid the <code>PatternCompiler.compile()</code>
     * throwing a NullPointerException and in this case, return true if the componentAttribute
     * is also null.
     * The pattern syntax can be found at the Jakarta RegExp API Documentation in {@link org.apache.regexp.RE}.
     *
     * @param componentAttribute    The attribute text of the component to match against
     * @param patternString         The pattern to match with
     * @return boolean whether the pattern is contained within the components' attribute text
     */
    private static boolean matchPattern(final String componentAttribute,
        final String patternString) {
        RE      re    = getPattern(patternString);
        boolean value = re.match(componentAttribute);

        return ((componentAttribute != null) && (re != null) && value);
    }

    /**
     * Operations for comparing strings.
     * <p>Title: class Operator.Operations</p>
     * <p>Description: Operation are specified in the
     * Operator.evaluate() method.</p>
     * <p>Copyright: Copyright (c) 2004</p>
     * <p>Company: JFCUnit OpenSource project.</p>
     * @author Kevin Wilson
     * @version 1.0
     */
    public static final class Operation {
        /**
         * String so for the OP codes.
         */
        private static final String[] CODE_STRINGS = new String[] {
                "match", "startswith", "endswith", "equals", "contains"
            };

        /**
         * Match the item using a Regular expression.
         */
        public static final int MATCH = 0;

        /**
         * Check that the value starts with the given
         * value.
         */
        public static final int STARTSWITH = 1;

        /**
         * Check that the value ends with the given
         * value.
         */
        public static final int ENDSWITH = 2;

        /**
         * Check that the values are equal.
         */
        public static final int EQUALS = 3;

        /**
         * Check that the value is contained.
         */
        public static final int CONTAINS = 4;

        /**
         * Private constructor. This class should not be instanciated.
         */
        private Operation() {
        }

        /**
         * Convert the String to a Operation code.
         * @param code String version of the Operation.
         * @return int version of the operation.
         */
        public static int getOperation(final String code) {
            for (int i = 0; i < CODE_STRINGS.length; i++) {
                if (CODE_STRINGS[i].equalsIgnoreCase(code)) {
                    return i;
                }
            }

            throw new java.lang.IllegalArgumentException("Operation not found:"
                + code);
        }

        /**
         * Get the string version of the operation.
         * @param code int Finder.OP_code
         * @return String version of the operation.
         */
        public static String toString(final int code) {
            if ((code < 0) || (code >= CODE_STRINGS.length)) {
                throw new IllegalArgumentException("Invalid Operation code:"
                    + code);
            }

            return CODE_STRINGS[code];
        }
    }
}
