package junit.extensions.xml;

import junit.extensions.xml.elements.AbstractTagHandler;

import org.w3c.dom.Element;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;

import java.util.HashMap;
import java.util.Map;
import java.util.ResourceBundle;


/**
 * This class is used to read in the properties and provide access to the tag handlers for various
 * elements based on their attribute values.
 *
 * @author <a href="mailto:vraravam@thoughtworks.com">Vijay Aravamudhan : ThoughtWorks Inc.</a>
 * @author Kevin Wilson
 */
public final class XMLTagResourceBundle {
    /**
     * Handle to the singleton instance.
     */
    private static XMLTagResourceBundle s_singleton;

    /**
     * Map of the API specified Tag Handlers.
     */
    private final Map m_map = new HashMap();

    /**
     * The ResourceBundle used to read in the properties.
     */
    private ResourceBundle m_bundle = null;

    /**
     * A default private constructor.
     * @see java.lang.Object#Object()
     */
    private XMLTagResourceBundle() {
        m_bundle = ResourceBundle.getBundle("TagMapping");
    }

    /**
         * A convenience method to get a handle to the AbstractTagHandler implementation.
     * @param element     The element for which we need the tag handler.
     * @param test   The IXMLTestSuite being run.
     * @param type        The type of element to be handled.
     * @return AbstractTagHandler    An implementation of the AbstractTagHandler which will be used
     * to process the element.
     * @throws XMLException if the tag handle cannot be instanciated.
     */
    public static AbstractTagHandler getTagHandler(final Element element,
        final IXMLTest test, final String type) throws XMLException {
        return getSingleton().getTagHandlerImpl(element, test, type);
    }

    /**
     * Add a tag handler. If the tag exists in the
     * resource bundle, then the added tag handler will
     * take precedence over the tag from the resource
     * bundle.
     * @param tagname Tag name to be registered.
     * @param classname Classname to be registered.
     */
    public static void addTagHandler(final String tagname,
        final String classname) {
        getSingleton().addTagHandlerImpl(tagname, classname);
    }

    /**
     * Remove a tag handler. If the tag name was over
     * ridden by a added tag, then access will be restored
     * to the tag provided by the resource bundle. Tags
     * in the resource bundle may not be removed.
     *
     * @param tagname Tagname to be removed.
     */
    public static void removeTagHandler(final String tagname) {
        getSingleton().removeTagHandlerImpl(tagname);
    }

    /**
     * The "default" method that is used to create and initialize a singleton instance.
     * @return XMLTagResourceBundle    The one and only instance of this class.
     */
    private static XMLTagResourceBundle getSingleton() {
        if (s_singleton == null) {
            s_singleton = new XMLTagResourceBundle();
        }

        return s_singleton;
    }

    /**
     * Get the Class for the type of tag handler given.
     * The local tag map will be used first. If the tag is
     * not found in the map, the bundle will be searched.
     * @param tagname TagName to be
     * @param debug True if debugging.
     * @return Class instance for the tagtype
     * @throws XMLException may be thrown.
     */
    private Class getClassFromTag(final String tagname, final boolean debug)
        throws XMLException {
        String clazzName = (String) m_map.get(tagname);

        if (clazzName == null) {
            clazzName = m_bundle.getString(tagname);

            if (clazzName != null) {
                if (debug) {
                    System.err.println("Found tag(" + tagname + "/" + clazzName
                        + ") in properties");
                }
            }
        } else {
            if (debug) {
                System.err.println("Found tag(" + tagname + "/" + clazzName
                    + ") in map");
            }
        }

        try {
            Class clazz = Class.forName(clazzName);

            return clazz;
        } catch (Exception e) {
            // Ignore
        }

        throw new XMLException("Could not load class:" + clazzName, null, null,
            null);
    }

    /**
     * Reads the short name (type) and creates an instance of the tag handler that corresponds to
     * the value from the resource bundle.
     * @param element     The element for which we need the tag handler.
     * @param test        The testCase being run.
     * @param type        The type of element to be handled.
     * @return AbstractTagHandler    An implementation of the AbstractTagHandler which will be used
     * to process the element.
     * @throws XMLException if the tag handler cannot be instanciated.
     */
    private AbstractTagHandler getTagHandlerImpl(final Element element,
        final IXMLTest test, final String type) throws XMLException {
        Class clazz = getClassFromTag(
                type,
                test.getDebug());
        Object[] params = new Object[] {element, test};

        try {
            Constructor cnstr;

            if (test instanceof IXMLTestCase) {
                cnstr = clazz.getConstructor(
                        new Class[] {Element.class, IXMLTestCase.class});
            } else if (test instanceof IXMLTestSuite) {
                cnstr = clazz.getConstructor(
                        new Class[] {Element.class, IXMLTestSuite.class});
            } else {
                throw new XMLException(
                    "Test is not instanceof IXMLTestSuite or IXMLTestCase");
            }

            return (AbstractTagHandler) cnstr.newInstance(params);
        } catch (InvocationTargetException ite) {
            Throwable t = ite.getCause();

            if (t instanceof XMLException) {
                throw (XMLException) t;
            }

            throw new XMLException(
                t.getMessage(),
                t,
                element,
                null);
        } catch (XMLException xmle) {
            throw xmle;
        } catch (Exception e) {
            throw new XMLException("Could not create tag handler for:" + type
                + "\n", e, element, null);
        } catch (Error er) {
            throw new XMLException("Could not create tag handler for:" + type
                + "\n", er, element, null);
        }
    }

    /**
     * Add a tag handler. If the tag exists in the
     * resource bundle, then the added tag handler will
     * take precedence over the tag from the resource
     * bundle.
     * @param tagname Tag name to be registered.
     * @param classname Classname to be registered.
     */
    private void addTagHandlerImpl(final String tagname, final String classname) {
        m_map.put(tagname, classname);
    }

    /**
     * Remove a tag handler. If the tag name was over
     * ridden by a added tag, then access will be restored
     * to the tag provided by the resource bundle. Tags
     * in the resource bundle may not be removed.
     *
     * @param tagname Tagname to be removed.
     */
    private void removeTagHandlerImpl(final String tagname) {
        m_map.remove(tagname);
    }
}
