package junit.extensions.xml.elements;

import junit.extensions.xml.IXMLTest;
import junit.extensions.xml.IXMLTestCase;
import junit.extensions.xml.IXMLTestSuite;
import junit.extensions.xml.XMLConstants;
import junit.extensions.xml.XMLException;
import junit.extensions.xml.XMLUtil;

import junit.framework.TestCase;

import org.w3c.dom.Element;

import java.awt.Point;

import java.text.MessageFormat;
import java.text.ParseException;


/**
 * Sub-classes of this class will handle the processing of elements.
 * It provides common access methods to the test case or test suite.
 * It also provides methods for reading attributes.
 *
 * @author <a href="mailto:vraravam@thoughtworks.com">Vijay Aravamudhan : ThoughtWorks Inc.</a>
 * @author Kevin Wilson
 */
public abstract class AbstractTagHandler implements XMLConstants {
    /**
     * The Element to be processed.
     */
    private Element m_element;

    /**
     * The IXMLTestCase that uses this element.
     */
    private IXMLTestCase m_testCase;

    /**
     * The IXMLTestSuite that uses this element.
     */
    private IXMLTestSuite m_testSuite;

    /**
     * Constructor for AbstractTagHandler.
     *
     * @param element    The element to be processed.
     * @param testCase   The testCase to be associated with the test.
     */
    public AbstractTagHandler(final Element element, final IXMLTestCase testCase) {
        if (element == null) {
            throw new java.lang.IllegalArgumentException(
                "element must not be null");
        }

        if (testCase == null) {
            throw new java.lang.IllegalArgumentException(
                "testCase must not be null");
        }

        this.m_element      = element;
        this.m_testCase     = testCase;
    }

    /**
     * Constructor for AbstractTagHandler.
     *
     * @param element    The element to be processed.
     * @param testSuite   The testSuite to be associated with the test.
     */
    public AbstractTagHandler(final Element element,
        final IXMLTestSuite testSuite) {
        if (element == null) {
            throw new java.lang.IllegalArgumentException(
                "Element must not be null");
        }

        if (testSuite == null) {
            throw new java.lang.IllegalArgumentException(
                "testSuite must not be null");
        }

        this.m_element       = element;
        this.m_testSuite     = testSuite;
    }

    /**
     * Returns the element.
     *
     * @return Element.
     */
    public final Element getElement() {
        return m_element;
    }

    /**
     * The JUnit framework test case.
     * @return TestCase in JUnit framework.
     */
    public final TestCase getTestCase() {
        return (TestCase) m_testCase;
    }

    /**
     * Returns the testCase.
     *
     * @return IXMLTestCase.
     */
    public final IXMLTestCase getXMLTestCase() {
        return m_testCase;
    }

    /**
     * Returns the testSuite.
     *
     * @return IXMLTestSuite.
     */
    public final IXMLTestSuite getXMLTestSuite() {
        return m_testSuite;
    }

    /**
     * This method is used to process the xml Element (i.e. parsing and setting the attributes on
     * XYZ objects from those parsed values.
     *
     * @throws XMLException Some kind of Exception might be thrown.
     */
    public abstract void processElement() throws XMLException;

    /**
     * Return the tag name of this element.
     * @return Tag name.
     */
    protected final String getTagName() {
        return m_element.getTagName();
    }

    /**
     * Since all sub-classes might check for a required attribute,
     * this method can be used for consistent messages.
     *
     * @param attrName    The names of the attributes being checked.
     * @throws XMLException    A validation exception is thrown.
     */
    protected final void checkAtLeastOneRequiredAttribute(
        final String[] attrName) throws XMLException {
        checkAtLeastOneRequiredAttribute(
            getElement(),
            attrName);
    }

    /**
     * Since all sub-classes might check for a required attribute,
     * this method can be used for consistent messages.
     *
     * @param root    The root Element whose attributes are being checked
     * @param attrName    The names of the attributes being checked.
     * @throws XMLException    A validation exception is thrown.
     */
    protected final void checkAtLeastOneRequiredAttribute(final Element root,
        final String[] attrName) throws XMLException {
        int found = 0;

        for (int i = 0; i < attrName.length; i++) {
            if (XMLUtil.hasAttribute(root, attrName[i])) {
                found++;
            }
        }

        if (found == 0) {
            String buf = "";

            for (int i = 0; i < attrName.length; i++) {
                buf = MessageFormat.format(
                        "{0} {1}",
                        new Object[] {buf, attrName[i]});
            }

            throw new XMLException(root.getNodeName()
                + " is missing one of the required attributes: " + buf, null,
                root,
                getTest().getPropertyCache());
        }
    }

    /**
     * Since all sub-classes will check for the element tag name,
     * this method can be used for consistent messages.
     *
     * @param expectedName    The non-null expected string name.
     * @throws XMLException    A validation exception is thrown.
     */
    protected final void checkElementTagName(final String expectedName)
        throws XMLException {
        if (!expectedName.equals(m_element.getTagName())) {
            throw new XMLException("Cannot process "
                + getElement().getTagName() + " node, expected '"
                + expectedName + "'", null,
                getElement(),
                getTest().getPropertyCache());
        }
    }

    /**
     * Since all sub-classes might check for a required attribute,
     * this method can be used for consistent messages.
     * @param e Element to be checked for the attribute.
     * @param attrName    The name of the attribute being checked.
     * @throws XMLException    A validation exception is thrown.
     */
    protected final void checkRequiredAttribute(final Element e,
        final String attrName) throws XMLException {
        if (!XMLUtil.hasAttribute(e, attrName)) {
            throw new XMLException("Missing the required attribute: "
                + attrName + " in element: " + m_element.getNodeName(), null,
                e,
                getTest().getPropertyCache());
        }
    }

    /**
     * Since all sub-classes might check for a required attribute,
     * this method can be used for consistent messages.
     *
     * @param attrName    The name of the attribute being checked.
     * @throws XMLException    A validation exception is thrown.
     */
    protected final void checkRequiredAttribute(final String attrName)
        throws XMLException {
        checkRequiredAttribute(
            getElement(),
            attrName);
    }

    /**
     * This method checks whether the specified attribute is present, and if so, returns the value
     * parsed as a primitive boolean, if not, then returns false.
     * @param attributeName    The name of the attribute whose value is needed.
     * @return boolean    The value of the attribute (false if not found).
     * @depricated Specify the default value.
     */
    protected boolean getBoolean(final String attributeName) {
        return getBoolean(
            getElement(),
            attributeName,
            false);
    }

    /**
     * This method checks whether the specified attribute is present, and if so, returns the value
     * parsed as a primitive boolean, if not, then returns false.
     * @param attributeName    The name of the attribute whose value is needed.
     * @param defaultValue     The default value
     * @return boolean    The value of the attribute (defaultValue if not found).
     */
    protected boolean getBoolean(final String attributeName,
        final boolean defaultValue) {
        return getBoolean(
            getElement(),
            attributeName,
            defaultValue);
    }

    /**
     * This method checks whether the specified attribute is present, and if so, returns the value
     * parsed as a primitive boolean, if not, then returns false.
     * @param e Element to be searched.
     * @param attributeName    The name of the attribute whose value is needed.
     * @return boolean    The value of the attribute (false) if not found).
     * @depricated Specify the default value.
     */
    protected final boolean getBoolean(final Element e,
        final String attributeName) {
        return getBoolean(e, attributeName, false);
    }

    /**
     * This method checks whether the specified attribute is present, and if so, returns the value
     * parsed as a primitive boolean, if not, then returns false.
     * @param e Element to be searched.
     * @param attributeName    The name of the attribute whose value is needed.
     * @param defaultValue     The default value.
     * @return boolean    The value of the attribute (defaultValue if not found).
     */
    protected final boolean getBoolean(final Element e,
        final String attributeName, final boolean defaultValue) {
        // this will default to false
        if (XMLUtil.hasAttribute(e, attributeName)) {
            String s = getString(e, attributeName);

            if (defaultValue) {
                return !"false".equalsIgnoreCase(s);
            } else {
                return "true".equalsIgnoreCase(s);
            }
        } else {
            return defaultValue;
        }
    }

    /**
     * This method checks whether the specified attribute is present, and if so, returns the value
         * parsed as a primitive int, if not, then returns the default value specified.
     * @param attributeName    The name of the attribute whose value is needed.
     * @param defaultValue     The default value to use if such an attribute is not present.
     * @return int    The value of the attribute.
     */
    protected int getInt(final String attributeName, final int defaultValue) {
        return getInt(
            getElement(),
            attributeName,
            defaultValue);
    }

    /**
     * This method checks whether the specified attribute is present, and if so, returns the value
     * parsed as a primitive int, if not, then returns the default value specified.
     * @param e Element to be processed.
     * @param attributeName    The name of the attribute whose value is needed.
     * @param defaultValue     The default value to use if such an attribute is not present.
     * @return int    The value of the attribute.
     */
    protected int getInt(final Element e, final String attributeName,
        final int defaultValue) {
        String s = getString(e, attributeName);

        try {
            if (s != null) {
                return Integer.parseInt(s);
            }
        } catch (NumberFormatException nfe) {
            // Ignore the number format exception.
        }

        return defaultValue;
    }

    /**
     * This method checks whether the specified attribute is present, and if so,
     * returns the value parsed as a primitive long, if not, then returns
     * the default value specified.
     *
     * @param attributeName    The name of the attribute whose value is needed.
     * @param defaultValue     The default value to use if such an attribute is not present.
     * @return long    The value of the attribute.
     */
    protected long getLong(final String attributeName, final long defaultValue) {
        return getLong(
            getElement(),
            attributeName,
            defaultValue);
    }

    /**
     * This method checks whether the specified attribute is present, and if so, returns the value
     * parsed as a primitive long, if not, then returns the default value specified.
     * @param e Element to be searched.
     * @param attributeName    The name of the attribute whose value is needed.
     * @param defaultValue     The default value to use if such an attribute is not present.
     * @return long    The value of the attribute.
     */
    protected long getLong(final Element e, final String attributeName,
        final long defaultValue) {
        String s = getString(e, attributeName);

        try {
            if (s != null) {
                return Long.parseLong(s);
            }
        } catch (NumberFormatException nfe) {
            ;
        }

        return defaultValue;
    }

    /**
     * This method checks whether the specified attribute is present, and if so, returns the value
     * parsed as a {@link Point}, if not, then returns the default value specified.
     * @param attributeName    The name of the attribute whose value is needed.
     * @param defaultValue     The default value to use if such an attribute is not present.
     * @return Point    The value of the attribute.
     * @throws XMLException if the parse did not work.
     */
    protected final Point getPoint(final String attributeName,
        final Point defaultValue) throws XMLException {
        return getPoint(
            getElement(),
            attributeName,
            defaultValue);
    }

    /**
     * This method checks whether the specified attribute is present, and if so, returns the value
     * parsed as a {@link Point}, if not, then returns the default value specified.
     * @param e  Element to be searched.
     * @param attributeName    The name of the attribute whose value is needed.
     * @param defaultValue     The default value to use if such an attribute is not present.
     * @return Point    The value of the attribute.
     * @throws XMLException is thrown if the element cannot be understood.
     */
    protected final Point getPoint(final Element e, final String attributeName,
        final Point defaultValue) throws XMLException {
        String s = getString(e, attributeName);

        if (s == null) {
            return defaultValue;
        }

        // somehow, using this does not work!!!
        //MessageFormat mf = new MessageFormat("{0,number},{1,number}");
        MessageFormat mf = new MessageFormat("{0},{1}");
        int           x = 0;
        int           y = 0;

        try {
            Object[] obs = mf.parse(s);
            x     = Integer.parseInt((String) obs[0]);
            y     = Integer.parseInt((String) obs[1]);
        } catch (NumberFormatException ex) {
            throw new XMLException("Invalid number format:" + s, ex, e,
                getTest().getPropertyCache());
        } catch (ParseException ex) {
            throw new XMLException("Parsing of Point failed." + s, ex, e,
                getTest().getPropertyCache());
        }

        return new Point(x, y);
    }

    /**
     * This method returns the value of the specified attribute.
     * @param attributeName    The name of the attribute whose value is needed.
     * @return String    The value of the attribute.
     */
    protected final String getString(final String attributeName) {
        return getString(
            getElement(),
            attributeName);
    }

    /**
     * Get a the string assigned to a attribute from the current element.
     * @param attributeName Name of the attribute to find.
     * @param defaultValue Default value to return if the attribute is not
     * present.
     * @return String containing the attribute. Variable names will have been
     * resolved.
     */
    protected final String getString(final String attributeName,
        final String defaultValue) {
        return getString(
            getElement(),
            attributeName,
            defaultValue);
    }

    /**
     * This method returns the value of the specified attribute.
     * @param e          Element to be searched;
     * @param attributeName    The name of the attribute whose value is needed.
     * @return String    The value of the attribute.
     */
    protected final String getString(final Element e, final String attributeName) {
        String s = XMLUtil.getAttribute(e, attributeName);

        return resolveVariables(s);
    }

    /**
     * Get the string assigned to the attributeName from the element.
     * If the attributeName does not exist then return the default value.
     * @param e Element to be checked.
     * @param attributeName Name of the attribute to be returned.
     * @param defaultValue  Default value to be returned.
     * @return String which has had variable names resolved.
     */
    protected final String getString(final Element e,
        final String attributeName, final String defaultValue) {
        if (e.hasAttribute(attributeName)) {
            return getString(e, attributeName);
        } else {
            return resolveVariables(defaultValue);
        }
    }

    /**
     * Since all sub-classes might check for a required attribute,
     * this method can be used for consistent messages.
     *
     * @param attrName    The names of the attributes being checked.
     * @throws XMLException    A validation exception is thrown.
     */
    protected void checkOneRequiredAttribute(final String[] attrName)
        throws XMLException {
        checkOneRequiredAttribute(
            getElement(),
            attrName);
    }

    /**
     * Since all sub-classes might check for a required attribute,
     * this method can be used for consistent messages.
     *
     * @param root    The root Element whose attributes are being checked
     * @param attrName    The names of the attributes being checked.
     * @throws XMLException    A validation exception is thrown.
     */
    protected void checkOneRequiredAttribute(final Element root,
        final String[] attrName) throws XMLException {
        int found = 0;

        if (attrName != null) {
            for (int i = 0; i < attrName.length; i++) {
                if (XMLUtil.hasAttribute(root, attrName[i])) {
                    found++;
                }
            }
        }

        if (found == 0) {
            String buf = "";

            for (int i = 0; i < attrName.length; i++) {
                buf = MessageFormat.format(
                        "{0} {1}",
                        new Object[] {buf, attrName[i]});
            }

            throw new XMLException(root.getNodeName()
                + " is missing one of the required attributes: " + buf, null,
                root,
                getTest().getPropertyCache());
        }

        if (found > 1) {
            String buf = "";

            for (int i = 0; i < attrName.length; i++) {
                buf = MessageFormat.format(
                        "{0} {1}",
                        new Object[] {buf, attrName[i]});
            }

            throw new XMLException("Too many attributes of:" + buf, null, root,
                getTest().getPropertyCache());
        }
    }

    /**
     * Replace variable names with the string of
     * the variable values.
     * @param s String to be substituted.
     * @return String with substitutions made.
     */
    protected final String resolveVariables(final String s) {
        if (getXMLTestCase() != null) {
            return getXMLTestCase().resolveProperties(s);
        } else if (getXMLTestSuite() != null) {
            return getXMLTestSuite().resolveProperties(s);
        }

        return s;
    }

    /**
     * Get the test.
     * @return IXMLTest IXMLTest test case or test suite.
     */
    protected IXMLTest getTest() {
        if (this.m_testCase != null) {
            return m_testCase;
        } else {
            return m_testSuite;
        }
    }

    /**
     * Sub-classes should implement this method to provide validation
     * of the element attributes, etc.
     *
     * @throws XMLException    A validation exception is thrown.
     */
    protected void validateElement() throws XMLException {
        if (m_element == null) {
            throw new XMLException("Element could not be found.", null,
                getElement(),
                getTest().getPropertyCache());
        }
    }
}
