package junit.extensions.xml.elements;

import junit.extensions.xml.IXMLTestSuite;
import junit.extensions.xml.XMLException;
import junit.extensions.xml.XMLTagResourceBundle;
import junit.extensions.xml.XMLUtil;

import org.w3c.dom.Element;

import java.io.File;


/**
 * This class includes the given file into a test suite.
 * <H3>Tag Name</H3>
 * file
 *
 * <H3>Required Attributes</H3>
 * name - file name of the given file.
 * relative - value true if the file should be located relative to
 * the current file. Default value is false for backward compatibility.
 * <H3>Example</H3>
 * <pre>
 * &lt;suite name="MyTest"&gt;
 *     &lt;file name="moresuites.xml"/&gt;
 *     &lt;file name="extras/moresuites.xml" relative="true"/&gt;
 * &lt;/suite&gt;
 * </pre>
 *
 * @author <a href="mailto:vraravam@thoughtworks.com">Vijay Aravamudhan : ThoughtWorks Inc.</a>
 */
public class FileTagHandler extends AbstractTagHandler {
    /**
     * Default constructor.
     *
     * @param element      The element to be processed.
     * @param testSuite    The XMLTestSuite that uses this element.
     */
    public FileTagHandler(final Element element, final IXMLTestSuite testSuite) {
        super(element, testSuite);
    }

    /**
     * Process the file element.
     * @throws XMLException is thrown if the element cannot be understood.
     */
    public void processElement() throws XMLException {
        validateElement();

        String name = getString(NAME);

        if (getBoolean(RELATIVE)) {
            Element curdoc   = getElement().getOwnerDocument()
                                   .getDocumentElement();
            String  location = curdoc.getAttribute(JFCFILELOC);
            int     index    = location.lastIndexOf(File.separator);
            name = location.substring(0, index + 1) + name;
        }

        try {
            Element doc = XMLUtil.parse(XMLUtil.readFileFromClasspath(name))
                                 .getDocumentElement();
            doc.setAttribute(JFCFILELOC, name);

            String tag = doc.getTagName();
            XMLTagResourceBundle.getTagHandler(
                doc,
                getTest(),
                tag).processElement();
        } catch (OutOfMemoryError err) {
            throw new XMLException("Out of memory check for recursive file includes.",
                err,
                getElement(),
                getXMLTestSuite().getPropertyCache());
        } catch (XMLException xe) {
            throw xe;
        } catch (Exception e) {
            throw new XMLException(
                e.getMessage(),
                e,
                getElement(),
                getTest().getPropertyCache());

            //      if (e.getCause() instanceof StackOverflowError) {
            //        throw new XMLException(
            //            "StackOverflowError check for recursive file includes.", e,
            //            getElement(), getXMLTestSuite().getPropertyCache());
            //      }
        }
    }

    /**
     * Validate that the name element has been added.
     * @throws XMLException if the name attribute is not specified.
     */
    public void validateElement() throws XMLException {
        // do the default validations from the super class
        super.validateElement();

        checkRequiredAttribute(NAME);
    }
}
