package junit.extensions.xml.elements;

import junit.extensions.xml.IXMLTestCase;
import junit.extensions.xml.XMLException;

import org.w3c.dom.Element;

import javax.swing.JComboBox;
import javax.swing.JList;
import javax.swing.JTabbedPane;
import javax.swing.JTable;
import javax.swing.ListModel;
import javax.swing.table.TableColumn;


/**
 * <p>Title: class ForeachTagHandler</p>
 * <p>Description: This tag handler implements a looping
 * construct to visit each cell or index in a table or list.
 * <H3>Summary</H3>
 * &lt;for-each id="propname" refid="componentname"
 *    type="tablecell|listitem"
 *    [row="rownumber"] [column="columnumber"]/&gt;
 *
 * <H3>Required</H3>
 * id - property name for the output values <p>
 * refid - component to be iterated over.<p>
 * type  - tablecell or listitem depending on the component type. If tablecell
 * then the refid object is assumed to be a JTable instance. If listitem then
 * the refid object is assumed to be a JList object or JComboBox.
 * <H3>Optional</H3>
 * row - row will only be used when type = tablecell.
 *       when row is set the row specified will be scanned.
 * column - column will only be used when type = tabecell.
 *       When column is set only the column specified will be
 *       iterated.
 * <H3>Children</H3>
 * The child nodes of this element will be processed.<p>
 * <H3>Outputs</H3>
 * The following will be the outputs for type = listitem. The
 * <I>id</I> will be replaced by the value of the id attribute:<p>
 * <I>id</I> is the value of model item being iterated.<p>
 * <I>id</I>.index where id will be replaced by the value of the id attribute. The
 * value of the index wil be the index of the item being iterated.<p><p>
 *
 * The following will be the outputs for type = tablecell. The
 * <I>id</I> will be replaced by the value of the id attribute:<p>
 * <I>id</I> is the value of model item being iterated.<p>
 * <I>id</I>.row row number of the current cell<p>
 * <I>id</I>.column column number of the current cell<p>
 *
 * <H3>Examples</H3>
 * Iterate over all cells of a table.<p>
 * <pre>
 * &lt;for-each type="tablecell" refid="Table" id="cell" /&gt;
 *     &lt;echo message="tablecell(${cell.row}, ${cell.column}) = ${cell}" mode="stdout"/&gt;
 * &lt;/for-each&gt;
 * </pre>
 * Iterate over a column of a table.<p>
 * <pre>
 *
 * &lt;for-each type="tablecell" refid="Table" id="cell" column="1"/&gt;
 *     &lt;echo message="tablecell(${cell.row}, ${cell.column}) = ${cell}" mode="stdout"/&gt;
 * &lt;/for-each&gt;
 * </pre>
 * Iterate over a row of a table.<p>
 * <pre>
 * &lt;for-each type="tablecell" refid="Table" id="cell" row="1"/&gt;
 *     &lt;echo message="tablecell(${cell.row}, ${cell.column}) = ${cell}" mode="stdout"/&gt;
 * &lt;/for-each&gt;
 * </pre>
 * Iterate over a list.<p>
 * <pre>
 * &lt;for-each type="listitem" refid="List" id="item"/&gt;
 *     &lt;echo message="listitem(${item.index}) = ${item}"
 *     mode="stdout"/&gt;
 * &lt;/for-each&gt;
 * </pre>
 * </p>
 * <p>Copyright: Copyright (c) 2003</p>
 * <p>Company: </p>
 * @author not attributable
 * @version 1.0
 */
public class ForeachTagHandler extends AbstractTagHandler {
    /**
     * Constructor.
     * @param element Element to be processed.
     * @param testCase Test case processing tag handler.
     */
    public ForeachTagHandler(final Element element, final IXMLTestCase testCase) {
        super(element, testCase);
    }

    /**
     * process the element.
     * @throws XMLException is thrown if the element cannot be understood.
     */
    public void processElement() throws XMLException {
        String type = getString(TYPE);

        String id    = getString(ID);
        String refid = getString(REFID);

        Object obj = getXMLTestCase().getProperty(refid);
        getTestCase().assertNotNull("for-each refid is null", obj);

        boolean debug = getXMLTestCase().getDebug();

        if (LISTITEM.equals(type)) {
            ListModel model = null;

            if (obj instanceof JList) {
                JList list = (JList) obj;
                model = list.getModel();
            } else if (obj instanceof JComboBox) {
                model = ((JComboBox) obj).getModel();
            }

            getTestCase().assertNotNull("could not get model from:"
                + obj.getClass().getName(), model);

            int size = model.getSize();

            for (int i = 0; i < size; i++) {
                Object value = model.getElementAt(i);

                if (debug) {
                    System.err.println("for-each: listitem for index(" + i
                        + ") " + value);
                }

                getXMLTestCase().addProperty(id, value);
                getXMLTestCase().addProperty(
                    id + "." + INDEX,
                    new Integer(i));
                getXMLTestCase().processChildren(getElement());
            }
        } else if ("tab".equals(type)) {
            getTestCase().assertTrue(refid + " not instance of JTabbedPane",
                (obj instanceof JTabbedPane));

            JTabbedPane tab      = (JTabbedPane) obj;
            int         columns  = tab.getTabCount();
            String      columnid = id + "." + INDEX;

            for (int c = 0; c < columns; c++) {
                String title = tab.getTitleAt(c);

                getXMLTestCase().addProperty(id, title);
                getXMLTestCase().addProperty(
                    columnid,
                    new Integer(c));
                getXMLTestCase().processChildren(getElement());
            }
        } else if ("tablecolumn".equals(type)) {
            getTestCase().assertTrue(refid + " not instance of JTable",
                (obj instanceof JTable));

            JTable table    = (JTable) obj;
            int    columns  = table.getColumnCount();
            String columnid = id + "." + COLUMN;

            for (int c = 0; c < columns; c++) {
                TableColumn tc = table.getColumnModel().getColumn(c);
                Object      o = tc.getHeaderValue();

                if (o == null) {
                    o = tc.getIdentifier();
                }

                getXMLTestCase().addProperty(id, o);
                getXMLTestCase().addProperty(
                    columnid,
                    new Integer(c));
                getXMLTestCase().processChildren(getElement());
            }
        } else if (TABLECELL.equals(type)) {
            getTestCase().assertTrue(refid + " not instance of JTable",
                (obj instanceof JTable));

            JTable table   = (JTable) obj;
            int    rows    = table.getRowCount();
            int    columns = table.getColumnCount();

            String rowid    = id + "." + ROW;
            String columnid = id + "." + COLUMN;

            int    row    = getInt(ROW, -1);
            int    column = getInt(COLUMN, -1);

            if ((row == -1) && (column == -1)) {
                for (int r = 0; r < rows; r++) {
                    for (int c = 0; c < columns; c++) {
                        Object cell = table.getValueAt(r, c);

                        if (debug) {
                            System.err.println("Foreach: tablecell(" + r + ","
                                + c + ") " + cell);
                        }

                        getXMLTestCase().addProperty(id, cell);
                        getXMLTestCase().addProperty(
                            rowid,
                            new Integer(r));
                        getXMLTestCase().addProperty(
                            columnid,
                            new Integer(c));
                        getXMLTestCase().processChildren(getElement());
                    }
                }
            } else if (row == -1) {
                for (int r = 0; r < rows; r++) {
                    Object cell = table.getValueAt(r, column);

                    if (debug) {
                        System.err.println("Foreach: tablecell(" + r + ","
                            + column + ") " + cell);
                    }

                    getXMLTestCase().addProperty(id, cell);
                    getXMLTestCase().addProperty(
                        rowid,
                        new Integer(r));
                    getXMLTestCase().addProperty(
                        columnid,
                        new Integer(column));
                    getXMLTestCase().processChildren(getElement());
                }
            } else if (column == -1) {
                for (int c = 0; c < columns; c++) {
                    Object cell = table.getValueAt(row, c);

                    if (debug) {
                        System.err.println("Foreach: tablecell(" + row + ","
                            + c + ") " + cell);
                    }

                    getXMLTestCase().addProperty(id, cell);
                    getXMLTestCase().addProperty(
                        rowid,
                        new Integer(row));
                    getXMLTestCase().addProperty(
                        columnid,
                        new Integer(c));
                    getXMLTestCase().processChildren(getElement());
                }
            } else {
                Object cell = table.getValueAt(row, column);

                if (debug) {
                    System.err.println("Foreach: tablecell(" + row + ","
                        + column + ") " + cell);
                }

                getXMLTestCase().addProperty(id, cell);
                getXMLTestCase().addProperty(
                    rowid,
                    new Integer(row));
                getXMLTestCase().addProperty(
                    columnid,
                    new Integer(column));
                getXMLTestCase().processChildren(getElement());
            }
        }
    }

    /**
     * Validate that the element is correct.
     * @throws XMLException may be thrown.
     */
    public void validateElement() throws XMLException {
        super.checkElementTagName(FOREACH);

        super.checkRequiredAttribute(TYPE);
        super.checkRequiredAttribute(ID);
        super.checkRequiredAttribute(REFID);
    }
}
