package junit.extensions.xml.elements;

import junit.extensions.xml.IXMLTestCase;
import junit.extensions.xml.XMLException;

import org.w3c.dom.Element;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;


/**
 * <p>Title: class SaveTagHandler</p>
 * <p>Description: This tag handler saves the XML tree to a filename given</p>
 * <p>Copyright: Copyright (c) 2002</p>
 * <p>Company: JFCUnit project</p>
 * <H3>Tag Name</H3>
 * save
 * <H3>Required Attributes</H3>
 * file - file name to be saved.
 * <H3>Optional Attributes</H3>
 * encoding - encoding to be used.
 * <H3>Examples</H3>
 * &lt;save file="myfile.xml"/&gt;
 * @author Kevin Wilson
 * @version 1.0
 */
public class SaveTagHandler extends AbstractTagHandler {
    /**
     * Construct the Tag handler.
     * @param element Element describing the save operation.
     * @param testCase The test case.
     */
    public SaveTagHandler(final Element element, final IXMLTestCase testCase) {
        super(element, testCase);
    }

    /**
     * Save the data to the filename given.
     * @throws XMLException is thrown if the element cannot be understood.
     */
    public void processElement() throws XMLException {
        validateElement();

        File f = new File(getFileName());
        f.delete();

        try {
            f.createNewFile();

            FileOutputStream out = new FileOutputStream(f);
            junit.extensions.xml.XMLUtil.writeFile(
                getEncoding(),
                out,
                getElement().getOwnerDocument());
        } catch (IOException ex) {
            throw new XMLException("Could not write to file:"
                + f.getAbsolutePath(), ex,
                getElement(),
                getXMLTestCase().getPropertyCache());
        }
    }

    /**
     * Insure that the file attribute is specified.
     * @throws XMLException if the tag file attribute
     * is not specified.
     */
    public void validateElement() throws XMLException {
        // do the default validations from the super class
        super.validateElement();

        // check the element tag name
        this.checkRequiredAttribute(FILE);
    }

    /**
     * Get the encoding to be used.
     * @return Encoding to be used.
     */
    private String getEncoding() {
        String encoding = getString(ENCODING);

        if ((encoding == null) || (encoding.trim().length() == 0)) {
            return "UTF-8";
        }

        return encoding;
    }

    /**
     * Return the file name stored in the XML Element.
     * @return file name.
     */
    private String getFileName() {
        String file = getString(FILE).trim();

        if ((file == null) || (file.length() == 0)) {
            return null;
        }

        return file;
    }
}
