/*
 * JBoss, the OpenSource J2EE webOS
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */
package org.jboss.cache.lock;

import EDU.oswego.cs.dl.util.concurrent.SynchronizedRef;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * Provide lock ownership mapping.
 *
 * @author Ben Wang
 * @version $Id: LockMap.java,v 1.11 2005/02/22 14:19:08 belaban Exp $
 */
public class LockMap
{
   public static final int OWNER_ANY = 0;
   public static final int OWNER_READ = 1;
   public static final int OWNER_WRITE = 2;

   /** SynchronizedRef<Object> */
   private SynchronizedRef writeOwner_;

   /** List<Object> */
   private List readOwnerList_;

   public LockMap()
   {
      writeOwner_ = new SynchronizedRef(null);
      readOwnerList_ = Collections.synchronizedList(new ArrayList());
   }


   private LockMap(Object writer, List readers)
   {
      writeOwner_ = new SynchronizedRef(writer);
      readOwnerList_ = Collections.synchronizedList(readers);
   }

   public LockMap copy()
   {
      return new LockMap(writeOwner_.get(), new ArrayList(readOwnerList_));
   }

   /**
    * Check whether this owner has reader or writer ownership.
    *
    * @param owner
    * @param ownership Either <code>OWNER_ANY</code>, <code>OWNER_READ</code>,
    *                  or <code>OWNER_WRITE</code>.
    * @return
    */
   public synchronized boolean isOwner(Object owner, int ownership)
   {
      checkOwnerNull(owner);
      boolean hasOwner = false;
      switch (ownership) {
         case OWNER_ANY:
            if ((getWriteOwner() != null && getWriteOwner().equals(owner) ||
                  readOwnerList_.size() != 0 && readOwnerList_.contains(owner)))
               hasOwner = true;
            break;
         case OWNER_READ:
            if (readOwnerList_.size() != 0 && readOwnerList_.contains(owner))
               hasOwner = true;
            break;
         case OWNER_WRITE:
            if (getWriteOwner() != null && getWriteOwner().equals(owner)) hasOwner = true;
            break;
         default:
            break;
      }
      return hasOwner;
   }

   private void checkOwnerNull(Object owner)
   {
      if (owner == null)
         throw new IllegalArgumentException("Owner object is null");
   }

   /**
    * Adding a reader owner.
    *
    * @param owner
    */
   public void addReader(Object owner)
   {
      checkOwnerNull(owner);
      synchronized(readOwnerList_) {
         if(!readOwnerList_.contains(owner))
            readOwnerList_.add(owner);
      }
   }

   /**
    * Adding a writer owner.
    *
    * @param owner
    */
   public void addWriter(Object owner)
   {
      checkOwnerNull(owner);
      synchronized(this) {
         if(getWriteOwner() != null)
            throw new IllegalStateException("addWriter(): owner already existed");
         addWriteOwner(owner);
      }
   }

   private void addWriteOwner(Object owner) {
      writeOwner_.set(owner);
   }

   private Object getWriteOwner() {
      return writeOwner_.get();
   }

   /**
    * Upgrading current reader ownership to writer one.
    *
    * @param owner
    * @return True if successful.
    */
   public boolean upgrade(Object owner) throws OwnerNotExistedException
   {
      checkOwnerNull(owner);
      synchronized(this) {
         if (!readOwnerList_.contains(owner))
            throw new OwnerNotExistedException("Can't upgrade lock. Read lock owner not existed");

         readOwnerList_.remove(owner);
         addWriteOwner(owner);
      }
      return true;
   }

   /**
    * @return List of reader owner objects. Size 0 if not found.
    */
   public List readerOwners()
   {
      return readOwnerList_;
   }

   /**
    * @return Writer owner object. Null if none.
    */
   public Object writerOwner()
   {
      return getWriteOwner();
   }

   /**
    * Remove reader ownership.
    *
    * @param owner
    * @return The owner object. Null if not found.
    */
   public boolean removeReader(Object owner)
   {
      checkOwnerNull(owner);
      return readOwnerList_.remove(owner);
   }

   /**
    * Remove writer ownership.
    *
    * @param owner
    * @return The owner object. Null if not found.
    */
   public Object removeWriter(Object owner)
   {
      checkOwnerNull(owner);
      Object tmp;
      synchronized(this) {
         tmp = getWriteOwner();
         addWriteOwner(null);
      }
      return tmp;
   }

   /**
    * Remove all ownership.
    */
   public void removeAll()
   {
      synchronized(this) {
         addWriteOwner(null);
         readOwnerList_.clear();
      }
   }

   /**
    * Debugging information.
    *
    * @return
    */
   public String printInfo()
   {
      StringBuffer buf = new StringBuffer();
      synchronized(this) {
         buf.append("Read lock owners: ").append(readOwnerList_).append("\n");
         buf.append("Write lock owner: ").append(getWriteOwner()).append("\n");
      }
      return buf.toString();
   }
}
