package junit.extensions.jfcunit.eventdata;

import junit.extensions.jfcunit.JFCTestCase;

import org.w3c.dom.Element;

import java.awt.AWTEvent;
import java.awt.Component;
import java.awt.Point;
import java.awt.event.MouseEvent;

import java.util.List;
import java.util.Vector;


/**
 * DragEventSource is a wrapper for Drag Events.
 * The event encapsulates the source and destination
 * locations for the drag with xxxMouseEventData
 * types.
 *
 * @author <a href="mailto:vraravam@thoughtworks.com">Vijay Aravamudhan : ThoughtWorks Inc.</a>
 */
public class DragEventData extends AbstractEventData {
    /**
     * Destination of the drag event.
     */
    private AbstractMouseEventData m_dest;

    /**
     * Source of the drag event.
     */
    private AbstractMouseEventData m_source;

    /**
     * List of points in the drag path.
     */
    private List m_points = new Vector();

    /**
     * Original reference location of component.
     */
    private Point m_origPoint = null;

    /**
     * Constructor: Assumes a null destination and default sleep time.
     *
     * @param testCase {@link JFCTestCase} to fire the sleeps upon.
     * @param source {@link AbstractMouseEventData} indicating the
     *                starting location of the drag event.
     */
    public DragEventData(final JFCTestCase testCase,
        final AbstractMouseEventData source) {
        this(testCase, source, null, DEFAULT_SLEEPTIME);
    }

    /**
     * Constructor for a drag event. Assumes the
     * default sleepTime.
     *
     * @param testCase {@link JFCTestCase} to fire the sleeps upon.
     * @param source {@link AbstractMouseEventData} indicating the
     *                starting location of the drag event.
     * @param dest {@link AbstractMouseEventData} indicating the
     *              ending location of the drag event.
     */
    public DragEventData(final JFCTestCase testCase,
        final AbstractMouseEventData source, final AbstractMouseEventData dest) {
        this(testCase, source, dest, DEFAULT_SLEEPTIME);
    }

    /**
     * Constructor for a drag event.
     *
     * @param testCase TestCase.
     * @param source {@link AbstractMouseEventData} indicating the
     *                starting location of the drag event.
     * @param dest {@link AbstractMouseEventData} indicating the
     *              ending location of the drag event.
     * @param delay sleepTime of the event.
     */
    public DragEventData(final JFCTestCase testCase,
        final AbstractMouseEventData source, final AbstractMouseEventData dest,
        final long delay) {
        setTestCase(testCase);
        setSource(source);
        setDest(dest);
        setSleepTime(delay);

        if ((getSource() != null) && getSource().isValid()) {
            setValid(true);
        }
    }

    /**
     * This method is provided here to close the
     * abstract base class.
     *
     * @return null This method always returns null.
     */
    public final Component getComponent() {
        return null;
    }

    /**
     * Get the default modifiers.
     * @return default modifiers for the drag event.
     */
    public final int getDefaultModifiers() {
        return m_source.getDefaultModifiers();
    }

    /**
     * Set the Destination of the drag event.
     *
     * @param dest destination {@link AbstractMouseEventData}.
     */
    public final void setDest(final AbstractMouseEventData dest) {
        m_dest = dest;
    }

    /**
     * Get the destination MouseEventData.
     *
     * @return The destination {@link AbstractMouseEventData}.
     */
    public final AbstractMouseEventData getDest() {
        return m_dest;
    }

    /**
     * Get the modifier text for the current.
     *
     * @return String modifier text.
     */
    public final String getModifierText() {
        return m_source.getModifierText();
    }

    /**
     * Set the points in the drag path.
     *
     * @param points Set of points to be hit in the
     *               drag path.
     */
    public final void setPoints(final Point[] points) {
        this.m_points.clear();

        for (int i = 0; i < points.length; i++) {
            this.m_points.add(points[i]);
        }
    }

    /**
     * Get the list of points in the drag path.
     *
     * @return {@link java.awt.Point}[] list of points.
     */
    public final Point[] getPoints() {
        return (Point[]) m_points.toArray(new Point[0]);
    }

    /**
     * Set the Source of the drag event.
     *
     * @param source {@link AbstractMouseEventData} indicating the
     *                starting location of the drag event.
     */
    public final void setSource(final AbstractMouseEventData source) {
        m_source = source;

        if ((m_source != null)
                && (m_source.getComponent() != null)
                && m_source.getComponent().isVisible()) {
            this.m_origPoint = m_source.getLocationOnScreen();
        }
    }

    /**
     * Get the Source of the drag event.
     *
     * @return The source {@link AbstractMouseEventData}.
     */
    public final AbstractMouseEventData getSource() {
        return m_source;
    }

    /**
     * Add a {@link java.awt.Point} to the drag path.
     *
     * @param p {@link java.awt.Point} to be hit along the drag path.
     */
    public final void addPoint(final Point p) {
        m_points.add(p);
    }

    /**
     * This prepares the source and dest components if set,
     * and inserts/appends the screen locations onto the list
     * of points.
     *
     * @return   boolean true if the source and destination events
     *           were able to "prepare" their components.
     */
    public final boolean prepareComponent() {
        boolean results = true;
        boolean resultd = true;

        if (m_source != null) {
            results = m_source.prepareComponent();
            m_points.add(
                0,
                m_source.getLocationOnScreen());
        }

        Point compLocation = m_source.getComponent().getLocationOnScreen();

        if (m_points.size() > 1) {
            for (int i = 1; i < m_points.size(); i++) {
                Point p = (Point) m_points.get(i);
                p.translate(compLocation.x, compLocation.y);
            }
        }

        if (m_dest != null) {
            resultd = m_dest.prepareComponent();
            m_points.add(m_dest.getLocationOnScreen());
        }

        return results & resultd;
    }

    /**
     * Check if this event can consume the {@link java.awt.AWTEvent}.
     *
     * @param ae {@link java.awt.AWTEvent} to be consumed.
     * @return boolean true if the event can be consumed.
     */
    public boolean canConsume(final AWTEvent ae) {
        if (!(ae instanceof MouseEvent)) {
            return false;
        }

        MouseEvent me = (MouseEvent) ae;

        if ((me.getID() == MouseEvent.MOUSE_DRAGGED)
                || (me.getID() == MouseEvent.MOUSE_EXITED)
                || (me.getID() == MouseEvent.MOUSE_ENTERED)) {
            return true;
        }

        if (isValid()) {
            return getSource().canConsume(ae);
        }

        return false;
    }

    /**
     * Consume the {@link java.awt.AWTEvent}.
     *
     * @param ae Event to be consumed.
     * @return boolean true if the event was consumed.
     */
    public boolean consume(final AWTEvent ae) {
        if (!(ae instanceof MouseEvent)) {
            return false;
        }

        MouseEvent me = (MouseEvent) ae;

        if ((me.getID() == MouseEvent.MOUSE_ENTERED)
                || (me.getID() == MouseEvent.MOUSE_EXITED)) {
            return true;
        }

        if (me.getID() == MouseEvent.MOUSE_DRAGGED) {
            Point p = convertToSource(
                    me.getComponent(),
                    me.getPoint());
            addPoint(p);
        } else {
            return getSource().consume(ae);
        }

        return true;
    }

    /**
     * Equals comparison.
     *
     * @param o Object to be compared.
     * @return true if the objects are the same.
     */
    public boolean equals(final Object o) {
        if (!(o instanceof DragEventData)) {
            return false;
        }

        DragEventData ded = (DragEventData) o;

        if (m_source == null) {
            if (ded.getSource() != null) {
                return false;
            }
        } else {
            if (!m_source.equals(ded.getSource())) {
                return false;
            }
        }

        if (m_dest == null) {
            if (ded.getDest() != null) {
                return false;
            }
        } else {
            if (!m_dest.equals(ded.getDest())) {
                return false;
            }
        }

        return true;
    }

    /**
     * Calculate a hashcode based upon the source and the
     * dest of this class.
     * @return int hashcode.
     */
    public int hashCode() {
        int hc = super.hashCode();

        if (m_source != null) {
            hc += m_source.hashCode();
        }

        if (m_dest != null) {
            hc += m_dest.hashCode();
        }

        return hc;
    }

    /**
     * Populate the element wiht the data.
     *
     * @param e element to be populated.
     */
    public void populate(final Element e) {
    }

    /**
     * Return a string representing the eventdata.
     *
     * @return String description of the event data.
     */
    public String toString() {
        StringBuffer buf = new StringBuffer(1000);
        buf.append("DragEventData:");

        if (!isValid()) {
            buf.append(" invalid");

            return buf.toString();
        }

        buf.append("(Source:" + getSource().toString() + ")");
        buf.append("Points:");

        for (int i = 0; i < m_points.size(); i++) {
            if (i != 0) {
                buf.append(",");
            }

            buf.append(m_points.get(i));
        }

        return buf.toString();
    }

    /**
     * Convert the point to the source coordinate system.
     * @param comp Souce component.
     * @param p Point to be converted.
     * @return point of p related to source coordinates.
     */
    private Point convertToSource(final Component comp, final Point p) {
        Point screen = comp.getLocationOnScreen();
        screen.translate(p.x, p.y);

        if (m_origPoint == null) {
            return p;
        }

        int   dx = screen.x - m_origPoint.x;
        int   dy = screen.y - m_origPoint.y;

        Point delta = new Point(p);
        delta.translate(dx, dy);

        return delta;
    }
}
