package junit.extensions.jfcunit.eventdata;

import junit.extensions.jfcunit.JFCTestCase;
import junit.extensions.jfcunit.xml.JFCXMLConstants;

import org.w3c.dom.Element;

import java.awt.AWTEvent;
import java.awt.Component;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.event.MouseEvent;

import javax.swing.JList;


/**
 * Data container class that holds all the data necessary for jfcUnit to fire mouse events.
 * This class is specific to events on a JList.
 *
 * @author <a href="mailto:vraravam@thoughtworks.com">Vijay Aravamudhan : ThoughtWorks Inc.</a>
 */
public class JListMouseEventData extends AbstractMouseEventData {
    /**
     * The JList on which to trigger the event.
     */
    private JList m_list;

    /**
         * The zero-based index of the specific element on which to trigger the event.
     */
    private int m_elementIndex;

    /**
     * Constructor.
     */
    public JListMouseEventData() {
        super();
        setValid(false);
    }

    /**
     * Constructor.
     *
     * @param testCase The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param list     The JList on which to trigger the event.
     * @param element  The specific element on which to trigger the event.
     * @param numberOfClicks
     *                  Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     */
    public JListMouseEventData(final JFCTestCase testCase, final JList list,
        final Object element, final int numberOfClicks) {
        this(testCase, list, element, numberOfClicks, DEFAULT_SLEEPTIME);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param list      The JList on which to trigger the event.
     * @param element  The specific element on which to trigger the event.
     * @param numberOfClicks
     *                   Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     * @param sleepTime
     *                   The wait time in ms between each event.
     */
    public JListMouseEventData(final JFCTestCase testCase, final JList list,
        final Object element, final int numberOfClicks, final long sleepTime) {
        this(testCase, list, element, numberOfClicks, DEFAULT_ISPOPUPTRIGGER,
            sleepTime);
    }

    /**
     * Constructor.
     *
     * @param testCase The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param list     The JList on which to trigger the event.
     * @param element  The specific element on which to trigger the event.
     * @param numberOfClicks
     *                  Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     * @param isPopupTrigger
     *                  boolean specifying whether this event will show a popup.
     */
    public JListMouseEventData(final JFCTestCase testCase, final JList list,
        final Object element, final int numberOfClicks,
        final boolean isPopupTrigger) {
        this(testCase, list, element, numberOfClicks,
            getDefaultModifiers(isPopupTrigger), isPopupTrigger,
            DEFAULT_SLEEPTIME);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param list      The JList on which to trigger the event.
     * @param element  The specific element on which to trigger the event.
     * @param numberOfClicks
     *                   Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     * @param isPopupTrigger
     *                   boolean specifying whether this event will show a popup.
     * @param sleepTime
     *                   The wait time in ms between each event.
     */
    public JListMouseEventData(final JFCTestCase testCase, final JList list,
        final Object element, final int numberOfClicks,
        final boolean isPopupTrigger, final long sleepTime) {
        this(testCase, list, element, numberOfClicks,
            getDefaultModifiers(isPopupTrigger), isPopupTrigger, sleepTime);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param list      The JList on which to trigger the event.
     * @param element  The specific element on which to trigger the event.
     * @param numberOfClicks
     *                   Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     * @param modifiers The modifier key values that need to be passed onto the event.
     * @param isPopupTrigger
     *                   boolean specifying whether this event will show a popup.
     * @param sleepTime
     *                   The wait time in ms between each event.
     */
    public JListMouseEventData(final JFCTestCase testCase, final JList list,
        final Object element, final int numberOfClicks, final int modifiers,
        final boolean isPopupTrigger, final long sleepTime) {
        this(testCase, list, element, numberOfClicks, modifiers,
            isPopupTrigger, sleepTime, DEFAULT_POSITION, null);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param list      The JList on which to trigger the event.
     * @param element  The specific element on which to trigger the event.
     * @param numberOfClicks
     *                   Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     * @param modifiers The modifier key values that need to be passed onto the event.
     * @param isPopupTrigger
     *                   boolean specifying whether this event will show a popup.
     * @param sleepTime
     *                   The wait time in ms between each event.
     * @param position  The relative mouse position within the cell.
     */
    public JListMouseEventData(final JFCTestCase testCase, final JList list,
        final Object element, final int numberOfClicks, final int modifiers,
        final boolean isPopupTrigger, final long sleepTime, final int position) {
        this(testCase, list, element, numberOfClicks, modifiers,
            isPopupTrigger, sleepTime, position, null);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param list      The JList on which to trigger the event.
     * @param element  The specific element on which to trigger the event.
     * @param numberOfClicks
     *                   Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     * @param modifiers The modifier key values that need to be passed onto the event.
     * @param isPopupTrigger
     *                   boolean specifying whether this event will show a popup.
     * @param sleepTime
     *                   The wait time in ms between each event.
     * @param referencePoint
     *                   The CUSTOM mouse position within the cell.
     */
    public JListMouseEventData(final JFCTestCase testCase, final JList list,
        final Object element, final int numberOfClicks, final int modifiers,
        final boolean isPopupTrigger, final long sleepTime,
        final Point referencePoint) {
        this(testCase, list, element, numberOfClicks, modifiers,
            isPopupTrigger, sleepTime, CUSTOM, referencePoint);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param list      The JList on which to trigger the event.
     * @param element  The specific element on which to trigger the event.
     * @param numberOfClicks
     *                   Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     * @param modifiers The modifier key values that need to be passed onto the event.
     * @param isPopupTrigger
     *                   boolean specifying whether this event will show a popup.
     * @param sleepTime
     *                   The wait time in ms between each event.
     * @param position  The relative mouse position within the cell.
     * @param referencePoint
     *                                   If position is CUSTOM then the point is a offset from
     *                                   the location of the component. If the position is PERCENT
     *                                   then the location is a percentage offset of the hight and width.
     *                                   Otherwise, the referencePoint is unused.
     */
    public JListMouseEventData(final JFCTestCase testCase, final JList list,
        final Object element, final int numberOfClicks, final int modifiers,
        final boolean isPopupTrigger, final long sleepTime, final int position,
        final Point referencePoint) {
        this(testCase, list,
            getIndexOf(list, element), numberOfClicks, modifiers,
            isPopupTrigger, sleepTime, position, referencePoint);
    }

    /**
     * Constructor.
     *
     * @param testCase The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param list     The JList on which to trigger the event.
     * @param elementIndex
     *                  The zero-based index of the specific element on which to trigger the event.
     * @param numberOfClicks
     *                  Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     */
    public JListMouseEventData(final JFCTestCase testCase, final JList list,
        final int elementIndex, final int numberOfClicks) {
        this(testCase, list, elementIndex, numberOfClicks, DEFAULT_SLEEPTIME);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param list      The JList on which to trigger the event.
     * @param elementIndex
     *                   The zero-based index of the specific element on which to trigger the event.
     * @param numberOfClicks
     *                   Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     * @param sleepTime
     *                   The wait time in ms between each event.
     */
    public JListMouseEventData(final JFCTestCase testCase, final JList list,
        final int elementIndex, final int numberOfClicks, final long sleepTime) {
        this(testCase, list, elementIndex, numberOfClicks,
            DEFAULT_ISPOPUPTRIGGER, sleepTime);
    }

    /**
     * Constructor.
     *
     * @param testCase The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param list     The JList on which to trigger the event.
     * @param elementIndex
     *                  The zero-based index of the specific element on which to trigger the event.
     * @param numberOfClicks
     *                  Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     * @param isPopupTrigger
     *                  boolean specifying whether this event will show a popup.
     */
    public JListMouseEventData(final JFCTestCase testCase, final JList list,
        final int elementIndex, final int numberOfClicks,
        final boolean isPopupTrigger) {
        this(testCase, list, elementIndex, numberOfClicks,
            getDefaultModifiers(isPopupTrigger), isPopupTrigger,
            DEFAULT_SLEEPTIME);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param list      The JList on which to trigger the event.
     * @param elementIndex
     *                   The zero-based index of the specific element on which to trigger the event.
     * @param numberOfClicks
     *                   Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     * @param isPopupTrigger
     *                   boolean specifying whether this event will show a popup.
     * @param sleepTime
     *                   The wait time in ms between each event.
     */
    public JListMouseEventData(final JFCTestCase testCase, final JList list,
        final int elementIndex, final int numberOfClicks,
        final boolean isPopupTrigger, final long sleepTime) {
        this(testCase, list, elementIndex, numberOfClicks,
            getDefaultModifiers(isPopupTrigger), isPopupTrigger, sleepTime);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param list      The JList on which to trigger the event.
     * @param elementIndex
     *                   The zero-based index of the specific element on which to trigger the event.
     * @param numberOfClicks
     *                   Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     * @param modifiers The modifier key values that need to be passed onto the event.
     * @param isPopupTrigger
     *                   boolean specifying whether this event will show a popup.
     * @param sleepTime
     *                   The wait time in ms between each event.
     */
    public JListMouseEventData(final JFCTestCase testCase, final JList list,
        final int elementIndex, final int numberOfClicks, final int modifiers,
        final boolean isPopupTrigger, final long sleepTime) {
        this(testCase, list, elementIndex, numberOfClicks, modifiers,
            isPopupTrigger, sleepTime, DEFAULT_POSITION, null);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param list      The JList on which to trigger the event.
     * @param elementIndex
     *                   The zero-based index of the specific element on which to trigger the event.
     * @param numberOfClicks
     *                   Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     * @param modifiers The modifier key values that need to be passed onto the event.
     * @param isPopupTrigger
     *                   boolean specifying whether this event will show a popup.
     * @param sleepTime
     *                   The wait time in ms between each event.
     * @param position  The relative mouse position within the cell.
     */
    public JListMouseEventData(final JFCTestCase testCase, final JList list,
        final int elementIndex, final int numberOfClicks, final int modifiers,
        final boolean isPopupTrigger, final long sleepTime, final int position) {
        this(testCase, list, elementIndex, numberOfClicks, modifiers,
            isPopupTrigger, sleepTime, position, null);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param list      The JList on which to trigger the event.
     * @param elementIndex
     *                   The zero-based index of the specific element on which to trigger the event.
     * @param numberOfClicks
     *                   Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     * @param modifiers The modifier key values that need to be passed onto the event.
     * @param isPopupTrigger
     *                   boolean specifying whether this event will show a popup.
     * @param sleepTime
     *                   The wait time in ms between each event.
     * @param referencePoint
     *                   The CUSTOM mouse position within the cell.
     */
    public JListMouseEventData(final JFCTestCase testCase, final JList list,
        final int elementIndex, final int numberOfClicks, final int modifiers,
        final boolean isPopupTrigger, final long sleepTime,
        final Point referencePoint) {
        this(testCase, list, elementIndex, numberOfClicks, modifiers,
            isPopupTrigger, sleepTime, CUSTOM, referencePoint);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param list      The JList on which to trigger the event.
     * @param elementIndex
     *                   The zero-based index of the specific element on which to trigger the event.
     * @param numberOfClicks
     *                   Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     * @param modifiers The modifier key values that need to be passed onto the event.
     * @param isPopupTrigger
     *                   boolean specifying whether this event will show a popup.
     * @param sleepTime
     *                   The wait time in ms between each event.
     * @param position  The relative mouse position within the cell.
     * @param referencePoint
     *                                   If position is CUSTOM then the point is a offset from
     *                                   the location of the component. If the position is PERCENT
     *                                   then the location is a percentage offset of the hight and width.
     *                                   Otherwise, the referencePoint is unused.
     */
    public JListMouseEventData(final JFCTestCase testCase, final JList list,
        final int elementIndex, final int numberOfClicks, final int modifiers,
        final boolean isPopupTrigger, final long sleepTime, final int position,
        final Point referencePoint) {
        setTestCase(testCase);
        setSource(list);
        setNumberOfClicks(numberOfClicks);
        setModifiers(modifiers);
        setPopupTrigger(isPopupTrigger);
        setElementIndex(elementIndex);
        setSleepTime(sleepTime);
        setPosition(position);
        setReferencePoint(referencePoint);
        setValid(true);
    }

    /**
     * The component on which the event has to be fired.
     *
     * @return The component
     */
    public Component getComponent() {
        // by default, the component is the same as the source
        return getSource();
    }

    /**
     * Set the attribute value.
     *
     * @param elementIndex The new value of the attribute
     */
    public void setElementIndex(final int elementIndex) {
        m_elementIndex = elementIndex;
    }

    /**
     * Get the attribute value.
     *
     * @return int    The value of the attribute
     */
    public int getElementIndex() {
        return m_elementIndex;
    }

    /**
     * Set the attribute value.
     *
     * @param list  The new value of the attribute.
     */
    public void setSource(final JList list) {
        m_list = list;
    }

    /**
     * Get the attribute value.
     *
     * @return JList    The value of the attribute.
     */
    public JList getSource() {
        return m_list;
    }

    /**
     * Check if this event can consume the event given.
     * @param ae AWTEvent to be consumed.
     * @return true if the event may be consumed.
     */
    public boolean canConsume(final AWTEvent ae) {
        if (!super.canConsume(ae)
                || !(ae.getSource() instanceof JList)
                || !sameSource(ae)) {
            return false;
        }

        if (isValid()) {
            JList source = (JList) ae.getSource();
            int   index = source.locationToIndex(
                    new Point(
                        ((MouseEvent) ae).getX(),
                        ((MouseEvent) ae).getY()));

            if (index != getElementIndex()) {
                return false;
            }
        }

        return true;
    }

    /**
     * Consume the event.
     * @param ae AWTEvent to be consumed.
     * @return boolean true if the event was consumed.
     */
    public boolean consume(final AWTEvent ae) {
        if (super.consume(ae)) {
            return true;
        }

        MouseEvent me     = (MouseEvent) ae;
        JList      source = (JList) me.getSource();
        setSource(source);
        setModifiers(me.getModifiers());
        setNumberOfClicks(me.getClickCount());
        setPopupTrigger(me.isPopupTrigger());

        Point p = new Point(
                me.getX(),
                me.getY());
        Point screen = source.getLocationOnScreen();
        screen.translate(p.x, p.y);
        setLocationOnScreen(screen);
        setSleepTime(getDefaultSleepTime());

        int index = m_list.locationToIndex(new Point(
                    me.getX(),
                    me.getY()));
        setElementIndex(index);

        setPosition(CENTER);
        setReferencePoint(null);

        setValid(true);

        return true;
    }

    /**
     * Compare to event datas and deterime if they are equal.
     *
     * @param o Object to be compared.
     * @return true if the events are the same.
     */
    public boolean equals(final Object o) {
        if (!super.equals(o)) {
            return false;
        }

        JListMouseEventData data = (JListMouseEventData) o;

        return (data.getElementIndex() == getElementIndex());
    }

    /**
     * Get the hashCode.
     * @return hashCode for this object.
     */
    public int hashCode() {
        return super.hashCode() + this.m_elementIndex;
    }

    /**
     * Populate the XML Element.
     * @param e element to be populated.
     */
    public void populate(final Element e) {
        super.populate(e);
        e.setAttribute(JFCXMLConstants.TYPE, "JListMouseEventData");
        e.setAttribute(JFCXMLConstants.INDEX, "" + getElementIndex());
    }

    /**
     * Prepare the component to receive the event.
     *
     * @return true if the component is ready to receive the event.
     */
    public boolean prepareComponent() {
        if (!isValidForProcessing(getSource())) {
            return false;
        }

        m_list.ensureIndexIsVisible(m_elementIndex);

        JFCTestCase testCase = getTestCase();

        if (testCase != null) {
            // try to clear the event queue
            testCase.flushAWT();
        }

        Rectangle cellRect = m_list.getCellBounds(m_elementIndex, m_elementIndex);

        if (testCase != null) {
            // try to clear the event queue
            testCase.pauseAWT();
        }

        Point p = calculatePoint(cellRect);

        if (!m_list.getVisibleRect().contains(p)) {
            Rectangle vis     = m_list.getVisibleRect();
            Rectangle newView = new Rectangle(p.x - (int) (vis.width / 2),
                    p.y - (int) (vis.height / 2), vis.width, vis.height);
            m_list.scrollRectToVisible(newView);
        }

        Point screen = m_list.getLocationOnScreen();
        screen.translate(p.x, p.y);
        setLocationOnScreen(screen);

        return true;
    }

    /**
     * Get string description of event.
     * @return String description of event.
     */
    public String toString() {
        if (!isValid()) {
            return super.toString();
        }

        StringBuffer buf = new StringBuffer(1000);
        buf.append(super.toString());
        buf.append(" index: " + getElementIndex());

        return buf.toString();
    }

    /**
     * Finds the index of the element passed in from the JComboBox.
     *
     * @param    list       The JList which holds the elements.
     * @param    element    The element whose index has to be found out.
     * @return   int   The zero-based index where the element occurs
     *                 (-1 is returned if not found)
     */
    private static int getIndexOf(final JList list, final Object element) {
        for (int i = 0; i < list.getModel().getSize(); i++) {
            if ((list.getModel().getElementAt(i) != null)
                    && list.getModel().getElementAt(i).equals(element)) {
                return i;
            }
        }

        return -1;
    }
}
