package junit.extensions.jfcunit.eventdata;

import junit.extensions.jfcunit.JFCTestCase;
import junit.extensions.jfcunit.xml.JFCXMLConstants;

import org.w3c.dom.Element;

import java.awt.AWTEvent;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.event.MouseEvent;

import javax.swing.JTable;


/**
 * Data container class that holds all the data necessary for jfcUnit to fire mouse events.
 * This class is specific to events on a JTable.
 *
 * @author <a href="mailto:vraravam@thoughtworks.com">Vijay Aravamudhan : ThoughtWorks Inc.</a>
 */
public class JTableMouseEventData extends AbstractMouseEventData {
    /**
     * The JTable on which to trigger the event.
     */
    private JTable m_table;

    /**
     * The zero-based column index of the specific cell on which to trigger the event.
     */
    private int m_columnIndex;

    /**
         * The zero-based row index of the specific cell on which to trigger the event.
     */
    private int m_rowIndex;

    /**
     * Constructor.
     */
    public JTableMouseEventData() {
        super();
        setValid(false);
    }

    /**
     * Constructor.
     *
     * @param testCase The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param table    The component on which to trigger the event.
     * @param rowIndex The zero-based row index of the specific cell on which to trigger the event.
     * @param columnIndex
     *                  The zero-based column index of the specific cell on which to trigger the event.
     * @param numberOfClicks
     *                  Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     */
    public JTableMouseEventData(final JFCTestCase testCase, final JTable table,
        final int rowIndex, final int columnIndex, final int numberOfClicks) {
        this(testCase, table, rowIndex, columnIndex, numberOfClicks,
            DEFAULT_SLEEPTIME);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param table     The component on which to trigger the event.
     * @param rowIndex  The zero-based row index of the specific cell on which to trigger the event.
     * @param columnIndex
     *                   The zero-based column index of the specific cell on which to trigger the event.
     * @param numberOfClicks
     *                   Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     * @param sleepTime
     *                   The wait time in ms between each event.
     */
    public JTableMouseEventData(final JFCTestCase testCase, final JTable table,
        final int rowIndex, final int columnIndex, final int numberOfClicks,
        final long sleepTime) {
        this(testCase, table, rowIndex, columnIndex, numberOfClicks,
            DEFAULT_ISPOPUPTRIGGER, sleepTime);
    }

    /**
     * Constructor.
     *
     * @param testCase The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param table    The component on which to trigger the event.
     * @param rowIndex The zero-based row index of the specific cell on which to trigger the event.
     * @param columnIndex
     *                  The zero-based column index of the specific cell on which to trigger the event.
     * @param numberOfClicks
     *                  Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     * @param isPopupTrigger
     *                  boolean specifying whether this event will show a popup.
     */
    public JTableMouseEventData(final JFCTestCase testCase, final JTable table,
        final int rowIndex, final int columnIndex, final int numberOfClicks,
        final boolean isPopupTrigger) {
        this(testCase, table, rowIndex, columnIndex, numberOfClicks,
            isPopupTrigger, DEFAULT_SLEEPTIME);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param table     The component on which to trigger the event.
     * @param rowIndex  The zero-based row index of the specific cell on which to trigger the event.
     * @param columnIndex
     *                   The zero-based column index of the specific cell on which to trigger the event.
     * @param numberOfClicks
     *                   Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     * @param isPopupTrigger
     *                   boolean specifying whether this event will show a popup.
     * @param sleepTime
     *                   The wait time in ms between each event.
     */
    public JTableMouseEventData(final JFCTestCase testCase, final JTable table,
        final int rowIndex, final int columnIndex, final int numberOfClicks,
        final boolean isPopupTrigger, final long sleepTime) {
        this(testCase, table, rowIndex, columnIndex, numberOfClicks,
            getDefaultModifiers(isPopupTrigger), isPopupTrigger, sleepTime);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param table     The component on which to trigger the event.
     * @param rowIndex  The zero-based row index of the specific cell on which to trigger the event.
     * @param columnIndex
     *                   The zero-based column index of the specific cell on which to trigger the event.
     * @param numberOfClicks
     *                   Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     * @param modifiers The modifier key values that need to be passed onto the event.
     * @param isPopupTrigger
     *                   boolean specifying whether this event will show a popup.
     * @param sleepTime
     *                   The wait time in ms between each event.
     */
    public JTableMouseEventData(final JFCTestCase testCase, final JTable table,
        final int rowIndex, final int columnIndex, final int numberOfClicks,
        final int modifiers, final boolean isPopupTrigger, final long sleepTime) {
        this(testCase, table, rowIndex, columnIndex, numberOfClicks, modifiers,
            isPopupTrigger, sleepTime, DEFAULT_POSITION, null);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param table     The component on which to trigger the event.
     * @param rowIndex  The zero-based row index of the specific cell on which to trigger the event.
     * @param columnIndex
     *                   The zero-based column index of the specific cell on which to trigger the event.
     * @param numberOfClicks
     *                   Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     * @param modifiers The modifier key values that need to be passed onto the event.
     * @param isPopupTrigger
     *                   boolean specifying whether this event will show a popup.
     * @param sleepTime
     *                   The wait time in ms between each event.
     * @param position  The relative mouse position within the cell.
     */
    public JTableMouseEventData(final JFCTestCase testCase, final JTable table,
        final int rowIndex, final int columnIndex, final int numberOfClicks,
        final int modifiers, final boolean isPopupTrigger,
        final long sleepTime, final int position) {
        this(testCase, table, rowIndex, columnIndex, numberOfClicks, modifiers,
            isPopupTrigger, sleepTime, position, null);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param table     The component on which to trigger the event.
     * @param rowIndex  The zero-based row index of the specific cell on which to trigger the event.
     * @param columnIndex
     *                   The zero-based column index of the specific cell on which to trigger the event.
     * @param numberOfClicks
     *                   Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     * @param modifiers The modifier key values that need to be passed onto the event.
     * @param isPopupTrigger
     *                   boolean specifying whether this event will show a popup.
     * @param sleepTime
     *                   The wait time in ms between each event.
     * @param referencePoint
     *                   The CUSTOM mouse position within the cell.
     */
    public JTableMouseEventData(final JFCTestCase testCase, final JTable table,
        final int rowIndex, final int columnIndex, final int numberOfClicks,
        final int modifiers, final boolean isPopupTrigger,
        final long sleepTime, final Point referencePoint) {
        this(testCase, table, rowIndex, columnIndex, numberOfClicks, modifiers,
            isPopupTrigger, sleepTime, CUSTOM, referencePoint);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param table     The component on which to trigger the event.
     * @param rowIndex  The zero-based row index of the specific cell on which to trigger the event.
     * @param columnIndex
     *                   The zero-based column index of the specific cell on which to trigger the event.
     * @param numberOfClicks
     *                   Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     * @param modifiers The modifier key values that need to be passed onto the event.
     * @param isPopupTrigger
     *                   boolean specifying whether this event will show a popup.
     * @param sleepTime
     *                   The wait time in ms between each event.
     * @param position  The relative mouse position within the cell.
     * @param referencePoint
     *                                   If position is CUSTOM then the point is a offset from
     *                                   the location of the component. If the position is PERCENT
     *                                   then the location is a percentage offset of the hight and width.
     *                                   Otherwise, the referencePoint is unused.
     */
    public JTableMouseEventData(final JFCTestCase testCase, final JTable table,
        final int rowIndex, final int columnIndex, final int numberOfClicks,
        final int modifiers, final boolean isPopupTrigger,
        final long sleepTime, final int position, final Point referencePoint) {
        setTestCase(testCase);
        setSource(table);
        setNumberOfClicks(numberOfClicks);
        setModifiers(modifiers);
        setPopupTrigger(isPopupTrigger);
        setRowIndex(rowIndex);
        setColumnIndex(columnIndex);
        setSleepTime(sleepTime);
        setPosition(position);
        setReferencePoint(referencePoint);
        setValid(true);
    }

    /**
     * Set the attribute value.
     *
     * @param columnIndex The new value of the attribute
     */
    public final void setColumnIndex(final int columnIndex) {
        m_columnIndex = columnIndex;
    }

    /**
     * Get the attribute value.
     *
     * @return int    The value of the attribute
     */
    public final int getColumnIndex() {
        return m_columnIndex;
    }

    /**
     * Set the attribute value.
     *
     * @param rowIndex The new value of the attribute
     */
    public final void setRowIndex(final int rowIndex) {
        m_rowIndex = rowIndex;
    }

    /**
     * Get the attribute value.
     *
     * @return int    The value of the attribute
     */
    public final int getRowIndex() {
        return m_rowIndex;
    }

    /**
     * Set the attribute value.
     *
     * @param table  The new value of the attribute
     */
    public final void setSource(final JTable table) {
        m_table = table;
    }

    /**
     * The component on which the event has to be fired.
     *
     * @return The component
     */
    public Component getComponent() {
        // by default, the component is the same as the source
        return getSource();
    }

    /**
     * Get the attribute value.
     *
     * @return JTable    The value of the attribute
     */
    public JTable getSource() {
        return m_table;
    }

    /**
     * Check if this event can consume the event given.
     * @param ae AWTEvent to be consumed.
     * @return true if the event may be consumed.
     */
    public boolean canConsume(final AWTEvent ae) {
        if ((ae.getSource() instanceof JTable)
                && super.canConsume(ae)
                && sameSource(ae)) {
            return true;
        }

        return false;
    }

    /**
     * Consume the event.
     * @param ae AWTEvent to be consumed.
     * @return boolean true if the event was consumed.
     */
    public boolean consume(final AWTEvent ae) {
        if (super.consume(ae)) {
            return true;
        }

        MouseEvent me     = (MouseEvent) ae;
        JTable     source = (JTable) me.getSource();
        setSource(source);
        setModifiers(me.getModifiers());
        setNumberOfClicks(me.getClickCount());
        setPopupTrigger(me.isPopupTrigger());

        Point p = new Point(
                me.getX(),
                me.getY());
        Point screen = source.getLocationOnScreen();
        screen.translate(p.x, p.y);
        setLocationOnScreen(screen);
        setSleepTime(getDefaultSleepTime());

        int row    = source.rowAtPoint(p);
        int column = source.columnAtPoint(p);

        setRowIndex(row);
        setColumnIndex(column);
        setPosition(CENTER);
        setReferencePoint(null);

        setValid(true);

        return true;
    }

    /**
     * Compare to event datas and deterime if they are equal.
     *
     * @param o Object to be compared.
     * @return true if the events are the same.
     */
    public boolean equals(final Object o) {
        if (!super.equals(o)) {
            return false;
        }

        JTableMouseEventData data = (JTableMouseEventData) o;

        return (data.getRowIndex() == getRowIndex())
            && (data.getColumnIndex() == getColumnIndex());
    }

    /**
     * Get the hashCode.
     * @return int hashCode.
     */
    public int hashCode() {
        return super.hashCode() + (m_rowIndex * m_columnIndex);
    }

    /**
     * Populate the XML Element with this objects attributes.
     * @param e Element to be populated.
     */
    public void populate(final Element e) {
        super.populate(e);
        e.setAttribute(JFCXMLConstants.TYPE, "JTableMouseEventData");
        e.setAttribute(JFCXMLConstants.ROW, "" + getRowIndex());
        e.setAttribute(JFCXMLConstants.COLUMN, "" + getColumnIndex());
    }

    /**
     * Prepare the component to receive the event.
     *
     * @return true if the component is ready to receive the event.
     */
    public boolean prepareComponent() {
        if (!isValidForProcessing(getSource())) {
            return false;
        }

        JFCTestCase testCase = getTestCase();

        if (testCase != null) {
            // try to clear the event queue
            testCase.pauseAWT();
        }

        Point p;

        if ((m_rowIndex == -1) || (m_columnIndex == -1)) {
            int       height = m_table.getHeight();
            Dimension vp = m_table.getPreferredScrollableViewportSize();
            int       y  = 0;

            if (height < vp.height) {
                height += ((vp.height - height) / 2);
            } else {
                height = height / 2;
            }

            int width = m_table.getWidth();

            if (width < vp.width) {
                width += ((vp.width - width) / 2);
            } else {
                width = width / 2;
            }

            p = new Point(width, height);

            Object parent = m_table.getParent();

            getTestCase().resumeAWT();
            getTestCase().getHelper().enterClickAndLeave(
                new MouseEventData(
                    getTestCase(),
                    (Component) parent,
                    getNumberOfClicks(),
                    getModifiers(),
                    getPopupTrigger(),
                    getSleepTime(),
                    CUSTOM,
                    p));

            return false;
        } else {
            Rectangle cellRect = m_table.getCellRect(m_rowIndex, m_columnIndex,
                    true);

            if (testCase != null) {
                // try to clear the event queue
                testCase.flushAWT();
            }

            p = calculatePoint(cellRect);
        }

        testCase.pauseAWT();

        if (!m_table.getVisibleRect().contains(p)) {
            Rectangle vis     = m_table.getVisibleRect();
            final Rectangle newView = new Rectangle(p.x - (int) (vis.width / 2),
                    p.y - (int) (vis.height / 2), vis.width, vis.height);
            m_table.scrollRectToVisible(newView);
        }

        Point screen = m_table.getLocationOnScreen();
        screen.translate(p.x, p.y);
        setLocationOnScreen(screen);

        return true;
    }

    /**
     * Get string description of event.
     * @return String description of event.
     */
    public String toString() {
        if (!isValid()) {
            return super.toString();
        }

        StringBuffer buf = new StringBuffer(1000);
        buf.append(super.toString());
        buf.append(" row: " + getRowIndex());
        buf.append(" column: " + getColumnIndex());

        return buf.toString();
    }
}
