package junit.extensions.jfcunit.eventdata;

import junit.extensions.jfcunit.JFCTestCase;
import junit.extensions.jfcunit.xml.JFCXMLConstants;

import java.awt.AWTEvent;
import java.awt.Component;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.event.MouseEvent;

import javax.swing.text.BadLocationException;
import javax.swing.text.JTextComponent;


/**
 * Data container class that holds all the data necessary for jfcUnit to
 * fire mouse events.
 *
 * @author <a href="mailto:vraravam@thoughtworks.com">
 * Vijay Aravamudhan : ThoughtWorks Inc.</a>
 */
public class JTextComponentMouseEventData extends AbstractMouseEventData {
    /**
     * The Component on which to trigger the event.
     */
    private JTextComponent m_comp;

    /**
     * Offset into the text component in characters.
     */
    private int m_offset = INVALID_TEXT_OFFSET;

    /**
     * Constructor.
     */
    public JTextComponentMouseEventData() {
        super();
        setValid(false);
    }

    /**
     * Constructor.
     *
     * @param testCase The JFCTestCase on whose thread <code>awtSleep()</code>
     *                 has to be invoked.
     * @param comp     The component on which to trigger the event.
     */
    public JTextComponentMouseEventData(final JFCTestCase testCase,
        final JTextComponent comp) {
        this(testCase, comp, DEFAULT_NUMBEROFCLICKS);
    }

    /**
     * Constructor.
     *
     * @param testCase The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param comp     The component on which to trigger the event.
     * @param numberOfClicks
     *                 Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     */
    public JTextComponentMouseEventData(final JFCTestCase testCase,
        final JTextComponent comp, final int numberOfClicks) {
        this(testCase, comp, numberOfClicks, DEFAULT_MOUSE_MODIFIERS);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code>
     *                  has to be invoked.
     * @param comp      The component on which to trigger the event.
     * @param sleepTime
     *                  The wait time in ms between each event.
     */
    public JTextComponentMouseEventData(final JFCTestCase testCase,
        final JTextComponent comp, final long sleepTime) {
        this(testCase, comp, DEFAULT_NUMBEROFCLICKS);
        setSleepTime(sleepTime);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code>
     *                  has to be invoked.
     * @param comp      The component on which to trigger the event.
     * @param numberOfClicks
     *                  Number of clicks in the MouseEvent (1 for single-click,
     *                  2 for double clicks)
     * @param modifiers The modifier key values that need to be passed onto the
     *                  event.
     */
    public JTextComponentMouseEventData(final JFCTestCase testCase,
        final JTextComponent comp, final int numberOfClicks, final int modifiers) {
        this(testCase, comp, numberOfClicks, modifiers, DEFAULT_ISPOPUPTRIGGER);
    }

    /**
     * Constructor.
     *
     * @param testCase The JFCTestCase on whose thread <code>awtSleep()</code>
     *                 has to be invoked.
     * @param comp     The component on which to trigger the event.
     * @param numberOfClicks
     *                 Number of clicks in the MouseEvent (1 for single-click,
     *                 2 for double clicks)
     * @param isPopupTrigger
     *                 boolean specifying whether this event will show a popup.
     */
    public JTextComponentMouseEventData(final JFCTestCase testCase,
        final JTextComponent comp, final int numberOfClicks,
        final boolean isPopupTrigger) {
        this(testCase, comp, numberOfClicks,
            getDefaultModifiers(isPopupTrigger), isPopupTrigger);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code>
     *                  has to be invoked.
     * @param comp      The component on which to trigger the event.
     * @param numberOfClicks
     *                  Number of clicks in the MouseEvent (1 for single-click,
     *                  2 for double clicks)
     * @param sleepTime
     *                  The wait time in ms between each event.
     */
    public JTextComponentMouseEventData(final JFCTestCase testCase,
        final JTextComponent comp, final int numberOfClicks,
        final long sleepTime) {
        this(testCase, comp, numberOfClicks, DEFAULT_MOUSE_MODIFIERS,
            DEFAULT_ISPOPUPTRIGGER, sleepTime);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code>
     *                  has to be invoked.
     * @param comp      The component on which to trigger the event.
     * @param numberOfClicks
     *                  Number of clicks in the MouseEvent (1 for single-click,
     *                  2 for double clicks)
     * @param modifiers The modifier key values that need to be passed onto the
     *                  event.
     * @param isPopupTrigger
     *                  boolean specifying whether this event will show a popup.
     */
    public JTextComponentMouseEventData(final JFCTestCase testCase,
        final JTextComponent comp, final int numberOfClicks,
        final int modifiers, final boolean isPopupTrigger) {
        this(testCase, comp, numberOfClicks, modifiers, isPopupTrigger,
            DEFAULT_SLEEPTIME);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code>
     *                  has to be invoked.
     * @param comp      The component on which to trigger the event.
     * @param numberOfClicks
     *                  Number of clicks in the MouseEvent (1 for single-click,
     *                  2 for double clicks)
     * @param isPopupTrigger
     *                  boolean specifying whether this event will show a popup.
     * @param sleepTime
     *                  The wait time in ms between each event.
     */
    public JTextComponentMouseEventData(final JFCTestCase testCase,
        final JTextComponent comp, final int numberOfClicks,
        final boolean isPopupTrigger, final long sleepTime) {
        this(testCase, comp, numberOfClicks,
            getDefaultModifiers(isPopupTrigger), isPopupTrigger, sleepTime);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code>
     *                  has to be invoked.
     * @param comp      The component on which to trigger the event.
     * @param numberOfClicks
     *                  Number of clicks in the MouseEvent (1 for single-click,
     *                  2 for double clicks)
     * @param modifiers The modifier key values that need to be passed onto the
     *                  event.
     * @param isPopupTrigger
     *                  boolean specifying whether this event will show a popup.
     * @param sleepTime The wait time in ms between each event.
     */
    public JTextComponentMouseEventData(final JFCTestCase testCase,
        final JTextComponent comp, final int numberOfClicks,
        final int modifiers, final boolean isPopupTrigger, final long sleepTime) {
        this(testCase, comp, numberOfClicks, modifiers, isPopupTrigger,
            sleepTime, DEFAULT_POSITION, null, INVALID_TEXT_OFFSET);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code>
     *                  has to be invoked.
     * @param comp      The component on which to trigger the event.
     * @param numberOfClicks
     *                   Number of clicks in the MouseEvent (1 for single-click,
     *                   2 for double clicks)
     * @param modifiers The modifier key values that need to be passed onto the
     *                  event.
     * @param isPopupTrigger
     *                   boolean specifying whether this event will show a popup.
     * @param sleepTime
     *                   The wait time in ms between each event.
     * @param position  The relative mouse position within the cell.
     */
    public JTextComponentMouseEventData(final JFCTestCase testCase,
        final JTextComponent comp, final int numberOfClicks,
        final int modifiers, final boolean isPopupTrigger,
        final long sleepTime, final int position) {
        this(testCase, comp, numberOfClicks, modifiers, isPopupTrigger,
            sleepTime, position, null, INVALID_TEXT_OFFSET);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code>
     *                  has to be invoked.
     * @param comp      The component on which to trigger the event.
     * @param numberOfClicks
     *                  Number of clicks in the MouseEvent (1 for single-click,
     *                  2 for double clicks)
     * @param modifiers The modifier key values that need to be passed onto the
     *                  event.
     * @param isPopupTrigger
     *                  boolean specifying whether this event will show a popup.
     * @param sleepTime
     *                  The wait time in ms between each event.
     * @param position  This parameter will be ignored. It will be set to OFFSET.
     * @param offset    The offset into the text component.
     */
    public JTextComponentMouseEventData(final JFCTestCase testCase,
        final JTextComponent comp, final int numberOfClicks,
        final int modifiers, final boolean isPopupTrigger,
        final long sleepTime, final int position, final int offset) {
        this(testCase, comp, numberOfClicks, modifiers, isPopupTrigger,
            sleepTime, OFFSET, null, offset);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code>
     *                  has to be invoked.
     * @param comp      The component on which to trigger the event.
     * @param numberOfClicks
     *                   Number of clicks in the MouseEvent (1 for single-click,
     *                   2 for double clicks)
     * @param modifiers The modifier key values that need to be passed onto the
     *                  event.
     * @param isPopupTrigger
     *                   boolean specifying whether this event will show a popup.
     * @param sleepTime
     *                   The wait time in ms between each event.
     * @param referencePoint     The CUSTOM mouse position within the cell.
     */
    public JTextComponentMouseEventData(final JFCTestCase testCase,
        final JTextComponent comp, final int numberOfClicks,
        final int modifiers, final boolean isPopupTrigger,
        final long sleepTime, final Point referencePoint) {
        this(testCase, comp, numberOfClicks, modifiers, isPopupTrigger,
            sleepTime, CUSTOM, referencePoint, INVALID_TEXT_OFFSET);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread
     *                  <code>awtSleep()</code> has to be invoked.
     * @param comp      The component on which to trigger the event.
     * @param numberOfClicks
     *                  Number of clicks in the MouseEvent (1 for single-click,
     *                  2 for double clicks)
     * @param modifiers The modifier key values that need to be passed
     *                  onto the event.
     * @param isPopupTrigger
     *                  boolean specifying whether this event will show a popup.
     * @param sleepTime The wait time in ms between each event.
     * @param position  The relative mouse position within the cell.
     * @param referencePoint
     *                   If position is CUSTOM then the point is a offset from
     *                   the location of the component. If the position is PERCENT
     *                   then the location is a percentage offset of the hight and width.
     *                   Otherwise, the referencePoint is unused.
     * @param offset    The character offset into the text component.
     */
    public JTextComponentMouseEventData(final JFCTestCase testCase,
        final JTextComponent comp, final int numberOfClicks,
        final int modifiers, final boolean isPopupTrigger,
        final long sleepTime, final int position, final Point referencePoint,
        final int offset) {
        setTestCase(testCase);
        setSource(comp);
        setNumberOfClicks(numberOfClicks);
        setModifiers(modifiers);
        setPopupTrigger(isPopupTrigger);
        setSleepTime(sleepTime);
        setPosition(position);
        setReferencePoint(referencePoint);
        setOffset(offset);
        setValid(true);
    }

    /**
     * Set the offset.
     *
     * @param offset   The new value of the offset
     */
    public final void setOffset(final int offset) {
        m_offset = offset;
    }

    /**
     * Get the offset.
     *
     * @return int    The value of the offset
     */
    public final int getOffset() {
        return m_offset;
    }

    /**
     * Set the attribute value.
     *
     * @param comp   The new value of the attribute
     */
    public final void setSource(final JTextComponent comp) {
        m_comp = comp;
    }

    /**
     * Get the attribute value.
     *
     * @return Component    The value of the attribute
     */
    public final JTextComponent getSource() {
        return m_comp;
    }

    /**
     * The component on which the event has to be fired.
     *
     * @return The component
     */
    public Component getComponent() {
        // by default, the component is the same as the source
        return getSource();
    }

    /**
     * Check if this event can consume the event given.
     * @param ae AWTEvent to be consumed.
     * @return true if the event may be consumed.
     */
    public boolean canConsume(final AWTEvent ae) {
        if ((ae.getSource() instanceof JTextComponent)
                && super.canConsume(ae)
                && sameSource(ae)) {
            return true;
        }

        return false;
    }

    /**
     * Consume the event.
     * @param ae AWTEvent to be consumed.
     * @return boolean true if the event was consumed.
     */
    public boolean consume(final AWTEvent ae) {
        if (super.consume(ae)) {
            return true;
        }

        MouseEvent     me     = (MouseEvent) ae;
        JTextComponent source = (JTextComponent) me.getSource();
        setSource(source);
        setModifiers(me.getModifiers());
        setNumberOfClicks(me.getClickCount());
        setPopupTrigger(me.isPopupTrigger());

        Point p = new Point(
                me.getX(),
                me.getY());
        Point screen = source.getLocationOnScreen();
        screen.translate(p.x, p.y);
        setLocationOnScreen(screen);

        int offset = ((JTextComponent) source).viewToModel(p);
        setOffset(offset);

        setPosition(CENTER);
        setReferencePoint(null);
        setSleepTime(getDefaultSleepTime());

        setValid(true);

        return true;
    }

    /**
     * Compare to event datas and deterime if they are equal.
     *
     * @param o Object to be compared.
     * @return true if the events are the same.
     */
    public boolean equals(final Object o) {
        if (!super.equals(o)) {
            return false;
        }

        JTextComponentMouseEventData data = (JTextComponentMouseEventData) o;

        return (data.getOffset() == getOffset());
    }

    /**
     * Get the hashCode.
     * @return int hashCode.
     */
    public int hashCode() {
        return super.hashCode() + m_offset;
    }

    /**
     * Populate the XML Element with this objects attributes.
     * @param e Element to be populated.
     */
    public void populate(final org.w3c.dom.Element e) {
        super.populate(e);
        e.setAttribute(JFCXMLConstants.TYPE, "JTextComponentMouseEventData");
        e.setAttribute(JFCXMLConstants.INDEX, "" + getOffset());
    }

    /**
     * Prepare the component to receive the event.
     *
     * @return true if the component is ready to receive the event.
     */
    public boolean prepareComponent() {
        if (!isValidForProcessing(getSource())) {
            return false;
        }

        Point p = null;
        if (getTestCase() != null) {
            getTestCase().flushAWT();
            getTestCase().pauseAWT();
        }

        if (getPosition() == OFFSET) {
            setPosition(EAST);

            if ((m_offset == -1)
                    || (m_offset > m_comp.getDocument().getLength())) {
                m_offset = m_comp.getDocument().getLength();
            }

            Rectangle rect = null;

            try {
                rect = m_comp.modelToView(m_offset);
            } catch (BadLocationException ex) {
                throw new RuntimeException("Invalid Offset:" + m_offset + ":"
                    + ex);
            }

            p = calculatePoint(rect);
        } else {
            p = calculatePoint(m_comp.getBounds());
        }

        Point screen = m_comp.getLocationOnScreen();
        screen.translate(p.x, p.y);
        setLocationOnScreen(screen);

        return true;
    }

    /**
     * Get string description of event.
     * @return String description of event.
     */
    public String toString() {
        if (!isValid()) {
            return super.toString();
        }

        StringBuffer buf = new StringBuffer(1000);
        buf.append(super.toString());
        buf.append(" offset: " + getOffset());

        return buf.toString();
    }
}
