package junit.extensions.jfcunit.eventdata;

import junit.extensions.jfcunit.JFCTestCase;
import junit.extensions.jfcunit.xml.JFCXMLConstants;

import junit.extensions.xml.IXMLTestCase;
import junit.extensions.xml.XMLException;
import junit.extensions.xml.elements.AbstractTagHandler;

import org.w3c.dom.Element;

import java.awt.Component;
import java.awt.event.InputEvent;
import java.awt.event.KeyEvent;

import java.util.StringTokenizer;


/**
 * This class will handle the processing of &lt;key&gt; nodes.
 *
 * <h3>Description</h3>
 * <p>
 *   This element will be used to submit key codes and strings
 *   to a component.
 * </p>
 *
 * <h3>Parameters</h3>
 * <table border="1" cellpadding="2" cellspacing="0">
 *   <tr>
 *     <td valign="top"><b>Attribute</b></td>
 *     <td valign="top"><b>Description</b></td>
 *     <td align="center" valign="top"><b>Required</b></td>
 *     <td valign="top"><b>Default</b></td>
 *     <td valign="top"><b>Values</b></td>
 *   </tr>
 *   <tr>
 *     <td valign="top">refid</td>
 *     <td valign="top">Id of a object reference which has been previously found.</td>
 *     <td valign="top" align="center">Yes</td>
 *     <td valign="top">N/A</td>
 *     <td valign="top">N/A</td>
 *   </tr>
 *   <tr>
 *     <td valign="top">string</td>
 *     <td valign="top">The string to be entered.</td>
 *     <td valign="top" align="center">One string or code attribute is required</td>
 *     <td valign="top">empty string</td>
 *     <td valign="top">Any characters</td>
 *   </tr>
 *   <tr>
 *     <td valign="top">code</td>
 *     <td valign="top">The key code to be entered.</td>
 *     <td valign="top" align="center">One string or code attribute is required</td>
 *     <td valign="top">None</td>
 *     <td valign="top">The key codes to use should follow the VK_(codes) of KeyEventData. Ex: VK_F4</td>
 *   </tr>
 *   <tr>
 *     <td valign="top">modifiers</td>
 *     <td valign="top">The modifiers to be used for this event.</td>
 *     <td valign="top" align="center">No</td>
 *     <td valign="top">None</td>
 *     <td valign="top">Alt Altgr Ctrl Meta Shift</td>
 *   </tr>
 *   <tr>
 *     <td valign="top">sleeptime</td>
 *     <td valign="top">Maximum sleep time</td>
 *     <td valign="top" align="center">No</td>
 *     <td valign="top">500</td>
 *     <td valign="top">Positive Integer representing the milli-seconds</td>
 *   </tr>
 *
 * </table>
 * <h3>Example</h3>
 * <blockquote><pre>
 * &lt;key
 *    refid=&quot;FileMenuItem&quot;
 *    code=&quot;VK_F2&quot;
 *    modifiers=&quot;Shift&quot;
 * /&gt;
 * </pre></blockquote>
 * <blockquote><pre>
 * &lt;key
 *    refid=&quot;FileMenuItem&quot;
 *    code=&quot;VK_F2&quot;
 *    modifiers=&quot;Ctrl+Shift&quot;
 * /&gt;
 * </pre></blockquote>
 * <p>
 * The above simulates pressing shift F2.
 * </p>
 * @see junit.extensions.jfcunit.eventdata.KeyEventData
 * @author Kevin Wilson
 * @author <a href="mailto:vraravam@thoughtworks.com">Vijay Aravamudhan : ThoughtWorks Inc.</a>
 */
public class KeyTagHandler extends AbstractTagHandler implements JFCXMLConstants {
    /**
     * Constructor for FindTagHandler.
     *
     * @param element     The element to be processed
     * @param testCase    The IXMLTestCase that uses this element
     */
    public KeyTagHandler(final Element element, final IXMLTestCase testCase) {
        super(element, testCase);
    }

    /**
     * Submit the keystrokes specified by the event.
     * @throws XMLException may be thrown.
     */
    public void processElement() throws XMLException {
        validateElement();

        String    refid = getRefId();
        Component comp = (Component) getXMLTestCase().getProperty(refid);

        if (comp == null) {
            throw new XMLException("Component not found for:" + refid, null,
                getElement(),
                getXMLTestCase().getPropertyCache());
        }

        int                  modifiers = getModifiers();
        long                 sleepTime = getSleepTime();

        String               string = getString(STRING);
        AbstractKeyEventData data   = null;

        if (string != null) {
            data = new StringEventData((JFCTestCase) getTestCase(), comp,
                    string, modifiers, sleepTime);
        } else {
            int character = getCharacter();
            data = new KeyEventData((JFCTestCase) getTestCase(), comp,
                    character, modifiers, sleepTime);
        }

        ((JFCTestCase) getTestCase()).getHelper().sendString(data);
    }

    /**
     * Validate that the refid attribute is specified and
     * either code or string attributes are specified.
     * @throws XMLException if the refid is missing or (code and string)
     * attributes are not present.
     */
    public void validateElement() throws XMLException {
        // do the default validations from the super class
        super.validateElement();

        // reqd attribute: refid
        checkRequiredAttribute(REFID);

        // Only one of the following is required.
        checkOneRequiredAttribute(
            getElement(),
            new String[] {CODE, STRING});
    }

    /**
     * Returns the value of the MODIFIERS attribute for this element. Defaults to
     * defaultValue.
     * @param defaultValue value to be returned if the element does not exist.
     * @return int  The value of the MODIFIERS attribute, defaultValue if not specified.
     */
    protected int getModifiers(final int defaultValue) {
        int    modifiers = 0;
        String s = getString(MODIFIERS);

        if (s == null) {
            return defaultValue;
        }

        try {
            // If we can parse a integer from the string then use the integer
            // value.
            return Integer.parseInt(s);
        } catch (NumberFormatException nfe) {
            // Ignore
        }

        StringTokenizer tok = new StringTokenizer(s, "+");

        while (tok.hasMoreElements()) {
            String token = tok.nextToken();

            if (token.equalsIgnoreCase(SHIFT)) {
                modifiers += InputEvent.SHIFT_MASK;
            }

            if (token.equalsIgnoreCase(CTRL)) {
                modifiers += InputEvent.CTRL_MASK;
            }

            if (token.equalsIgnoreCase(ALT)) {
                modifiers += InputEvent.ALT_MASK;
            }

            if (token.equalsIgnoreCase(ALTGR)) {
                modifiers += InputEvent.ALT_GRAPH_MASK;
            }

            if (token.equalsIgnoreCase(META)) {
                modifiers += InputEvent.META_MASK;
            }
        }

        return modifiers;
    }

    /**
     * Returns the value of the SLEEPTIME attribute for this element. Defaults to DEFAULT_SLEEPTIME.
     * @return String  The value of the SLEEPTIME attribute, DEFAULT_SLEEPTIME if not specified.
     */
    protected long getSleepTime() {
        return getLong(SLEEPTIME, DEFAULT_SLEEPTIME);
    }

    /**
     * Returns the value of the CODE attribute for this element.
     * @return String  The value of the CODE attribute.
     */
    private int getCharacter() {
        String keyCodeStr = getString(CODE);

        try {
            Object code = KeyEvent.class.getField(keyCodeStr).get(null);

            return ((Integer) code).intValue();
        } catch (Exception e) {
            // Ignore
        }

        return Integer.parseInt(keyCodeStr);
    }

    /**
     * Returns the value of the MODIFIERS attribute for this element. Defaults to zero.
     * @return int  The value of the MODIFIERS attribute, zero if not specified.
     */
    private int getModifiers() {
        return getModifiers(0);
    }

    /**
     * Returns the value of the REFID attribute for this element.
     * @return String  The value of the REFID attribute.
     */
    private String getRefId() {
        return getString(REFID);
    }
}
