package junit.extensions.jfcunit.eventdata;

import junit.extensions.jfcunit.JFCTestCase;

import java.awt.AWTEvent;
import java.awt.Component;
import java.awt.Container;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.event.MouseEvent;

import java.lang.reflect.Method;

import javax.swing.JViewport;


/**
 * Data container class that holds all the data necessary for jfcUnit to fire mouse events.
 *
 * @author <a href="mailto:vraravam@thoughtworks.com">Vijay Aravamudhan : ThoughtWorks Inc.</a>
 */
public class MouseWheelEventData extends MouseEventData {
    /**
     * The amount to scroll by default.
     */
    public static final int DEFAULT_MOUSE_CLICKS = 0;

    /**
     * MouseWheelEvent event id.
     */
    private static int s_mouseWheelEvent;

    /**
     * Class of the MouseWheelEvent.
     * Used so that the code is backward compatible.
     */
    private static Class s_mweClass;

    /**
     * Method for getting the rotation of the mouse from the
     * event. Used so that the code is backward compatible.
     */
    private static Method s_getWheelRotationMethod;

    /**
     * Method for getting the scroll amount from the
     * event. Used so that the code is backward compatible.
     */
    private static Method s_getScrollAmountMethod;

    static {
        try {
            s_mweClass                   = Class.forName(
                    "java.awt.event.MouseWheelEvent");
            s_mouseWheelEvent            = s_mweClass.getField("MOUSE_WHEEL")
                                                     .getInt(null);
            s_getWheelRotationMethod     = s_mweClass.getMethod(
                    "getWheelRotation",
                    new Class[0]);
            s_getScrollAmountMethod = s_mweClass.getMethod(
                    "getScrollAmount",
                    new Class[0]);
        } catch (Exception e) {
            s_mouseWheelEvent = 0;
        }
    }

    /**
     * The Component on which to trigger the event.
     */
    private Component m_comp;

    /**
     * Amount to scroll for each wheel rotation.
     */
    private int m_scrollAmount = 0;

    /**
     * Number of wheel clicks to rotate.
     */
    private int m_wheelRotation = 0;

    /**
     * Default Constructor.
     */
    public MouseWheelEventData() {
        super();
    }

    /**
     * Constructor.
     *
     * @param testCase The JFCTestCase on whose thread <code>awtSleep()</code>
     *                 has to be invoked.
     * @param comp     The component on which to trigger the event.
     */
    public MouseWheelEventData(final JFCTestCase testCase,
                               final Component comp) {
        this(testCase, comp, DEFAULT_SCROLL_AMOUNT, DEFAULT_WHEEL_ROTATION);
    }

    /**
     * Constructor.
     *
     * @param testCase The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param comp     The component on which to trigger the event.
     * @param wheelRotation The amount to rotate the wheel positive
     * for values towards the user. Negative for away from the user.
     */
    public MouseWheelEventData(final JFCTestCase testCase,
        final Component comp, final int wheelRotation) {
        this(testCase, comp, DEFAULT_SCROLL_AMOUNT, wheelRotation,
            DEFAULT_MOUSE_CLICKS);
    }

    /**
     * Constructor.
     *
     * @param testCase The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param comp     The component on which to trigger the event.
     * @param amount   The amount to scroll for each rotation.
     * @param wheelRotation The amount to rotate the wheel positive
     * for values towards the user. Negative for away from the user.
     */
    public MouseWheelEventData(final JFCTestCase testCase,
        final Component comp, final int amount, final int wheelRotation) {
        this(testCase, comp, amount, wheelRotation, DEFAULT_MOUSE_CLICKS);
    }

    /**
     * Constructor.
     *
     * @param testCase The JFCTestCase on whose thread <code>awtSleep()</code>
     *                 has to be invoked.
     * @param comp     The component on which to trigger the event.
     * @param amount   The amount to scroll for each rotation.
     * @param wheelRotation The amount to rotate the wheel positive
     * for values towards the user. Negative for away from the user.
     * @param numberOfClicks
     *                 Number of clicks in the MouseEvent (1 for single-click,
     *                 2 for double clicks)
     */
    public MouseWheelEventData(final JFCTestCase testCase,
        final Component comp, final int amount, final int wheelRotation,
        final int numberOfClicks) {
        this(testCase, comp, amount, wheelRotation, numberOfClicks,
            DEFAULT_MOUSE_MODIFIERS);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code>
     *                  has to be invoked.
     * @param comp      The component on which to trigger the event.
     * @param amount   The amount to scroll for each rotation.
     * @param wheelRotation The amount to rotate the wheel positive
     * for values towards the user. Negative for away from the user.
     * @param numberOfClicks
     *                  Number of clicks in the MouseEvent (1 for single-click,
     *                  2 for double clicks)
     * @param modifiers The modifier key values that need to be passed onto the
     *                  event.
     */
    public MouseWheelEventData(final JFCTestCase testCase,
        final Component comp, final int amount, final int wheelRotation,
        final int numberOfClicks, final int modifiers) {
        this(testCase, comp, amount, wheelRotation, numberOfClicks, modifiers,
            DEFAULT_ISPOPUPTRIGGER);
    }

    /**
     * Constructor.
     *
     * @param testCase The JFCTestCase on whose thread <code>awtSleep()</code>
     *                 has to be invoked.
     * @param comp     The component on which to trigger the event.
     * @param amount   The amount to scroll for each rotation.
     * @param wheelRotation The amount to rotate the wheel positive
     * for values towards the user. Negative for away from the user.
     * @param numberOfClicks
     *                 Number of clicks in the MouseEvent (1 for single-click,
     *                 2 for double clicks)
     * @param isPopupTrigger
     *                 boolean specifying whether this event will show a popup.
     */
    public MouseWheelEventData(final JFCTestCase testCase,
        final Component comp, final int amount, final int wheelRotation,
        final int numberOfClicks, final boolean isPopupTrigger) {
        this(testCase, comp, amount, wheelRotation, numberOfClicks,
            getDefaultModifiers(isPopupTrigger), isPopupTrigger);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code>
     *                  has to be invoked.
     * @param comp      The component on which to trigger the event.
     * @param amount   The amount to scroll for each rotation.
     * @param wheelRotation The amount to rotate the wheel positive
     * for values towards the user. Negative for away from the user.
     * @param numberOfClicks
     *                  Number of clicks in the MouseEvent (1 for single-click,
     *                  2 for double clicks)
     * @param sleepTime
     *                  The wait time in ms between each event.
     */
    public MouseWheelEventData(final JFCTestCase testCase,
        final Component comp, final int amount, final int wheelRotation,
        final int numberOfClicks, final long sleepTime) {
        this(testCase, comp, amount, wheelRotation, numberOfClicks,
            DEFAULT_MOUSE_MODIFIERS, DEFAULT_ISPOPUPTRIGGER, sleepTime);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code>
     *                  has to be invoked.
     * @param comp      The component on which to trigger the event.
     * @param amount   The amount to scroll for each rotation.
     * @param wheelRotation The amount to rotate the wheel positive
     * for values towards the user. Negative for away from the user.
     * @param numberOfClicks
     *                  Number of clicks in the MouseEvent (1 for single-click,
     *                  2 for double clicks)
     * @param modifiers The modifier key values that need to be passed onto the
     *                  event.
     * @param isPopupTrigger
     *                  boolean specifying whether this event will show a popup.
     */
    public MouseWheelEventData(final JFCTestCase testCase,
        final Component comp, final int amount, final int wheelRotation,
        final int numberOfClicks, final int modifiers,
        final boolean isPopupTrigger) {
        this(testCase, comp, amount, wheelRotation, numberOfClicks, modifiers,
            isPopupTrigger, DEFAULT_SLEEPTIME);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code>
     *                  has to be invoked.
     * @param comp      The component on which to trigger the event.
     * @param amount   The amount to scroll for each rotation.
     * @param wheelRotation The amount to rotate the wheel positive
     * for values towards the user. Negative for away from the user.
     * @param numberOfClicks
     *                  Number of clicks in the MouseEvent (1 for single-click,
     *                  2 for double clicks)
     * @param isPopupTrigger
     *                  boolean specifying whether this event will show a popup.
     * @param sleepTime
     *                  The wait time in ms between each event.
     */
    public MouseWheelEventData(final JFCTestCase testCase,
        final Component comp, final int amount, final int wheelRotation,
        final int numberOfClicks, final boolean isPopupTrigger,
        final long sleepTime) {
        this(testCase, comp, amount, wheelRotation, numberOfClicks,
            getDefaultModifiers(isPopupTrigger), isPopupTrigger, sleepTime);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code>
     *                  has to be invoked.
     * @param comp      The component on which to trigger the event.
     * @param amount   The amount to scroll for each rotation.
     * @param wheelRotation The amount to rotate the wheel positive
     * for values towards the user. Negative for away from the user.
     * @param numberOfClicks
     *                   Number of clicks in the MouseEvent (1 for single-click,
     *                   2 for double clicks)
     * @param modifiers The modifier key values that need to be passed onto the event.
     * @param isPopupTrigger
     *                   boolean specifying whether this event will show a popup.
     * @param sleepTime
     *                   The wait time in ms between each event.
     */
    public MouseWheelEventData(final JFCTestCase testCase,
        final Component comp, final int amount, final int wheelRotation,
        final int numberOfClicks, final int modifiers,
        final boolean isPopupTrigger, final long sleepTime) {
        this(testCase, comp, amount, wheelRotation, numberOfClicks, modifiers,
            isPopupTrigger, sleepTime, DEFAULT_POSITION, null);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param comp      The component on which to trigger the event.
     * @param amount   The amount to scroll for each rotation.
     * @param wheelRotation The amount to rotate the wheel positive
     * for values towards the user. Negative for away from the user.
     * @param numberOfClicks
     *                   Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     * @param modifiers The modifier key values that need to be passed onto the event.
     * @param isPopupTrigger
     *                   boolean specifying whether this event will show a popup.
     * @param sleepTime
     *                   The wait time in ms between each event.
     * @param position  The relative mouse position within the cell.
     */
    public MouseWheelEventData(final JFCTestCase testCase,
        final Component comp, final int amount, final int wheelRotation,
        final int numberOfClicks, final int modifiers,
        final boolean isPopupTrigger, final long sleepTime, final int position) {
        this(testCase, comp, amount, wheelRotation, numberOfClicks, modifiers,
            isPopupTrigger, sleepTime, position, null);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param comp      The component on which to trigger the event.
     * @param amount   The amount to scroll for each rotation.
     * @param wheelRotation The amount to rotate the wheel positive
     * for values towards the user. Negative for away from the user.
     * @param numberOfClicks
     *                   Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     * @param modifiers The modifier key values that need to be passed onto the event.
     * @param isPopupTrigger
     *                   boolean specifying whether this event will show a popup.
     * @param sleepTime
     *                   The wait time in ms between each event.
     * @param referencePoint     The CUSTOM mouse position within the cell.
     */
    public MouseWheelEventData(final JFCTestCase testCase,
        final Component comp, final int amount, final int wheelRotation,
        final int numberOfClicks, final int modifiers,
        final boolean isPopupTrigger, final long sleepTime,
        final Point referencePoint) {
        this(testCase, comp, amount, wheelRotation, numberOfClicks, modifiers,
            isPopupTrigger, sleepTime, CUSTOM, referencePoint);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code>
     *                  has to be invoked.
     * @param comp      The component on which to trigger the event.
     * @param amount    The amount to scroll for each rotation.
     * @param wheelRotation The amount to rotate the wheel positive
     * for values towards the user. Negative for away from the user.
     * @param numberOfClicks
     *                  Number of clicks in the MouseEvent (1 for single-click,
     *                  2 for double clicks)
     * @param modifiers The modifier key values that need to be passed
     *                  onto the event.
     * @param isPopupTrigger
     *                   boolean specifying whether this event will show a popup.
     * @param sleepTime
     *                   The wait time in ms between each event.
     * @param position  The relative mouse position within the cell.
     * @param referencePoint
     *                   If position is CUSTOM then the point is a offset from
     *                   the location of the component. If the position is PERCENT
     *                   then the location is a percentage offset of the hight and width.
     *                   Otherwise, the referencePoint is unused.
     */
    public MouseWheelEventData(final JFCTestCase testCase,
        final Component comp, final int amount, final int wheelRotation,
        final int numberOfClicks, final int modifiers,
        final boolean isPopupTrigger, final long sleepTime, final int position,
        final Point referencePoint) {
        setTestCase(testCase);
        setSource(comp);
        setScrollAmount(amount);
        setWheelRotation(wheelRotation);
        setNumberOfClicks(numberOfClicks);
        setModifiers(modifiers);
        setPopupTrigger(isPopupTrigger);
        setSleepTime(sleepTime);
        setPosition(position);
        setReferencePoint(referencePoint);
        setValid(true);
    }

    /**
     * Set the attribute value.
     *
     * @param comp   The new value of the attribute.
     */
    public final void setSource(final Component comp) {
        m_comp = comp;
    }

    /**
     * Get the attribute value.
     *
     * @return Component    The value of the attribute.
     */
    public final Component getSource() {
        return m_comp;
    }

    /**
     * The component on which the event has to be fired.
     *
     * @return The component.
     */
    public Component getComponent() {
        // by default, the component is the same as the source
        return getSource();
    }

    /**
     * Set the scroll amount.
     *
     * @param scrollAmount Amount to scroll for each wheel click.
     */
    public final void setScrollAmount(final int scrollAmount) {
        this.m_scrollAmount = scrollAmount;
    }

    /**
     * Get the scroll amount.
     *
     * @return int scroll amount.
     */
    public final int getScrollAmount() {
        return m_scrollAmount;
    }

    /**
     * Set the wheel rotation.
     *
     * @param wheelRotation amount to rotate the wheel.
     */
    public final void setWheelRotation(final int wheelRotation) {
        this.m_wheelRotation = wheelRotation;
    }

    /**
     * Get the wheel rotation.
     * @return int number of wheel rotations.
     */
    public final int getWheelRotation() {
        return m_wheelRotation;
    }

    /**
     * Returns true if the event can be consumed by
     * this instnace of event data.
     *
     * @param ae Event to be consumed.
     * @return true if the event was consumed.
     */
    public boolean canConsume(final AWTEvent ae) {
        // Cannot consume if the event ID cannot be accessed
        // or the event is not a mouse wheel event.
        if ((s_mouseWheelEvent == 0) || (ae.getID() != s_mouseWheelEvent)) {
            return false;
        }

        if (isValid()) {
            int modifiers = ((MouseEvent) ae).getModifiers();

            if (ae.getSource() != getSource()) {
                return false;
            }

            if (modifiers != getModifiers()) {
                return false;
            }

            try {
                int wheelRotation = ((Integer) s_getWheelRotationMethod.invoke(
                        ae,
                        new Object[0])).intValue();

                // Change of direction if result is negative.
                if ((wheelRotation * getWheelRotation()) < 0) {
                    return false;
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }

        return true;
    }

    /**
     * Consume the event.
     *
     * @param ae AWTEvent to be consumed.
     * @return boolean true if the event was consumed.
     */
    public boolean consume(final AWTEvent ae) {
        MouseEvent me = (MouseEvent) ae;

        try {
            int amount = ((Integer) s_getScrollAmountMethod.invoke(
                    ae,
                    new Object[0])).intValue();
            setScrollAmount(amount);

            int wheelRotation = ((Integer) s_getWheelRotationMethod.invoke(
                    ae,
                    new Object[0])).intValue();
            setWheelRotation(wheelRotation + getWheelRotation());
        } catch (Exception e) {
            e.printStackTrace();
        }

        // If the event is valid then we do not need to
        // reprocess the rest of the event.
        if (isValid()) {
            return true;
        }

        Component source = (Component) me.getSource();
        setSource(source);
        setModifiers(me.getModifiers());
        setNumberOfClicks(me.getClickCount());
        setPopupTrigger(me.isPopupTrigger());

        Point p = new Point(
                me.getX(),
                me.getY());
        setSleepTime(getDefaultSleepTime());

        Point screen = source.getLocationOnScreen();
        screen.translate(p.x, p.y);
        setLocationOnScreen(screen);

        setPosition(CENTER);
        setReferencePoint(null);
        setValid(true);

        return true;
    }

    /**
     * Compare to event data and deterime if they are equal.
     *
     * @param o Object to be compared.
     * @return true if the events are the same.
     */
    public boolean equals(final Object o) {
        if (!super.equals(o)) {
            return false;
        }

        MouseWheelEventData data = (MouseWheelEventData) o;

        return (data.getWheelRotation() == getWheelRotation())
            && (data.getScrollAmount() == getScrollAmount());
    }

    /**
     * Get the hashCode for this object.
     * @return int hashCode.
     */
    public int hashCode() {
        return super.hashCode();
    }

    /**
     * Prepare the component for firing the mouse event.
     *
     * @return boolean true if the component is ready.
     */
    public boolean prepareComponent() {
        if (!isValidForProcessing(getSource())) {
            return false;
        }

        // Locate the ViewPort
        Container parent = null;

        if (getSource() instanceof Container) {
            parent = (Container) getSource();
        } else {
            parent = getSource().getParent();
        }

        while ((parent.getParent() != null) && !(parent instanceof JViewport)) {
            parent = parent.getParent();
        }

        if (!(parent instanceof JViewport)) {
            return false;
        }

        JFCTestCase testCase = getTestCase();

        if (testCase != null) {
            testCase.awtSleep(getSleepTime());
        }

        Rectangle bounds   = parent.getBounds();
        Point     location = parent.getLocationOnScreen();
        bounds.x += location.x;
        bounds.y += location.y;
        setLocationOnScreen(calculatePoint(bounds));

        return true;
    }

    /**
     * Generate a string representation of the event.
     *
     * @return String form of the event data.
     */
    public String toString() {
        StringBuffer buf = new StringBuffer(1000);
        buf.append(super.toString());
        buf.append(" amount:" + getScrollAmount());
        buf.append(" rotation:" + getWheelRotation());

        return buf.toString();
    }
}
