package junit.extensions.jfcunit.eventdata;

import junit.extensions.xml.IXMLTestCase;
import junit.extensions.xml.XMLException;

import org.w3c.dom.Element;


/**
 * This element implements the mouse wheel movement.
 *
 * <h3>Description</h3>
 * <p>
 *   Invokes the junit.extensions.jfcunit.TestHelper.mouseWheel() method
 *   with the parameters given by this element.
 * </p>
 * <p>
 *   To use the wheel task, set the <i>refid</i> attribute to the name of the
 *   component which has been found.
 * </p>
 *
 * <h3>Parameters</h3>
 * <table border="1" cellpadding="2" cellspacing="0">
 *   <tr>
 *     <td valign="top"><b>Attribute</b></td>
 *     <td valign="top"><b>Description</b></td>
 *     <td align="center" valign="top"><b>Required</b></td>
 *     <td valign="top"><b>Default</b></td>
 *   </tr>
 *   <tr>
 *     <td valign="top">refid</td>
 *     <td valign="top">Id of a object reference which has been previously found.</td>
 *     <td valign="top" align="center">Yes</td>
 *     <td valign="top">N/A</td>
 *   </tr>
 *   <tr>
 *     <td valign="top">amount</td>
 *     <td valign="top">The number of units that should be scrolled in response to this event.</td>
 *     <td valign="top" align="center">No</td>
 *     <td valign="top">3</td>
 *   </tr>
 *   <tr>
 *     <td valign="top">rotation</td>
 *     <td valign="top">The number of "clicks" the mouse wheel was rotated.</td>
 *     <td valign="top" align="center">No</td>
 *     <td valign="top">1</td>
 *   </tr>
 *
 * </table>
 * <h3>Example</h3>
 * <blockquote><pre>
 * &lt;wheel
 *    refid=&quot;ScrollPane&quot;
 *    amount=&quot;3&quot;
 *    rotation=&quot;50&quot;
 * /&gt;
 * </pre></blockquote>
 * @see junit.extensions.jfcunit.eventdata.MouseWheelEventData
 * @author Kevin Wilson
 */
public class MouseWheelEventDataTagHandler extends BaseEventDataTagHandler {
    /**
     * Constructor for MouseWheelEventDataTagHandler.
     *
     * @param element     The element to be processed
     * @param testCase    The IXMLTestCase that uses this element
     */
    public MouseWheelEventDataTagHandler(final Element element,
        final IXMLTestCase testCase) {
        super(element, testCase);
    }

    /**
     * @see junit.extensions.xml.elements.AbstractTagHandler#processElement()
     */
    public void processElement() throws XMLException {
        validateElement();

        MouseWheelEventData event = new MouseWheelEventData(
                getJFCTestCase(),
                getComponent(),
                getAmount(),
                getRotation(),
                getClicks(),
                getModifiers(),
                getPopupTrigger(),
                getSleepTime(),
                getPosition(),
                getReference());
        getJFCTestCase().getHelper().enterMouseWheel(event);
    }

    /**
     * @see junit.extensions.xml.elements.AbstractTagHandler#validateElement()
     */
    public void validateElement() throws XMLException {
        // do the default validations from the super class
        super.validateElement();
    }

    /**
     * Get the number of clicks. Default for number of clicks
     * is zero for mouse wheel events.
     *
     * @return int number of clicks.
     */
    protected int getClicks() {
        return getInt(CLICKS, 0);
    }

    /**
     * Get the modifiers for the mouse wheel event.
     * Default modifiers is Zero.
     * @return int modifiers.
     * @see java.awt.event.MouseEvent for modifiers.
     */
    protected int getModifiers() {
        // Default clicks for this method should be zero.
        return getModifiers(0);
    }

    /**
     * Get the amount from the XML.
     * @return int amount to be scrolled.
     */
    private int getAmount() {
        return getInt(AMOUNT, DEFAULT_SCROLL_AMOUNT);
    }

    /**
     * Get the rotation of the wheel from the XML.
     * @return int rotation of the mouse wheel.
     */
    private int getRotation() {
        return getInt(ROTATION, DEFAULT_WHEEL_ROTATION);
    }
}
