package junit.extensions.jfcunit.finder;

import java.awt.Component;

import javax.swing.AbstractButton;
import javax.swing.Icon;


/**
 * Class for checking if the ({@link javax.swing.AbstractButton}) component being searched for has been found.
 * The pattern syntax can be found at the Jakarta RegExp API Documentation in {@link org.apache.regexp.RE}.
 * This class delegates the matching of the Icon to an instance of {@link junit.extensions.jfcunit.finder.IconMatcher}
 *
 * @author <a href="mailto:vraravam@thoughtworks.com">Vijay Aravamudhan : ThoughtWorks Inc.</a>
 */
public class AbstractButtonFinder extends Finder {
    /**
     * The matcher for the icon of the {@link javax.swing.AbstractButton} component.
     */
    private IconMatcher m_iconMatcher;

    /**
     * The text of the {@link javax.swing.AbstractButton} component.
     */
    private String m_label;

    /**
     * A boolean specifying whether the filtration is case insensitive.
     */
    private boolean m_caseIndependent = false;

    /**
     * Constructor accepting all arguments needed to filter the component.
     *
     * @param text    The desired pattern for the text of the component.
     */
    public AbstractButtonFinder(final String text) {
        this(text, null, false);
    }

    /**
     * Constructor accepting all arguments needed to filter the component.
     *
     * @param text               The desired pattern for the text of the component.
     * @param caseIndependent    Whether the match should be case independent (true) or not (false).
     */
    public AbstractButtonFinder(final String text, final boolean caseIndependent) {
        this(text, null, caseIndependent);
    }

    /**
     * Constructor accepting all arguments needed to filter the component.
     *
     * @param icon   The desired pattern for the icon of the component.
     */
    public AbstractButtonFinder(final Icon icon) {
        this(null, icon, false);
    }

    /**
     * Constructor accepting all arguments needed to filter the component.
     *
     * @param str    The desired pattern for the text of the component.
     * @param icon   The desired pattern for the icon of the component.
     */
    public AbstractButtonFinder(final String str, final Icon icon) {
        this(str, icon, false);
    }

    /**
     * Constructor accepting all arguments needed to filter the component.
     *
     * @param str                The desired pattern for the text of the component.
     * @param icon               The desired pattern for the icon of the component.
     * @param caseIndependent    Whether the match should be case independent (true) or not (false)
     */
    public AbstractButtonFinder(final String str, final Icon icon,
        final boolean caseIndependent) {
        setText(str);
        setIcon(icon);

        this.m_caseIndependent = caseIndependent;
        recreatePatternMatcher(m_label, caseIndependent);
    }

    /**
     * Set the text of the button to be found.
     * @param text Text of the button.
     */
    public final void setText(final String text) {
        m_label = text;
        recreatePatternMatcher(m_label, m_caseIndependent);
    }

    /**
     * Set the finder into a case independent mode.
     * @param ignoreCase true if case should be ignored.
     */
    public void setCaseIndependent(final boolean ignoreCase) {
        super.setCaseIndependent(ignoreCase);
        m_caseIndependent = ignoreCase;
        recreatePatternMatcher(m_label, m_caseIndependent);
    }


    /**
     * Get the text of the label to be matched.
     * @return String text of the label.
     */
    public final String getText() {
        return m_label;
    }

    /**
     * Set the icon to be matched.
     * @param icon Icon to be matched.
     */
    public final void setIcon(final Icon icon) {
        try {
            m_iconMatcher = new IconMatcher(icon);
        } catch (InterruptedException ex) {
            m_iconMatcher = null;
        }
    }

    /**
     * Get the icon to be matched.
     * @return Iocn to be matched.
     */
    public final Icon getIcon() {
        return m_iconMatcher.getIcon();
    }

    /**
     * Method that returns true if the given component matches the search
     * criteria.
     *
     * @param comp   The component to test.
     * @return true if this component is a match.
     */
    public boolean testComponent(final Component comp) {
        return ((comp != null)
        && isValidForProcessing(comp, AbstractButton.class)
        && ((m_label == null)
        || evaluate(
            ((AbstractButton) comp).getText(),
            m_label))
        && m_iconMatcher.matches(((AbstractButton) comp).getIcon()));
    }
}
