package junit.extensions.jfcunit.finder;

import junit.extensions.jfcunit.TestHelper;
import junit.extensions.jfcunit.tools.Operator;
import junit.extensions.jfcunit.xml.JFCXMLConstants;

import junit.extensions.xml.IXMLTestCase;
import junit.extensions.xml.XMLException;
import junit.extensions.xml.elements.AbstractTagHandler;

import org.w3c.dom.Element;

import java.awt.Component;
import java.awt.Container;
import java.awt.Window;

import javax.swing.Icon;
import javax.swing.ImageIcon;
import javax.swing.JMenu;


/**
 * This class provides the basic handling for all of the FindTagHandler
 * implementations.
 *
 * <H3>Optional Attributes:</H3>
 * <pre>
 * operation - match, equals, startswith, endswith, contains
 * show      - show the candidate items by moving the cursor
 *             to each item passing the finder. A Message dialog
 *             will show the current index.
 * </pre>
 * @author <a href="mailto:vraravam@thoughtworks.com">Vijay Aravamudhan : ThoughtWorks Inc.</a>
 * @author Kevin Wilson
 */
public abstract class BaseFindTagHandler extends AbstractTagHandler
    implements JFCXMLConstants {
    /**
     * Constructor for FindTagHandler.
     *
     * @param element     The element to be processed.
     * @param testCase    The IXMLTestCase that uses this element.
     */
    public BaseFindTagHandler(final Element element, final IXMLTestCase testCase) {
        super(element, testCase);
    }

    /**
     * @see junit.extensions.xml.elements.AbstractTagHandler#validateElement()
     */
    public abstract void processElement() throws XMLException;

    /**
     * @see junit.extensions.xml.elements.AbstractTagHandler#validateElement()
     */
    public void validateElement() throws XMLException {
        // do the default validations from the super class
        super.validateElement();

        // check the element tag name
        checkElementTagName(FIND);

        // reqd attribute: id
        checkRequiredAttribute(ID);

        // reqd attribute: index
        checkRequiredAttribute(INDEX);
    }

    /**
     * Returns the value of the CASEINDEPENDENT attribute for this element.
     * @return boolean  The value of the CASEINDEPENDENT attribute.
     */
    protected final boolean getCaseIndependent() {
        return getBoolean(CASEINDEPENDENT);
    }

    /**
     * Returns the value of the CLASS attribute for this element.
     * @return String  The value of the CLASS attribute.
     */
    protected final String getClassName() {
        return getString(CLASS);
    }

    /**
     * Returns the value of the CONTAINER attribute for this element.
     * @return String  The value of the CONTAINER attribute.
     */
    protected final String getContainerId() {
        return getString(CONTAINER);
    }

    /**
     * Returns the Icon described by the ICONFILE and ICONDESCRIPTION attributes for this element.
     * @return Icon  The ImageIcon described by the value of the ICONFILE and ICONDESCRIPTION
     * attributes.
     */
    protected final Icon getIcon() {
        Icon   icon = null;
        String file = getString(ICONFILE);
        String desc = getString(ICONDESCRIPTION);

        if ((file != null) && !file.equals("")) {
            icon = new ImageIcon(file, desc);
        }

        return icon;
    }

    /**
     * Returns the value of the ID attribute for this element.
     * @return String  The value of the ID attribute.
     */
    protected final String getId() {
        return getString(ID);
    }

    /**
     * Returns the value of the INDEX attribute for this element. Defaults to 0 (zero) if not found.
     * @return int  The value of the parsed INDEX attribute, zero if not found.
     */
    protected final int getIndex() {
        return getInt(INDEX, 0);
    }

    /**
     * Returns the value of the LABEL attribute for this element.
     * @return String  The value of the LABEL attribute.
     */
    protected final String getLabel() {
        return getString(LABEL);
    }

    /**
     * Returns the value of the NAME attribute for this element.
     * @return String  The value of the NAME attribute.
     */
    protected final String getName() {
        return getString(NAME);
    }

    /**
     * Get the operation as a string and
     * translate to a OP code.
     *
     * @return Finder.OP_code
     */
    protected final int getOperation() {
        String op = super.getString(OPERATION);

        if ((op == null) || (op.length() == 0)) {
            return Finder.OP_MATCH;
        }

        return Operator.Operation.getOperation(op);
    }

    /**
     * Get the value of the show attribute.
     * @return true if the finder is to show the
     * candidate items.
     */
    protected final boolean getShow() {
        return getBoolean(SHOW, false);
    }

    /**
     * Retrun the value of the TITLE attribute for this element.
     * @return String The value of the TITLE attribute.
     */
    protected final String getTitle() {
        return getString(TITLE);
    }

    /**
     * Get the Wait time for the finder.
     * @return Duration for the Finder to try again until the
     * object is found.
     */
    protected int getWait() {
        return super.getInt(WAIT, 5);
    }

    /**
     * Execute the common find on the TestHelper.
     * @param finder Finder to be used to locate the object.
     */
    protected void find(final Finder finder) {
        int index = getIndex();

        finder.setWait(getWait());
        finder.setShowDebug(getShow());

        String containerRefId = getContainerId();
        finder.setOperation(getOperation());

        Component comp;

        if (containerRefId != null) {
            Container cont = (Container) getXMLTestCase().getProperty(containerRefId);

            if (cont instanceof JMenu) {
                cont = ((JMenu) cont).getPopupMenu();
            }

            comp = finder.find(cont, index);
        } else {
            comp = finder.find(index);
        }

        String id = getId();
        getXMLTestCase().addProperty(id, comp);
    }

    /**
     * Execute the common findWindow on the TestHelper.
     * @param wfinder Finder to be used to locate the object.
     */
    protected void findWindow(final AbstractWindowFinder wfinder) {
        Window w = null;
        wfinder.setWait(getWait());
        wfinder.setOperation(getOperation());
        w = TestHelper.getWindow(wfinder);

        String id = getId();
        getXMLTestCase().addProperty(id, w);
    }
}
