package junit.extensions.jfcunit.finder;

import java.awt.Component;

import javax.swing.JLabel;


/**
 * Class for checking if the component being searched for has been found
 * The pattern syntax can be found at the Jakarta RegExp API Documentation in {@link org.apache.regexp.RE}.
 *
 * @author <a href="mailto:vraravam@thoughtworks.com">Vijay Aravamudhan : ThoughtWorks Inc.</a>
 */
public class JLabelFinder extends Finder {
    /**
     * The text of the component.
     */
    private String m_text;

    /**
     * A boolean specifying whether the filtration is case insensitive.
     */
    private boolean m_caseIndependent = false;

    /**
     * Constructor accepting all arguments needed to filter component.
     *
     * @param text   The desired pattern for the text of the component.
     */
    public JLabelFinder(final String text) {
        this(text, false);
    }

    /**
     * Constructor accepting all arguments needed to filter component.
     *
     * @param text               The desired pattern for the text of the component.
     * @param caseIndependent    Whether the match should be case independent
     * (true) or not (false)
     */
    public JLabelFinder(final String text, final boolean caseIndependent) {
        m_text = text;
        this.m_caseIndependent = caseIndependent;
        createPatternMatcher(m_text, caseIndependent);
    }


    /**
     * Set the finder into a case independent mode.
     * @param ignoreCase true if case should be ignored.
     */
    public void setCaseIndependent(final boolean ignoreCase) {
        super.setCaseIndependent(ignoreCase);
        m_caseIndependent = ignoreCase;
        createPatternMatcher(m_text, m_caseIndependent);
    }

    /**
     * Set the label text to be matched.
     * @param text Text to be matched.
     */
    public final void setText(final String text) {
        m_text = text;
        createPatternMatcher(m_text, m_caseIndependent);
    }

    /**
     * Get the label text to be matched.
     * @return label text to be matched.
     */
    public final String getText() {
        return m_text;
    }

    /**
     * Method that returns true if the given component matches the search
     * criteria.
     *
     * @param comp   The component to test
     * @return true if this component is a match
     */
    public boolean testComponent(final Component comp) {
        return ((comp != null) && isValidForProcessing(comp, JLabel.class)
        && evaluate(
            ((JLabel) comp).getText(),
            m_text));
    }
}
