package junit.extensions.jfcunit.xml;

import junit.extensions.jfcunit.JFCTestCase;

import junit.extensions.xml.IXMLTestCase;
import junit.extensions.xml.XMLException;
import junit.extensions.xml.elements.AbstractTagHandler;

import org.w3c.dom.Element;


/**
 * This class controls the AWTEventQueue. The queue may be
 * paused, resumed or flushed.
 *
 * <H3>Tag Name</H3>
 * awteventqueue
 *
 * <H3>Required Attributes</H3>
 * <pre>
 * action - pause|resume|flush
 *          pause is used to suspend the AWT Event queue
 *                to allow for the processing of GUI components.
 *          resume action should be used after a pause. To
 *                allow the AWTEventQueue to continue processing.
 *          flush action will insure that each of the current
 *                events on the event queue are processed.
 *
 * </pre>
 * <H3>Examples</H3>
 * Pause the queue.
 * <pre>
 * &lt;awteventqueue action="pause"/&gt;
 * </pre>
 * Resume the queue.
 * <pre>
 * &lt;awteventqueue action="resume"/&gt;
 * </pre>
 * Flush the queue.
 * <pre>
 * &lt;awteventqueue action="flush"/&gt;
 * </pre>
 *
 * @author <a href="mailto:kwilson227@users.sourceforge.net">Kevin L Wilson</a>
 */
public class AWTEventQueueTagHandler extends AbstractTagHandler
    implements JFCXMLConstants {
    /**
     * Constructor for AWTThreadTagHandler.
     *
     * @param element     The element to be processed
     * @param testCase    The IXMLTestCase that uses this element
     */
    public AWTEventQueueTagHandler(final Element element,
        final IXMLTestCase testCase) {
        super(element, testCase);
    }

    /**
     * @see junit.extensions.xml.elements.AbstractTagHandler#processElement()
     */
    public void processElement() throws XMLException {
        validateElement();

        String action = getString(ACTION);

        if (FLUSH.equals(action)) {
            ((JFCTestCase) getTestCase()).flushAWT();
        } else if (PAUSE.equals(action)) {
            ((JFCTestCase) getTestCase()).pauseAWT();
        } else if (RESUME.equals(action)) {
            ((JFCTestCase) getTestCase()).resumeAWT();
        } else if (SLEEP.equals(action)) {
            long sleep = getSleepTime();

            if (sleep > 0) {
                ((JFCTestCase) getTestCase()).sleep(sleep);
            } else {
                for (;;) {
                    ((JFCTestCase) getTestCase()).sleep(10000);
                }
            }
        }
    }

    /**
     * @see junit.extensions.xml.elements.AbstractTagHandler#validateElement()
     */
    public void validateElement() throws XMLException {
        // do the default validations from the super class
        super.validateElement();

        // check the element tag name
        checkElementTagName(AWTTHREAD);

        // reqd attribute: action
        checkRequiredAttribute(ACTION);

        if (SLEEP.equals(getString(ACTION))) {
            checkRequiredAttribute(DURATION);
        }
    }

    /**
     * Returns the value of the SLEEPTIME attribute for this element, DEFAULT_SLEEPTIME if nothing
     * was specified.
     * @return long    The sleepTime for this element.
     */
    protected long getSleepTime() {
        if ("forever".equals(getString(DURATION))) {
            return -1;
        }

        return getLong(DURATION, DEFAULT_SLEEPTIME);
    }
}
