package junit.extensions.jfcunit.xml;

import junit.extensions.jfcunit.JFCTestCase;

import junit.extensions.xml.IXMLTestCase;
import junit.extensions.xml.XMLException;
import junit.extensions.xml.elements.AbstractTagHandler;

import org.w3c.dom.Element;


/**
 * This class will handle causes the test case to sleep
 * and allows the AWTEventQueue to run for the specified
 * number of milliseconds.
 *
 * <H3>Tag Name</H3>
 * sleep
 * <H3>Required Attribute</H3>
 * duration - number of milliseconds to sleep or
 *            "forever" to sleep forever.
 * <H3>Examples</H3>
 * Sleep for 1 second.
 * <pre>
 * &lt;sleep duration="1000"/&gt
 * </pre>
 * Sleep forever
 * <pre>
 * &lt;sleep duration="forever"/&gt
 * </pre>
 * @author <a href="mailto:klwilson227@users.sourceforge.net">Kevin L Wilson</a>
 */
public class SleepTagHandler extends AbstractTagHandler
    implements JFCXMLConstants {
    /**
     * Constructor for AWTThreadTagHandler.
     *
     * @param element     The element to be processed
     * @param testCase    The IXMLTestCase that uses this element
     */
    public SleepTagHandler(final Element element, final IXMLTestCase testCase) {
        super(element, testCase);
    }

    /**
     * @see junit.extensions.xml.elements.AbstractTagHandler#processElement()
     */
    public void processElement() throws XMLException {
        validateElement();

        long sleep = getSleepTime();

        if (sleep >= 0) {
            ((JFCTestCase) getTestCase()).sleep(sleep);
        } else {
            for (;;) {
                ((JFCTestCase) getTestCase()).sleep(10000);
            }
        }
    }

    /**
     * @see junit.extensions.xml.elements.AbstractTagHandler#validateElement()
     */
    public void validateElement() throws XMLException {
        // do the default validations from the super class
        super.validateElement();

        checkRequiredAttribute(DURATION);
    }

    /**
     * Returns the value of the SLEEPTIME attribute for this element, DEFAULT_SLEEPTIME if nothing
     * was specified.
     * @return long    The sleepTime for this element.
     */
    protected long getSleepTime() {
        if ("forever".equals(getString(DURATION))) {
            return -1;
        }

        return getLong(DURATION, DEFAULT_SLEEPTIME);
    }
}
