package junit.extensions.xml;


/**
 * <p>Title: XMLObjectCache</p>
 * <p>Description: Multi level hashmap to contain
 * name value pairs. A parent level can be explicitly
 * accessed by prefixing the name with "../"
 * otherwise the parent level will only be searched
 * if the object does not exist at the current level. </p>
 * <p>Copyright: Copyright (c) 2003</p>
 * <p>Company: jfcunit project</p>
 * @author Kevin Wilson
 * @version 1.0
 */
public class XMLPropertyCache extends XMLObjectCache {
    /**
     * Empty contructor. Parent is assumed to be null. It can be
     * set at a later time.
     */
    public XMLPropertyCache() {
        this(null);
    }

    /**
     * Constructor.
     * @param parent Parent object cache to be set.
     */
    public XMLPropertyCache(final XMLPropertyCache parent) {
        super(parent);
    }

    /**
     * Get a object from the cache. If the object does not
     * exist in the cache then try the parent cache. If a parent
     * cache does not exist then try the System properties.
     * @param name Name of the value to be retrieved.
     * @return Return the value.
     */
    public Object get(final String name) {
        Object o = super.get(name);

        if (o == null) {
            o = System.getProperty(name);
        }

        return o;
    }

    /**
     * Resolve any parameter names in the string given, and
     * make the appropriate substitutions.
     * @param s String String which may contain the property names.
     * @return String Resuting string which will have the property values
     * substituted.
     */
    public final String resolve(final String s) {
        if (s == null) {
            return null;
        }

        int    index = -1;
        String str = new String(s);

        do {
            index = str.lastIndexOf("${");

            if (index != -1) {
                int    lindex = str.indexOf("}", index);

                String variableName = str.substring(index + 2, lindex);

                if (variableName.length() == 0) {
                    throw new RuntimeException(
                        "Variable name expected in ${} cannot resolve :" + s);
                }

                Object value = get(variableName);

                if (value == null) {
                    System.err.println("Evaluation of \"" + s
                        + "\" Variable not found assuming empty string for:"
                        + variableName);
                    value = "";
                }

                String valueString = value.toString();
                str = str.substring(0, index) + valueString
                    + str.substring(lindex + 1);
            }
        } while (index != -1);

        return str;
    }
}
