package junit.extensions.xml;

import junit.framework.Test;
import junit.framework.TestSuite;

import org.w3c.dom.Element;
import org.w3c.dom.NodeList;

import java.io.InputStream;


/**
 * Test Case for running XML Script based testing.
 *
 * @author Kevin Wilson
 */
public class XMLTestSuite extends TestSuite implements IXMLTestSuite,
    XMLConstants {
    /**
     * Element for this test suite.
     */
    private Element m_element;

    /**
     * Parent Test Suite.
     */
    private IXMLTest m_parent = null;

    /**
     * The name of the test xml script file - found from the classpath.
     */
    private String m_fileName;

    /**
     * Collection of valid procedures for the test case.
     */
    private final XMLObjectCache m_procedures = new XMLObjectCache();

    /**
     * Object cahce used to hold globals set by the test case.
     */
    private XMLPropertyCache m_propertyCache = new XMLPropertyCache();

    /**
     * A no-argument constructor to support junit 3.8.
     */
    public XMLTestSuite() {
    }

    /**
     * A constructor accepting the name of the test script file.
     *
     * @param  fileName    The name of the test script file (found from the classpath)
     */
    public XMLTestSuite(final String fileName) {
        this(fileName,
            XMLUtil.parse(XMLUtil.readFileFromClasspath(fileName))
                   .getDocumentElement());
    }

    /**
     * A constructor accepting the name of the test script file.
     *
     * @param  name      A dummy name for the file
     * @param  stream    The contents of the test script file
     */
    public XMLTestSuite(final String name, final InputStream stream) {
        this(name,
            XMLUtil.parse(stream).getDocumentElement());
    }

    /**
     * A constructor accepting the name of the test script file and the element to be processed.
     *
     * @param  fileName    The name of the test script file (found from the classpath)
     * @param  element     The element to be processed.
     */
    public XMLTestSuite(final String fileName, final Element element) {
        super(fileName + ":" + XMLUtil.getPath(element));
        this.m_fileName     = fileName;
        this.m_element      = element;
        processChildren(m_element);
    }

    /**
     * Get the debug state.
     * @return true if debug is enabled.
     */
    public boolean getDebug() {
        Object str = getProperty(DEBUG);

        if (str instanceof String) {
            return Boolean.valueOf((String) str).booleanValue();
        } else if (str instanceof Boolean) {
            return ((Boolean) str).booleanValue();
        }

        return (m_propertyCache.get(DEBUG) != null);
    }

    /**
     * Set the parent of this test case.
     * @param parent Parent test.
     */
    public final void setParent(final IXMLTest parent) {
        this.m_parent = parent;
        m_propertyCache.setParent(parent.getPropertyCache());
        m_procedures.setParent(parent.getProcedureCache());
    }

    /**
     * Returns the filename.
     *
     * @return String   The name of the test xml script file
     */
    public String getFileName() {
        return m_fileName;
    }

    /**
     * Get the procedure by the given name.
     * @param name The name of the procedure to
     * be returned.
     * @return XMLProcedure matching the name
     * or null if it does not exist.
     */
    public IXMLProcedure getProcedure(final String name) {
        return (IXMLProcedure) m_procedures.get(name);
    }

    /**
     * Get the procedure cache.
     * @return procedure cache for this test case.
     */
    public XMLObjectCache getProcedureCache() {
        return m_procedures;
    }

    /**
     * Get the value of a property.
     * @param name Name of the property to be returned.
     * @return Value of the property.
     */
    public Object getProperty(final String name) {
        return getPropertyCache().get(name);
    }

    /**
     * Get the found object cache.
     * @return Object cache for this test case.
     */
    public XMLObjectCache getPropertyCache() {
        return m_propertyCache;
    }

    /**
     * Reverse lookup the property name from the value of the property.
     * @param value Value to be looked up.
     * @return First property with the same value.
     */
    public String getPropertyName(final Object value) {
        return getPropertyCache().getName(value);
    }

    /**
     * Get the property names.
     * @return Get a array of property names
     */
    public String[] getPropertyNames() {
        return getPropertyCache().getNames();
    }

    /**
     * Add the given XML file to the TestSuite.
     *
     * @param  xmlFileName    The name of the test script file to be added.
     */
    public void addFile(final String xmlFileName) {
        XMLTestSuite suite = new XMLTestSuite(xmlFileName);
        suite.setParent(this);
        addTest(suite);
    }

    /**
     * Add a procedure to the test suite.
     * @param proc Procedure to be added.
     */
    public void addProcedure(final IXMLProcedure proc) {
        if (proc == null) {
            throw new IllegalArgumentException("procedure must be specified");
        }

        m_procedures.put(
            proc.getName(),
            proc);
    }

    /**
     * Adds a found object to the cache.
     * @param name Name of the object.
     * @param value Value of the object.
     */
    public void addProperty(final String name, final Object value) {
        getPropertyCache().put(name, value);
    }

    /**
     * Add a test to the test suite.
     * @param test Test to be added.
     */
    public void addTest(final Test test) {
        if (test instanceof IXMLTest) {
            ((IXMLTest) test).setParent(this);
        }

        super.addTest(test);
    }

    /**
     * Clear the properties.
     */
    public void clearProperties() {
        getPropertyCache().clear();
    }

    /**
     * Remove a property.
     * @param name Name of the property to be removed.
     */
    public void removeProperty(final String name) {
        getPropertyCache().remove(name);
    }

    /**
     * Resolve embeded property names withing a string.
     * @param s String to be resolved.
     * @return String without property names.
     */
    public String resolveProperties(final String s) {
        return m_propertyCache.resolve(s);
    }

    /**
     * Process the children of the test suite.
     * @param e Element which contains the children to be processed.
     */
    private void processChildren(final Element e) {
        // Get the children and add to the suite.
        NodeList children = e.getChildNodes();
        Element  child;

        for (int i = 0; i < children.getLength(); i++) {
            if (children.item(i) instanceof Element) {
                child = (Element) children.item(i);

                String name = child.getTagName();

                try {
                    XMLTagResourceBundle.getTagHandler(child, this, name)
                                        .processElement();
                } catch (XMLException xe) {
                    throw xe;
                } catch (Throwable t) {
                    throw new XMLException(
                        t.getMessage(),
                        t,
                        child,
                        getPropertyCache());
                }
            }
        }
    }
}
