package junit.extensions.xml.elements;

import junit.extensions.xml.IXMLTestCase;
import junit.extensions.xml.XMLException;

import org.w3c.dom.Element;


/**
 * This tag handler provides the basics for the
 * assert tag handers to be derived.  The methods
 * provide a consistent way to validate and read
 * the attributes of the commands.
 * @author Kevin Wilson
 */
public abstract class AbstractAssertTagHandler extends AbstractTagHandler {
    /**
     * Constructor for AssertNotNullTagHandler.
     *
     * @param element     The element to be processed
     * @param testCase    The IXMLTestCase that uses this element
     */
    public AbstractAssertTagHandler(final Element element,
        final IXMLTestCase testCase) {
        super(element, testCase);
    }

    /**
     * Returns the value of the ACTUALOBJ attribute for this element.
     * If the element contains an actual ref id attribute and that object
     * was found previously, then that object is returned.
     * @return String  The value of the ACTUALOBJ attribute. (If an actual ref id
     *                 object was found and was not null, then that value is returned.
     */
    protected Object getActualObject() {
        if (getActualRefId() != null) {
            return getXMLTestCase().getProperty(getActualRefId());
        } else {
            return getString(ACTUALOBJ);
        }
    }

    /**
     * Returns the value of the EXPECTEDOBJ attribute for this element.
     * If the element contains an expected ref id attribute and that object
     * was found previously, then that object is returned.
         * @return String  The value of the EXPECTEDOBJ attribute. (If an expected ref id
     *                 object was found and was not null, then that value is returned.
     */
    protected Object getExpectedObject() {
        if (getExpectedRefId() != null) {
            return getXMLTestCase().getProperty(getExpectedRefId());
        } else {
            return getString(EXPECTEDOBJ);
        }
    }

    /**
     * Returns the value of the MESSAGE attribute for this element.
     * @return String  The value of the MESSAGE attribute.
     */
    protected String getMessage() {
        return getString(MESSAGE);
    }

    /**
     * Returns the value of the REFID attribute for this element.
     * @return String  The value of the REFID attribute.
     */
    protected String getRefId() {
        return getString(REFID);
    }

    /**
     * Check that either a actualobj or actualrefid is specified.
     * @throws XMLException thrown if one of the above is not present.
     * Or if both are present.
     */
    protected void checkActual() throws XMLException {
        super.checkAtLeastOneRequiredAttribute(
            new String[] {ACTUALOBJ, ACTUALREFID});
    }

    /**
     * Check that either a expectedobj or expectedrefid is specified.
     * @throws XMLException thrown if one of the above is not present.
     * Or if both are present.
     */
    protected void checkExpected() throws XMLException {
        super.checkAtLeastOneRequiredAttribute(
            new String[] {EXPECTEDOBJ, EXPECTEDREFID});
    }

    /**
     * Returns the value of the ACTUALREFID attribute for this element.
     * @return String  The value of the ACTUALREFID attribute.
     */
    private String getActualRefId() {
        return getString(ACTUALREFID);
    }

    /**
     * Returns the value of the EXPECTEDREFID attribute for this element.
     * @return String  The value of the EXPECTEDREFID attribute.
     */
    private String getExpectedRefId() {
        return getString(EXPECTEDREFID);
    }
}
