/*
 * Created on Oct 27, 2003
 *
 * To change the template for this generated file go to
 * Window>Preferences>Java>Code Generation>Code and Comments
 */
package junit.extensions.xml.elements;

import junit.extensions.xml.IXMLTestCase;
import junit.extensions.xml.XMLException;

import junit.framework.Assert;

import org.apache.regexp.RE;
import org.apache.regexp.RESyntaxException;

import org.w3c.dom.Element;

import javax.swing.JTable;


/**
 * Provide a mechanism for validating the contents of a table.
 *
 * <H3>Tag Name:</H3>
 * asserttablecontains
 * <H3>Attributes:</H3>
 * <pre>
 * id      - the table to inspect                            [required]
 * value   - the expected value                              [required]
 * row     - the row to inspect                              [required]
 * col     - the column to inspect (int index or column name)[required]
 * useRE   - allow the value to contain a regular expression [optional]
 *           default - is false exact string match needed
 *           If set to true partial strings will work.
 * </pre>
 * <H3>Examples:</H3>
 * &lt;assertTableContains id="component1" value="Joe"/&gt;
 * @author Kevin Wilson
 */
public class AssertTableContainsTagHandler extends AbstractTagHandler {
    /**
     * Constructor.
     * @param element Element to be processed.
     * @param testcase TestCase containing the element.
     */
    public AssertTableContainsTagHandler(final Element element,
        final IXMLTestCase testcase) {
        super(element, testcase);
    }

    /**
     * Get the column index from the attibute column.
     * The attribute value may be a integer index of the
     * column. Or the name of the column according to the
     * table model.
     * @param table JTable containing the data.
     * @return index of the column.
     */
    public int getColumn(final JTable table) {
        String col = getString(COLUMN);

        try {
            int i = Integer.parseInt(col);

            return i;
        } catch (NumberFormatException e) {
            return getColumnIndex(col, table);
        }
    }

    /**
     * Handle the XML processing of the tag 'assertTableContains'.
     * @throws XMLException when assert fails.
     */
    public void processElement() throws XMLException {
        validateElement();

        String id = getString(ID);

        Object o = ((IXMLTestCase) getTestCase()).getProperty(id);

        if (o == null) {
            Assert.fail("Error: Unable to locate object id:" + id);
        } else {
            if (o instanceof JTable) {
                String expectedval = getString(VALUE);
                JTable table = (JTable) o;
                int    row   = getInt(ROW, -1);
                Assert.assertTrue(row != -1);
                Assert.assertTrue(row < table.getRowCount());

                int col = getColumn(table);
                Assert.assertTrue(col != -1);
                Assert.assertTrue(row < table.getRowCount());

                String tableval = table.getValueAt(row, col).toString();

                String msg = "table contains[" + row + "," + col
                    + "] Component: " + id + " Expected: " + expectedval
                    + " Retrieved: " + tableval;

                if (getXMLTestCase().getDebug()) {
                    System.out.println("msg: " + msg);
                }

                boolean useRegExp = getBoolean(USERE);

                if (useRegExp) {
                    RE regexp = null;

                    try {
                        regexp = new RE(expectedval);
                    } catch (RESyntaxException ex) {
                        throw new XMLException(
                            "Could not create regular expression:"
                            + expectedval,
                            ex,
                            getElement(),
                            getXMLTestCase().getPropertyCache());
                    }

                    Assert.assertTrue(
                        msg,
                        regexp.match(tableval));
                } else {
                    if (!tableval.equals(expectedval)) {
                        System.out.println("Assertion failed: " + msg);
                    }

                    Assert.assertEquals(expectedval, tableval);
                }
            }
        }
    }

    /**
     * Insure that the required attributes are specified.
     * @throws XMLException if a required attribute is missing.
     */
    public void validateElement() throws XMLException {
        super.validateElement();

        // id is a required attribute
        checkRequiredAttribute(ID);

        checkRequiredAttribute(ROW);
        checkRequiredAttribute(COLUMN);

        // value is a required attribute
        checkRequiredAttribute(VALUE);
    }

    /**
     * Get the column index matching the name.
     * @param columnName Name of the column.
     * @param  table JTable Table to search.
     * @return int index of the column.
     */
    private int getColumnIndex(final String columnName, final JTable table) {
        int count = table.getColumnCount();

        for (int index = 0; index < count; index++) {
            if (columnName.equals(table.getColumnName(index))) {
                return index;
            }
        }

        return -1;
    }
}
