package junit.extensions.jfcunit;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import java.awt.Component;
import java.awt.Container;
import java.awt.Frame;
import java.awt.Point;
import java.awt.Window;
import java.awt.event.InputEvent;
import java.awt.event.KeyEvent;
import java.awt.event.WindowEvent;
import javax.swing.JDialog;
import javax.swing.JFileChooser;
import javax.swing.JLabel;

import org.apache.regexp.RE;
import org.apache.regexp.RESyntaxException;
import junit.extensions.jfcunit.eventdata.AbstractKeyEventData;
import junit.extensions.jfcunit.eventdata.AbstractMouseEventData;
import junit.extensions.jfcunit.eventdata.DragEventData;
import junit.extensions.jfcunit.eventdata.KeyEventData;
import junit.extensions.jfcunit.eventdata.MouseWheelEventData;
import junit.extensions.jfcunit.finder.AbstractWindowFinder;
import junit.extensions.jfcunit.finder.ComponentFinder;
import junit.extensions.jfcunit.finder.DialogFinder;
import junit.extensions.jfcunit.finder.Finder;
import junit.extensions.jfcunit.finder.FrameFinder;
import junit.extensions.jfcunit.finder.NamedComponentFinder;
import junit.extensions.jfcunit.keyboard.DefaultKeyMapping;
import junit.extensions.jfcunit.keyboard.JFCKeyStroke;
import junit.extensions.jfcunit.keyboard.KeyMapping;


/**
 * A helper class that provides facilities for locating and posting events to components
 * within a GUI. To use, create a new instance of TestHelper in your setUp.
 * Caveat: Windows can only be located once they have been shown once to the user.
 *
 * @author Matt Caswell
 * @author <a href="mailto:vraravam@thoughtworks.com">Vijay Aravamudhan : ThoughtWorks Inc.</a>
 */
public abstract class TestHelper {
    /**
     * The length of time that the test class will sleep for between each
     * event being posted.
     */
    public static final long DEFAULTSLEEP = 300L;

    /**
     * Current test.
     */
    private static JFCTestCase s_test;

    /**
     * KeyMapping to be used with the helper.
     */
    private static KeyMapping s_keyMapping = null;

    /**
     * System Windows List. These windows should not be closed
     * when the cleanUp is called. The vector should contain
     * the Regular expression strings for matching the name to the
     * title.
     */
    private static HashMap s_systemWindows = new HashMap();

    /**
     * Lock for setting the key mapping.
     */
    private static final Object LOCK = new Object();

    /**
     * Step size for mouse movement events.
     */
    private int m_step = 10;

    /**
     * Constructor.
     */
    protected TestHelper() {
    }

    /**
     * Get all of the windows we know about including heavy weight
     * popups.
     *
     * @return Window[] All of the windows we know about.
     * @deprecated 2.02 use WindowMonitor.getWindows();
     */
    public static Window[] getAllWindows() {
        return WindowMonitor.getWindows();
    }

    /**
     * Set the current test case.
     * @param tc TestCase to set as the current test.
     */
    public static void setCurrentTestCase(final JFCTestCase tc) {
        s_test = tc;
    }

    /**
     * Get the current test case.
     * @return JFCTestCase which is the current test.
     */
    public static JFCTestCase getCurrentTestCase() {
        return s_test;
    }

    /**
     * Returns a unmodifiable list of
     * the system window regular expressions.
     *
     * @return unmodifiable Set containing the regular
     *         expressions for matching windows.
     */
    public static final Set getSystemWindows() {
        return Collections.unmodifiableSet(s_systemWindows.keySet());
    }

    /**
     * Add a regular expression to match the window titles
     * which should not be closed during cleanup.
     *
     * @param matchTitleRE Regular expression defining the windows
     *                     which should not be closed.
     * @throws RESyntaxException if the syntax is not a valid Regular expression.
     */
    public static final void addSystemWindow(final String matchTitleRE)
        throws RESyntaxException {
        if (matchTitleRE == null) {
            return;
        }

        RE patternMatcher = new RE(matchTitleRE);
        s_systemWindows.put(matchTitleRE, patternMatcher);
    }

    /**
     * Set the key mapping to be used.
     *
     * @param km KeyMapping to be used.
     */
    public static void setKeyMapping(final KeyMapping km) {
        synchronized (LOCK) {
            s_keyMapping = km;
            LOCK.notifyAll();
        }
    }

    /**
     * Get the key mapping to be used.
     *
     * @return the key mapping.
     */
    public static KeyMapping getKeyMapping() {
        synchronized (LOCK) {
            if (s_keyMapping == null) {
                setKeyMapping(new DefaultKeyMapping());
            }

            LOCK.notifyAll();
        }

        return s_keyMapping;
    }

    /**
     * Returns a set of all the Dialogs that are currently visible for all
     * opened, visible windows - including the special case of the shared, hidden
     * frame which is the owner of JDialogs whose owner is not specified.
     *
     * @return The full list of dialogs.
     * @deprecated 2.05 Use DialogFinder(null).findAll()
     */
    public static List getShowingDialogs() {
        return getShowingDialogs((String) null);
    }

    /**
     * Helper method to obtain a list of the dialogs owned by a given window.
     *
     * @param win    The window to find dialogs for.
     * @return The list of dialogs owned by the given window (including the window if it is a Dialog itself).
     * @deprecated 2.05 Use new DialogFinder(null).findAll(win);
     */
    public static List getShowingDialogs(final Window win) {
        return getShowingDialogs(win, null);
    }

    /**
     * Returns a set of all the Dialogs that are currently visible and the title
     * contains the given titlematch string - including the special case of the
     * shared, hidden frame which is the owner of JDialogs whose owner is not specified.
     *
     * @param titlematch The string pattern to search for in the dialog's title.
     * @return The list of dialogs owned by the given window (including the window if it is a Dialog itself).
     * @deprecated 2.05 use new DialogFinder(titlematch).findAll();
     */
    public static List getShowingDialogs(final String titlematch) {
        List openWindows = getWindows();

        // We also have to add the shared, hidden frame used by JDialogs which did not
        // have an owner at the beginning
        openWindows.add((new JDialog()).getOwner());

        Window[] owners = new Window[openWindows.size() + 1];
        owners = (Window[]) openWindows.toArray(owners);

        return getShowingDialogs(owners, titlematch);
    }

    /**
     * Returns a set of all the Dialogs that are currently visible and the title
     * contains the given titlematch string.
     *
     * @param win        The window to find dialogs for.
     * @param titlematch The string pattern to search for in the dialog's title.
     * @return The list of dialogs owned by the given window (including the window if it is a Dialog itself).
     * @deprecated 2.05 use new DialogFinder(titleMatch).findAll(win);
     */
    public static List getShowingDialogs(final Window win,
        final String titlematch) {
        return getShowingDialogs(
            new Window[] {win},
            titlematch);
    }

    /**
     * Returns a set of all the Dialogs that are currently visible and the title
     * contains the given titlematch string - including the special case of the
     * shared, hidden frame which is the owner of JDialogs whose owner is not specified.
     *
     * @param owners     The array of windows to find dialogs for.
     * @param titlematch The string pattern to search for in the dialog's title.
     * @return The list of dialogs owned by the given window (including the window if it is a Dialog itself).
     * @deprecated 2.05 use new DialogFinder(titlematch).findAll(owners);
     */
    public static List getShowingDialogs(final Window[] owners,
        final String titlematch) {
        return getShowingDialogs(
            new ArrayList(),
            owners,
            titlematch);
    }

    /**
     * Returns a set of all the Dialogs that are currently visible and the title
     * contains the given titlematch string - including the special case of the
     * shared, hidden frame which is the owner of JDialogs whose owner is not specified.
     *
     * @param ret        The list of already filtered and accepted dialogs.
     * @param owners     The array of windows to find dialogs for.
     * @param titlematch The string pattern to search for in the dialog's title.
     * @return The list of dialogs owned by the given window (including the window if it is a Dialog itself).
     * @deprecated 2.05 Use new DialogFinder(titlematch).findAll(owners);
     */
    public static List getShowingDialogs(final List ret, final Window[] owners,
        final String titlematch) {
        return getShowingDialogs(
            ret,
            owners,
            new DialogFinder(titlematch));
    }

    /**
     * Returns a set of all the Dialogs that are currently visible and the title
     * contains the given titlematch string - including the special case of the
     * shared, hidden frame which is the owner of JDialogs whose owner is not specified.
     *
     * @param ret        The list of already filtered and accepted dialogs.
     * @param owners     The array of windows to find dialogs for.
     * @param finder     The DialogFinder which is used to filter using a title match
     * @return The list of dialogs owned by the given window (including the window if it is a Dialog itself).
     * @deprecated 2.05 use new DialogFinder(null).findAll(owners);
     */
    public static List getShowingDialogs(final List ret, final Window[] owners,
        final Finder finder) {
        List arr;

        if (ret == null) {
            arr = new ArrayList();
        } else {
            arr = ret;
        }

        if ((owners == null) || (finder == null)) {
            return arr;
        }

        for (int i = 0; i < owners.length; i++) {
            if (owners[i] == null) {
                continue;
            }

            if (finder.testComponent(owners[i])
                && !ret.contains(owners[i])) {
                arr.add(owners[i]);
            }

            getShowingDialogs(
                arr,
                owners[i].getOwnedWindows(),
                finder);
        }

        return arr;
    }

    /**
     * Returns a set of all the Windows that are currently visible based on the
     * Finder being passed in.
     *
     * @param finder  The AbstractWindowFinder which is used to filter
     * @return Set of Window objects.
     * @deprecated 2.05 use finder.findAll();
     */
    public static List getWindows(final AbstractWindowFinder finder) {
        return finder.findAll();
    }

    /**
     * Returns a set of all the Windows that are currently visible and the title
     * contains the given titlematch string.
     *
     * @param ret        The list of already filtered and accepted windows.
     * @param windows    The array of windows to filter and add.
     * @param titlematch The string pattern to search for in the window's title.
     * @return Set of Window objects.
     * @deprecated 2.05 use new FrameFinder(titlematch).findAll();
     */
    public static List getWindows(final List ret, final Window[] windows,
        final String titlematch) {
        return getWindows(
            ret,
            windows,
            new FrameFinder(titlematch));
    }

    /**
     * Returns a set of all the Windows that are currently visible and the title
     * contains the given titlematch string.
     *
     * @param ret     The list of already filtered and accepted windows.
     * @param windows The array of windows to filter and add.
     * @param finder  The FrameFinder which is used to filter using a title match
     * @return Set of Window objects.
     * @deprecated 2.05 use finder.findAll(windows);
     */
    public static List getWindows(final List ret, final Window[] windows,
        final Finder finder) {
        return AbstractWindowFinder.getWindows(ret, windows, finder);
    }

    /**
     * Calculate the next point along the path to the
     * destingation point.
     *
     * @param src Source point.
     * @param dest Dest point.
     * @param step Number of pixels to step.
     * @return src point.
     */
    public static Point calcNextPoint(final Point src, final Point dest,
        final int step) {
        Point dir = new Point(
                getDir(src.x, dest.x),
                getDir(src.y, dest.y));

        if (isBounded(src.x + (dir.x * step), dest.x, src.x)) {
            dir.x = (dir.x * step);
        }

        if (isBounded(src.y + (dir.y * step), dest.y, src.y)) {
            dir.y = dir.y * step;
        }

        dir.translate(src.x, src.y);

        return dir;
    }

    /**
     * This method is used to clean up any existing open windows, etc
     * if ase lettersa test fails midway. To be called in the tearDown() method.
     *
     * @param testCase The test case that is firing the event.
     * @see #addSystemWindow
     * @see #removeSystemWindow
     * @see #removeAllSystemWindows
     */
    public static void cleanUp(final JFCTestCase testCase) {
        cleanUp(testCase, DEFAULTSLEEP);
    }

    /**
     * This method is used to clean up any existing open windows, etc
     * if a test fails midway. To be called in the tearDown() method.
     *
     * @param testCase The test case that is firing the event.
     * @param awtSleepTime
     *                 The wait time in ms between each event.
     */
    public static void cleanUp(final JFCTestCase testCase,
        final long awtSleepTime) {
        Iterator iter = getWindows().iterator();

        while (iter.hasNext()) {
            boolean ignore = false;
            Window  w     = (Window) iter.next();
            String  title = null;

            if (w instanceof java.awt.Dialog) {
                title = ((java.awt.Dialog) w).getTitle();
            } else if (w instanceof Frame) {
                title = ((java.awt.Frame) w).getTitle();
            }

            if (title != null) {
                Iterator sw = s_systemWindows.values().iterator();

                while (sw.hasNext() && !ignore) {
                    RE re = (RE) sw.next();

                    if (re.match(title)) {
                        ignore = true;
                    }
                }
            }

            if (!ignore) {
                disposeWindow(w, testCase, awtSleepTime);
            }
        }
    }

    /**
     * This method does the actual work of destroying the window.
     *
     * @param window   The window that needs to be destroyed.
     * @param testCase The test case that is firing the event.
     */
    public static void disposeWindow(final Window window,
        final JFCTestCase testCase) {
        disposeWindow(window, testCase, DEFAULTSLEEP);
    }

    /**
     * This method does the actual work of destroying the window and any dialogs that it owns.
     *
     * @param window   The window that needs to be destroyed.
     * @param testCase The test case that is firing the event.
     * @param awtSleepTime
     *                 The wait time in ms between each event.
     */
    public static void disposeWindow(final Window window,
        final JFCTestCase testCase, final long awtSleepTime) {
        if (window != null) {
            // need to leave the AWTThread in the same state that is in at the end
            boolean awt = testCase.isAWTRunning();

            if (awt) {
                testCase.pauseAWT();
            }

            try {
                if (!(window instanceof JDialog)) {
                    // dispose of any showing dialogs first
                    List dialogs = getShowingDialogs(window);

                    for (int i = 0; i < dialogs.size(); i++) {
                        disposeWindow((JDialog) dialogs.get(i), testCase,
                            awtSleepTime);
                    }
                }

                window.setVisible(false);

                // try to clear the event queue
                testCase.flushAWT();

                if (!awt) {
                    testCase.pauseAWT();
                }

                try {
                    window.getToolkit().getSystemEventQueue().postEvent(
                        new WindowEvent(window, WindowEvent.WINDOW_CLOSING));
                } catch (ExitException exe) {
                    ; // don't do anything here
                }
            } finally {
                // somehow, this sleeptime needs to be set after the first flush
                testCase.setSleepTime(awtSleepTime);

                // try to clear the event queue
                testCase.flushAWT();

                if (!awt) {
                    testCase.pauseAWT();
                }

                testCase.awtSleep();
            }
        }
    }

    /**
     * Remove all System Window regular expressions.
     */
    public static final void removeAllSystemWindows() {
        s_systemWindows.clear();
    }

    /**
     * Remove a regular expression to match the window titles
     * which should not be closed during cleanup.
     *
     * @param matchTitleRE Regular expression defining the windows
     *                     which should not be closed.
     */
    public static final void removeSystemWindow(final String matchTitleRE) {
        s_systemWindows.remove(matchTitleRE);
    }

    /**
     * A helper method to find the message being displayed in a dialog box.
     *
     * @param dialog The dialog box displaying the message
     * @return The messagebeing displayed
     */
    public String getMessageFromJDialog(final JDialog dialog) {
        if ((dialog == null) || !dialog.isShowing()) {
            return null;
        }

        // since there are usually 2 JLabels, use a sufficiently high number ---> 10
        List list = Finder.findComponentList(
                new ComponentFinder(JLabel.class),
                dialog,
                new ArrayList(),
                10);
        JLabel label;

        for (int i = 0; i < list.size(); i++) {
            label = (JLabel) list.get(i);

            if ((label != null) && (label.getText() != null)) {
                return label.getText();
            }
        }

        return null;
    }

    /**
     * Returns a single window that is showing with given string in the title.
     * If there is more than one window that match, then it is undefined which
     * one will be returned.
     *
     * @param title  The string that should be present in the title.
     * @return A showing window with the given title pattern.
     * @deprecated 2.02 use new FrameFinder(title).getWindow
     */
    public static Window getWindow(final String title) {
        return getWindow(new FrameFinder(title));
    }

    /**
     * Returns a single window that is showing that passes the filtration by the Finder passed in.
     * If there is more than one window that match, then it is undefined which
     * one will be returned.
     *
     * @param finder  The AbstractWindowFinder which is used to filter
     * @return A showing window with the given title.
     */
    public static Window getWindow(final AbstractWindowFinder finder) {
        Iterator iter = getWindows(finder).iterator();

        if (!iter.hasNext()) {
            return null;
        }

        return (Window) iter.next();
    }

    /**
     * Returns a set of all the Windows that are currently visible.
     *
     * @return Set of Window objects.
     */
    public static List getWindows() {
        FrameFinder f = new FrameFinder(null);
        f.setWait(1);

        return getWindows(f);
    }

    /**
     * Returns a set of all the Windows that are currently visible and the title
     * contains the given titlematch string.
     *
     * @param titlematch The string pattern to search for in the window's title.
     * @return Set of Window objects.
     */
    public static List getWindows(final String titlematch) {
        return getWindows(new FrameFinder(titlematch));
    }

    /**
     * Finds a JFileChooser being shown by a window.
     *
     * @param win    The window to owning the chooser.
     * @return The chooser that has been found, or null if there are none.
     * @deprecated 2.05 use JFileChooserFinder(null).find(win, 0);
     */
    public JFileChooser getShowingJFileChooser(final Window win) {
        List choosers = getShowingJFileChoosers(win);

        if (choosers.isEmpty()) {
            return null;
        }

        return (JFileChooser) choosers.get(0);
    }

    /**
     * Finds a list of all the JFileChoosers being shown by a window.
     *
     * @param win    The window to owning the chooser.
     * @return The list of JFileChoosers.
     * @deprecated 2.05 use JFileChooserFinder(null).findAl(win);
     */
    public List getShowingJFileChoosers(final Window win) {
        List         ret      = new ArrayList();
        List         subComps;
        JFileChooser chooser;

        Window[]     owned  = win.getOwnedWindows();
        Finder       finder = new ComponentFinder(JFileChooser.class);

        for (int i = 0; i < owned.length; i++) {
            if (!(owned[i] instanceof JDialog)) {
                continue;
            }

            subComps = Finder.findComponentList(
                    finder,
                    owned[i],
                    new ArrayList(),
                    0);

            Iterator iter = subComps.iterator();

            while (iter.hasNext()) {
                chooser = (JFileChooser) iter.next();

                if (finder.testComponent(chooser)) {
                    ret.add(chooser);
                }
            }
        }

        return ret;
    }

    /**
     * Finds a component which is an instance of the given class by
     * searching from all top level windows.
     *
     * @param compCls Component class
     * @param index  The index of the component. The first component matching the
     *               criteria will have index 0, the second 1, etc.
     * @return Component Component found. Null if index components
     *         do not exist.
     * @deprecated 2.02 Use new ComponentFinder(aClass).find(index);
     */
    public static Component findComponent(final Class compCls, final int index) {
        return new ComponentFinder(compCls).find(index);
    }

    /**
     * Finds a component which is an instance of the given class within a given
     * container.
     *
     * @param compCls The class of the component.
     * @param cont    The container that the component will be in.
     * @param index   The index of the component. The first component matching the
     *                criteria will have index 0, the second 1, etc.
     * @return The component that has been found (or null if none found).
     * @deprecated 2.02 Use new ComponentFinder(compCls).find(cont, index);
     */
    public static Component findComponent(final Class compCls,
        final Container cont, final int index) {
        return new ComponentFinder(compCls).find(cont, index);
    }

    /**
     * Finds a component which is an instance of the given class by
     * searching from all top level windows.
     *
     * @param finder Finder class to be used for locating components.
     * @param index  The index of the component. The first component matching the
     *               criteria will have index 0, the second 1, etc.
     * @return Component Component found. Null if index components
     *         do not exist.
     * @deprecated 2.02 Use finder.find(index);
     */
    public static Component findComponent(final Finder finder, final int index) {
        return (Component) finder.find(index);
    }

    /**
     * Finds a component matching the given criteria within a given container.
     *
     * @param finder The Finder that is used to test components for a
     *               match.
     * @param cont   The container that the component will be in.
     * @param index  The index of the component. The first component matching the
     *               criteria will have index 0, the second 1, etc.
     * @return The component that has been found (or null if none found).
     * @deprecated 2.02 use finder.find(cont, index);
     */
    public static Component findComponent(final Finder finder,
        final Container cont, final int index) {
        return finder.find(cont, index);
    }

    /**
     * Finds a component which is an instance of the given class by
     * searching from all top level windows.
     *
     * @param compCls Component class
     * @param index  The index of the component. The first component matching the
     *               criteria will have index 0, the second 1, etc.
     * @return Component Component found. Null if index components
     *         do not exist.
     * @deprecated 2.02 Use new ComponentFinder(compCls).find(cont, index);
     */
    public static Component findNamedComponent(final Class compCls,
        final int index) {
        return new NamedComponentFinder(compCls, null).find(index);
    }

    /**
     * Finds a indexed component which has the given name by
     * searching from all top level windows.
     *
     * @param name The pattern for the name of the component to find.
     * @param index  The index of the component. The first component matching the
     *               criteria will have index 0, the second 1, etc.
     * @return Component Component found. Null if index components
     *         do not exist.
     * @deprecated 2.02 Use new NamedComponentFinder(null,name).find(index);
     */
    public static Component findNamedComponent(final String name,
        final int index) {
        return new NamedComponentFinder(null, name).find(index);
    }

    /**
     * Finds a component which is an instance of the given class
     * and has the given name by searching from all top level windows.
     *
     * @param compCls Component class
     * @param name The pattern for the name of the component to find.
     * @param index  The index of the component. The first component matching the
     *               criteria will have index 0, the second 1, etc.
     * @return Component Component found. Null if index components
     *         do not exist.
     * @deprecated 2.02 Use new NamedComponentFinder(compCls, name).find(index);
     */
    public static Component findNamedComponent(final Class compCls,
        final String name, final int index) {
        return new NamedComponentFinder(compCls, name).find(index);
    }

    /**
     * Finds a component matching the given criteria within a given container.
     *
     * @param name   The pattern for the name of the component to find.
     * @param cont   The container that the component will be in.
     * @param index  The index of the component. The first component matching the
     *               criteria will have index 0, the second 1, etc.
     * @return The component that has been found (or null if none found).
     * @deprecated 2.02 Use new NamedComponentFinder(null, name).find(cont, index);
     */
    public static Component findNamedComponent(final String name,
        final Container cont, final int index) {
        return findNamedComponent(null, name, cont, index);
    }

    /**
     * Finds a component matching the given criteria within a given container.
     *
     * @param aClass The component's class.
     * @param cont   The container that the component will be in.
     * @param index  The index of the component. The first component matching the
     *               criteria will have index 0, the second 1, etc.
     * @return The component that has been found (or null if none found).
     * @deprecated 2.02 Use new NamedComponentFinder(aClass, null).find(cont, index);
     */
    public static Component findNamedComponent(final Class aClass,
        final Container cont, final int index) {
        return findNamedComponent(aClass, null, cont, index);
    }

    /**
     * Finds a component matching the given criteria within a given container.
     *
     * @param aClass The component's class.
     * @param name   The pattern for the name of the component to find.
     * @param cont   The container that the component will be in.
     * @param index  The index of the component. The first component matching the
     *               criteria will have index 0, the second 1, etc.
     * @return The component that has been found (or null if none found).
     * @deprecated 2.02 Use new NamedComponentFinder(aClass, name).find(cont, index);
     */
    public static Component findNamedComponent(final Class aClass,
        final String name, final Container cont, final int index) {
        return new NamedComponentFinder(aClass, name).find(cont, index);
    }

    /**
     * Find the index of the component given.
     * @param finder Finder used for indexing.
     * @param cont Container to look for the component in.
     * @param comp Component to find the index of.
     * @return Index of the component
     */
    public static int indexOf(final Finder finder, final Container cont,
        final Component comp) {
        Container[] containers = null;

        if (cont != null) {
            if (!cont.isShowing()) {
                finder.setIgnoreVisibility(true);
            }

            containers = new Container[] {cont};
        } else {
            containers = getAllWindows();
        }

        int index = 0;

        for (int i = 0; i < containers.length; i++) {
            List list = Finder.findComponentList(
                    finder,
                    containers[i],
                    new ArrayList(),
                    999);

            if (list.contains(comp)) {
                return index + list.indexOf(comp);
            } else {
                index += list.size();
            }
        }

        return -1;
    }

    /**
     * Set the step size of mouse move events.
     * @param step size in pixels.
     */
    public void setStep(final int step) {
        this.m_step = step;
    }

    /**
     * Returns the step size for the move events.
     * @return int size in pixels.
     */
    public int getStep() {
        return m_step;
    }

    /**
     * Helper method to fire appropriate events to click(s) a component with a
     * custom wait method.
     *
     * @param evtData The event data container
     */
    public void enterClickAndLeave(final AbstractMouseEventData evtData) {
        if (evtData == null) {
            return;
        }

        evtData.getTestCase().pauseAWT();

        if (!evtData.prepareComponent()) {
            evtData.getTestCase().resumeAWT();

            return;
        }

        evtData.getTestCase().resumeAWT();

        int       numberOfClicks = evtData.getNumberOfClicks();
        int       modifiers      = evtData.getModifiers();
        boolean   isPopupTrigger = evtData.getPopupTrigger();

        Component ultimate = evtData.getRoot();

        // try to clear the event queue
        evtData.getTestCase().flushAWT();

        // Translate the screen coordinates returned by the event to frame coordinates.
        Point screen = evtData.getLocationOnScreen();
        pressModifiers(
            ultimate,
            mouseToKeyModifiers(modifiers));

        // Insure a move event.
        mouseMoved(ultimate, screen.x - 5, screen.y - 5);
        mouseMoved(ultimate, screen.x, screen.y);

        for (int click = 1; click <= numberOfClicks; click++) {
            mousePressed(ultimate, modifiers, click, isPopupTrigger);
            mouseReleased(ultimate, modifiers, click, isPopupTrigger);
        }

        releaseModifiers(
            ultimate,
            mouseToKeyModifiers(modifiers));

        // try to clear the event queue
        evtData.getTestCase().flushAWT();
    }

    /**
     * Helper method to fire appropriate events to drag and drop a component with a
     * custom wait method.
     *
     * Note this will not work with java.awt.dnd API implementations.
     * The java.awt.dnd implementation is driven from a level below the
     * AWT EventQueue. Use the  RobotTestHelper to work with java.awt.dnd.
     *
     * @param srcEvtData The source  event data container
     * @param dstEvtData The destination event data container
     * @param incr       Amount to increment the coords while "moving" from
     *                   source to destination.
     */
    public void enterDragAndLeave(final AbstractMouseEventData srcEvtData,
        final AbstractMouseEventData dstEvtData, final int incr) {
        DragEventData ded = new DragEventData(
                srcEvtData.getTestCase(),
                srcEvtData,
                dstEvtData);
        setStep(incr);
        enterDragAndLeave(ded);
    }

    /**
     * Helper method to fire appropriate events to drag and drop a component with a
     * custom wait method.
     *
     * Note this will not work with java.awt.dnd API implementations.
     * The java.awt.dnd implementation is driven from a level below the
     * AWT EventQueue. Use the  RobotTestHelper to work with java.awt.dnd.
     *
     * @param ded    The destination event data container
     */
    public void enterDragAndLeave(final DragEventData ded) {
        // Is component 2 to be visible from the get go.
        // It may not be if autoScrolling is to take place.
        // If autoScrolling then when do we adjust the screen?
        // How do we calculate the target X/Y?
        // For now we will prepare the second component and
        // attempt to insure visibility
        ded.getTestCase().pauseAWT();

        if (!ded.prepareComponent()) {
            ded.getTestCase().resumeAWT();

            return;
        }

        ded.getTestCase().resumeAWT();

        int numberOfClicks  = ded.getSource().getNumberOfClicks();
        int modifiers       = ded.getSource().getModifiers();
        int keyModifiers    = mouseToKeyModifiers(modifiers);
        int mouseButtonMask = modifiers
            & (InputEvent.BUTTON1_MASK | InputEvent.BUTTON2_MASK
            | InputEvent.BUTTON3_MASK);
        boolean   isPopupTrigger = ded.getSource().getPopupTrigger();

        Component ultimate = ded.getSource().getComponent();
        Point     p        = ded.getSource().getLocationOnScreen();

        // try to clear the event queue
        ded.getTestCase().flushAWT();

        mouseMoved(ultimate, p.x - 5, p.y - 5);
        mouseMoved(ultimate, p.x, p.y);
        pressModifiers(ultimate, keyModifiers);
        mousePressed(ultimate, mouseButtonMask, 0, isPopupTrigger);

        Point[] pts = ded.getPoints();

        for (int i = 0; i < pts.length; i++) {
            mouseMoved(ultimate, pts[i].x, pts[i].y);
        }

        mouseReleased(ultimate, mouseButtonMask, numberOfClicks, isPopupTrigger);
        releaseModifiers(ultimate, keyModifiers);

        // try to clear the event queue
        ded.getTestCase().flushAWT();
    }

    /**
     * Enter a mouseWheel event.
     *
     * @param evtData MouseWheelEventData to be processed.
     */
    public void enterMouseWheel(final MouseWheelEventData evtData) {
        if (evtData == null) {
            return;
        }

        evtData.getTestCase().pauseAWT();

        if (!evtData.prepareComponent()) {
            evtData.getTestCase().resumeAWT();

            return;
        }

        evtData.getTestCase().resumeAWT();

        int       numberOfClicks = evtData.getWheelRotation();
        int       modifiers      = evtData.getModifiers();
        boolean   isPopupTrigger = evtData.getPopupTrigger();

        Component ultimate = evtData.getSource();

        // Translate the screen coordinates returned by the event to frame coordinates.
        Point screen = evtData.getLocationOnScreen();
        int   mods = mouseToKeyModifiers(modifiers);
        pressModifiers(ultimate, mods);

        // try to clear the event queue
        evtData.getTestCase().flushAWT();

        // Ensure a move event.
        mouseMoved(ultimate, screen.x - 5, screen.y - 5);
        mouseMoved(ultimate, screen.x, screen.y);
        evtData.getTestCase().flushAWT();

        int inc = 1;

        if (numberOfClicks < 0) {
            inc = -1;
        }

        int mx = numberOfClicks;

        if (numberOfClicks < 0) {
            mx *= -1;
        }

        for (int i = 0; i < mx; i++) {
            mouseWheel(
                ultimate,
                evtData.getScrollAmount(),
                inc);
        }

        evtData.getTestCase().flushAWT();

        releaseModifiers(ultimate, mods);

        // try to clear the event queue
        evtData.getTestCase().flushAWT();
    }

    /**
     * This method is used to send KeyPressed, KeyTyped
     * and KeyReleased events (in that order) to the specified
     * component. This method should be used only to send Action key events.
     * NOTE: This method will not call 'requestFocus()' on the component - the developer
     * should include it in the test code if needed.
     *
     * @param evtData  The event data container.
     */
    public void sendKeyAction(final KeyEventData evtData) {
        sendString(evtData);
    }

    /**
     * This method is used to send KeyPressed, KeyTyped
     * and KeyReleased events (in that order) to the specified
     * component. This method can also differentiate between
     * upper and lower case characters in the specified string.
     * NOTE: This method will not call 'requestFocus()' on the component - the developer
     * should include it in the test code if needed.
     *
     * @param evtData  The event data container.
     */
    public void sendString(final AbstractKeyEventData evtData) {
        if (evtData == null) {
            return;
        }

        evtData.getTestCase().pauseAWT();

        if (!evtData.prepareComponent()) {
            evtData.getTestCase().resumeAWT();

            return;
        }

        evtData.getTestCase().resumeAWT();

        Component ultimate  = evtData.getRoot();
        int       modifiers = evtData.getModifiers();

        // try to clear the event queue
        evtData.getTestCase().flushAWT();

        pressModifiers(ultimate, modifiers);

        JFCKeyStroke[] keyStroke = evtData.getKeyStrokes();

        for (int k = 0; k < keyStroke.length; k++) {
            adjustModifiers(
                ultimate,
                modifiers,
                keyStroke[k].getModifiers());
            modifiers = keyStroke[k].getModifiers();
            keyPressed(ultimate, keyStroke[k]);
            keyReleased(ultimate, keyStroke[k]);
        }

        releaseModifiers(ultimate, modifiers);

        // try to clear the event queue
        evtData.getTestCase().flushAWT();
    }

    /**
     * Convert the modifiers from mouse modifiers to
     * keyboard modifiers. ALT modifier overlaps with
     * BUTTON2 and META overlaps with BUTTON3.
     *
     * @param  modifiers   The mouse modifiers that need to be converted.
     * @return int         The corresponding keyboard modifiers.
     */
    protected static int mouseToKeyModifiers(final int modifiers) {
        int keyModifiers = modifiers;

        int mouseButtons = InputEvent.BUTTON1_MASK + InputEvent.BUTTON2_MASK
            + InputEvent.BUTTON3_MASK;
        keyModifiers = keyModifiers & (0xFFFFFFFF ^ mouseButtons);

        return keyModifiers;
    }

    /**
     * Send the modifier key release event for the modifiers.
     *
     * @param ultimate  Component to receive the events.
     * @param modifiers The modifiers to be released.
     * @param newModifiers The new modifiers which should be pressed.
     */
    protected final void adjustModifiers(final Component ultimate,
        final int modifiers, final int newModifiers) {
        if (modifiers == newModifiers) {
            return;
        }

        boolean s1 = (modifiers & InputEvent.SHIFT_MASK) > 0;
        boolean s2 = (newModifiers & InputEvent.SHIFT_MASK) > 0;
        boolean c1 = (modifiers & InputEvent.CTRL_MASK) > 0;
        boolean c2 = (newModifiers & InputEvent.CTRL_MASK) > 0;
        boolean m1 = (modifiers & InputEvent.META_MASK) > 0;
        boolean m2 = (newModifiers & InputEvent.META_MASK) > 0;
        boolean a1 = (modifiers & InputEvent.ALT_MASK) > 0;
        boolean a2 = (newModifiers & InputEvent.ALT_MASK) > 0;
        boolean g1 = (modifiers & InputEvent.ALT_GRAPH_MASK) > 0;
        boolean g2 = (newModifiers & InputEvent.ALT_GRAPH_MASK) > 0;

        int     mods = modifiers;

        if (s1 && !s2) {
            // Release shift
            mods = mods & (-1 ^ InputEvent.SHIFT_MASK);

            JFCKeyStroke[] strokes = getKeyMapping().getKeyStrokes(KeyEvent.VK_SHIFT);
            strokes[0].setModifiers(mods);
            keyReleased(ultimate, strokes[0]);
        } else if (!s1 && s2) {
            // Press shift
            mods = mods | InputEvent.SHIFT_MASK;

            JFCKeyStroke[] strokes = getKeyMapping().getKeyStrokes(KeyEvent.VK_SHIFT);
            strokes[0].setModifiers(mods);
            keyPressed(ultimate, strokes[0]);
        }

        if (c1 && !c2) {
            // Release control
            mods = mods & (-1 ^ InputEvent.CTRL_MASK);

            JFCKeyStroke[] strokes = getKeyMapping().getKeyStrokes(KeyEvent.VK_CONTROL);
            strokes[0].setModifiers(mods);
            keyReleased(ultimate, strokes[0]);
        } else if (!c1 && c2) {
            // Press control
            mods = mods | InputEvent.CTRL_MASK;

            JFCKeyStroke[] strokes = getKeyMapping().getKeyStrokes(KeyEvent.VK_CONTROL);
            strokes[0].setModifiers(mods);
            keyPressed(ultimate, strokes[0]);
        }

        if (a1 && !a2) {
            // Release alt
            mods = mods & (-1 ^ InputEvent.ALT_MASK);

            JFCKeyStroke[] strokes = getKeyMapping().getKeyStrokes(KeyEvent.VK_ALT);
            strokes[0].setModifiers(mods);
            keyReleased(ultimate, strokes[0]);
        } else if (!a1 && a2) {
            // Press alt
            mods = mods | InputEvent.ALT_MASK;

            JFCKeyStroke[] strokes = getKeyMapping().getKeyStrokes(KeyEvent.VK_ALT);
            strokes[0].setModifiers(mods);
            keyPressed(ultimate, strokes[0]);
        }

        if (g1 && !g2) {
            // Release alt graph
            mods = mods & (-1 ^ InputEvent.ALT_GRAPH_MASK);

            JFCKeyStroke[] strokes = getKeyMapping().getKeyStrokes(KeyEvent.VK_ALT_GRAPH);
            strokes[0].setModifiers(mods);
            keyReleased(ultimate, strokes[0]);
        } else if (!g1 && g2) {
            // Press alt graph
            mods = mods | InputEvent.ALT_GRAPH_MASK;

            JFCKeyStroke[] strokes = getKeyMapping().getKeyStrokes(KeyEvent.VK_ALT_GRAPH);
            strokes[0].setModifiers(mods);
            keyPressed(ultimate, strokes[0]);
        }

        if (m1 && !m2) {
            // Release meta
            mods = mods & (-1 ^ InputEvent.META_MASK);

            JFCKeyStroke[] strokes = getKeyMapping().getKeyStrokes(KeyEvent.VK_META);
            strokes[0].setModifiers(mods);
            keyReleased(ultimate, strokes[0]);
        } else if (!m1 && m2) {
            // Press meta
            mods = mods | InputEvent.META_MASK;

            JFCKeyStroke[] strokes = getKeyMapping().getKeyStrokes(KeyEvent.VK_META);
            strokes[0].setModifiers(mods);
            keyPressed(ultimate, strokes[0]);
        }
    }

    /**
     * Send the modifier key press event for the modifiers.
     *
     * @param ultimate  Component to receive the events.
     * @param modifiers The modifiers to be pressed.
     */
    protected final void pressModifiers(final Component ultimate,
        final int modifiers) {
        adjustModifiers(ultimate, 0, modifiers);
    }

    /**
     * Send the modifier key release event for the modifiers.
     *
     * @param ultimate  Component to receive the events.
     * @param modifiers The modifiers to be released.
     */
    protected final void releaseModifiers(final Component ultimate,
        final int modifiers) {
        adjustModifiers(ultimate, modifiers, 0);
    }

    /**
     * Check to see if the value is bounded by the min and max.
     * @param val Value to be checked.
     * @param vMin minimum value of range.
     * @param vMax maximum value of range.
     * @return true if the value is between vMin and vMax.
     */
    protected static boolean isBounded(final int val, final int vMin,
        final int vMax) {
        int min = Math.min(vMin, vMax);
        int max = Math.max(vMin, vMax);

        if ((min < val) && (val < max)) {
            return true;
        }

        return false;
    }

    /**
     * Process a key press event on a component.
     *
     * @param   ultimate    The ultimate parent Component
     * @param   stroke      The JFCKeyStroke to be performed.
     */
    protected abstract void keyPressed(final Component ultimate,
        final JFCKeyStroke stroke);

    /**
     * Process a key release event on a component.
     *
     * @param   ultimate    The ultimate parent Component
     * @param   stroke      The JFCKeyStroke to be performed.
     */
    protected abstract void keyReleased(final Component ultimate,
        final JFCKeyStroke stroke);

    /**
     * Process a mouse move event on a component.
     *eve
     * @param   ultimate    The ultimate parent Component
     * @param   x           The x coordinate of the point where the mouse is being moved to.
     * @param   y           The y coordinate of the point where the mouse is being moved to.
     */
    protected abstract void mouseMoved(final Component ultimate, final int x,
        final int y);

    /**
     * Process a mouse press event on a component.
     *
     * @param   ultimate          The ultimate parent Component
     * @param   modifiers         The modifiers associated with this mouse event.
     * @param   click             The number of clicks associated with this mouse event.
     * @param   isPopupTrigger    Whether this mouse event will generate a popup.
     */
    protected abstract void mousePressed(final Component ultimate,
        final int modifiers, final int click, final boolean isPopupTrigger);

    /**
     * Process a mouse release event on a component.
     *
     * @param   ultimate          The ultimate parent Component
     * @param   modifiers         The modifiers associated with this mouse event.
     * @param   click             The number of clicks associated with this mouse event.
     * @param   isPopupTrigger    Whether this mouse event will generate a popup.
     */
    protected abstract void mouseReleased(final Component ultimate,
        final int modifiers, final int click, final boolean isPopupTrigger);

    /**
     * Simulate rotating the mouseWheel.
     *
     * @param ultimate Component to fire the events on.
     * @param amount Amount to scroll for each rotation.
     * @param rotation clicks to rotate the mouse wheel.
     */
    protected abstract void mouseWheel(final Component ultimate,
        final int amount, final int rotation);

    /**
     * Calculate the direction of travel given the src
     * and dest point.
     * @param src Source
     * @param dest Destination
     * @return -1 if dest < src, 0 if dest = src, and 1 if dest > src
     */
    private static int getDir(final int src, final int dest) {
        if (src == dest) {
            return 0;
        } else if (src < dest) {
            return 1;
        } else {
            return -1;
        }
    }
}
