package junit.extensions.jfcunit.eventdata;

import junit.extensions.jfcunit.JFCTestCase;
import junit.extensions.jfcunit.xml.JFCXMLConstants;

import junit.extensions.xml.IXMLTestCase;
import junit.extensions.xml.XMLException;
import junit.extensions.xml.elements.AbstractTagHandler;

import junit.framework.Assert;

import org.w3c.dom.Element;

import java.awt.Component;
import java.awt.Point;
import java.awt.event.InputEvent;

import java.util.StringTokenizer;


/**
 * Implements the common methods used by many of the EventData classes.
 * @author Kevin Wilson
 */
public abstract class BaseEventDataTagHandler extends AbstractTagHandler
    implements EventDataConstants,
        JFCXMLConstants {
    /**
     * Event data generated by the tag handler implementation.
     */
    private AbstractMouseEventData m_eventData = null;

    /**
     * Constructor for BaseEventDataTagHandler.
     *
     * @param element     The element to be processed
     * @param testCase    The IXMLTestCase that uses this element
     */
    public BaseEventDataTagHandler(final Element element,
        final IXMLTestCase testCase) {
        super(element, testCase);
    }

    /**
     * Returns the test case.
     * @return testCase
     */
    public JFCTestCase getJFCTestCase() {
        IXMLTestCase tc = super.getXMLTestCase();

        if (tc instanceof JFCTestCase) {
            return (JFCTestCase) tc;
        }

        return null;
    }

    /**
     * @see junit.extensions.xml.elements.AbstractTagHandler#validateElement()
     * @throws XMLException when the required elements are not present.
     */
    public void validateElement() throws XMLException {
        // do the default validations from the super class
        super.validateElement();

        // reqd attribute: refid
        checkRequiredAttribute(REFID);
    }

    /**
     * Returns the value of the CLICKS attribute for this element. Defaults to
     * DEFAULT_NUMBEROFCLICKS.
     * @return int  The value of the CLICKS attribute, DEFAULT_NUMBEROFCLICKS if not specified.
     */
    protected int getClicks() {
        return getInt(CLICKS, DEFAULT_NUMBEROFCLICKS);
    }

    /**
     * Returns the value of the (previously found) component whose name is the value of the REFID
     * attribute for this element.
     * @return Component The found component whose name matches with the REFID attribute.
     */
    protected Component getComponent() {
        Component comp = (Component) getXMLTestCase().getProperty(
                getString(REFID));
        Assert.assertNotNull("Component for " + getString(REFID) + " is null",
            comp);

        return comp;
    }

    /**
     * Returns the value of theMODIFIERS attribute for this element. Defaults
     * to DEFAULT_MOUSE_MODIFIERS.
     * @return int The value of the MODIFIERS attribute, DEFAULT_MOUSE_MODIFIERS if not specified.
     */
    protected int getModifiers() {
        if (this.getPopupTrigger()) {
            return getModifiers(DEFAULT_POPUP_MODIFIERS);
        }

        return getModifiers(DEFAULT_MOUSE_MODIFIERS);
    }

    /**
     * Returns the value of the MODIFIERS attribute for this element. Defaults to
     * defaultValue.
     * @param defaultValue value to be returned if the element does not exist.
     * @return int  The value of the MODIFIERS attribute, defaultValue if not specified.
     */
    protected int getModifiers(final int defaultValue) {
        int    modifiers = 0;
        String s = getString(MODIFIERS);

        if (s == null) {
            return defaultValue;
        }

        try {
            // If we can parse a integer from the string then use the integer
            // value.
            return Integer.parseInt(s);
        } catch (NumberFormatException nfe) {
            // Ignore
        }

        StringTokenizer tok    = new StringTokenizer(s, "+");
        boolean         button = false;

        while (tok.hasMoreElements()) {
            String token = tok.nextToken();

            if (token.equalsIgnoreCase(SHIFT)) {
                modifiers += InputEvent.SHIFT_MASK;
            }

            if (token.equalsIgnoreCase(CTRL)) {
                modifiers += InputEvent.CTRL_MASK;
            }

            if (token.equalsIgnoreCase(ALT)) {
                modifiers += InputEvent.ALT_MASK;
            }

            if (token.equalsIgnoreCase(ALTGR)) {
                modifiers += InputEvent.ALT_GRAPH_MASK;
            }

            if (token.equalsIgnoreCase(META)) {
                modifiers += InputEvent.META_MASK;
            }

            if (token.equalsIgnoreCase(BUTTON1)) {
                button = true;
                modifiers += InputEvent.BUTTON1_MASK;
            }

            if (token.equalsIgnoreCase(BUTTON2)) {
                button = true;
                modifiers += InputEvent.BUTTON2_MASK;
            }

            if (token.equalsIgnoreCase(BUTTON3)) {
                button = true;
                modifiers += InputEvent.BUTTON3_MASK;
            }
        }

        if (!button) {
            int defModifiers = AbstractMouseEventData.getDefaultModifiers(
                    getPopupTrigger());
            modifiers = (modifiers | defModifiers);
        }

        return modifiers;
    }

    /**
     * Returns the value of the INDEX attribute for this element. Defaults to 0 (zero) if not found.
     * @return int  The value of the parsed INDEX attribute, zero if not found.
     */
    protected int getOffset() {
        try {
            return getInt(INDEX, INVALID_TEXT_OFFSET);
        } catch (NumberFormatException e) {
            throw new NumberFormatException("Invalid offset specification");
        }
    }

    /**
     * Returns the value of the POPUPTRIGGER attribute for this element.
     * @return boolean  The value of the POPUPTRIGGER attribute.
     */
    protected boolean getPopupTrigger() {
        return getBoolean(POPUPTRIGGER);
    }

    /**
     * Returns the constant (defined in {@link EventDataConstants}) which corresponds to the value
     * of the POSITION attribute of this element.
     * @return int  The corresponding constant from EventDataConstants.
     */
    protected int getPosition() {
        String str = getString(POSITION);

        if ("CENTER".equalsIgnoreCase(str)) {
            return CENTER;
        } else if ("NORTH".equalsIgnoreCase(str) || "N".equalsIgnoreCase(str)) {
            return NORTH;
        } else if ("NORTH_EAST".equalsIgnoreCase(str)
                || "NORTHEAST".equalsIgnoreCase(str)
                || "NE".equalsIgnoreCase(str)) {
            return NORTH_EAST;
        } else if ("EAST".equalsIgnoreCase(str) || "E".equalsIgnoreCase(str)) {
            return EAST;
        } else if ("SOUTH_EAST".equalsIgnoreCase(str)
                || "SOUTHEAST".equalsIgnoreCase(str)
                || "SE".equalsIgnoreCase(str)) {
            return SOUTH_EAST;
        } else if ("SOUTH".equalsIgnoreCase(str) || "S".equalsIgnoreCase(str)) {
            return SOUTH;
        } else if ("SOUTH_WEST".equalsIgnoreCase(str)
                || "SOUTHWEST".equalsIgnoreCase(str)
                || "SW".equalsIgnoreCase(str)) {
            return SOUTH_WEST;
        } else if ("WEST".equalsIgnoreCase(str) || "W".equalsIgnoreCase(str)) {
            return WEST;
        } else if ("NORTH_WEST".equalsIgnoreCase(str)
                || "NORTHWEST".equalsIgnoreCase(str)
                || "NW".equalsIgnoreCase(str)) {
            return NORTH_WEST;
        } else if ("CUSTOM".equalsIgnoreCase(str)) {
            return CUSTOM;
        } else if ("PERCENT".equalsIgnoreCase(str)) {
            return PERCENT;
        }

        // default to CENTER
        return getInt(POSITION, CENTER);
    }

    /**
     * Returns the value of the REFERENCE attribute for this element. Defaults to null.
     * @return Point  The value of the REFERENCE attribute. Defaults to null of not found.
     * @throws XMLException may be throws if the point cannot be parsed properly.
     */
    protected Point getReference() throws XMLException {
        return getPoint(REFERENCE, null);
    }

    /**
     * Returns the value of the SLEEPTIME attribute for this element. Defaults to DEFAULT_SLEEPTIME
     * if not found.
     * @return long  The value of the SLEEPTIME attribute. Defaults to DEFAULT_SLEEPTIME.
     */
    protected long getSleepTime() {
        return getLong(SLEEPTIME, DEFAULT_SLEEPTIME);
    }

    /**
     * Set the event data.
     * @param data Data to be used as the event data for this TagHandler.
     */
    void setEventData(final AbstractMouseEventData data) {
        m_eventData = data;
    }

    /**
     * Gets the event data generated by this class.
     * @return event data created.
     * @throws XMLException may be thrown when processing children.
     */
    AbstractMouseEventData getEventData() throws XMLException {
        processElement();

        return m_eventData;
    }
}
