package junit.extensions.jfcunit.finder;

import java.awt.Component;

import javax.swing.JLabel;
import java.awt.Container;


/**
 * Find a component based upon the label attached via JLabel.setLabelFor(Component comp).
 *
 * @author <a href="mailto:vraravam@thoughtworks.com">Vijay Aravamudhan : ThoughtWorks Inc.</a>
 */
public class LabeledComponentFinder extends Finder {
    /**
     * The label Regular expression to be found.
     */
    private String m_text = null;

    /**
     * Ignore the case of the search.
     */
    private boolean m_caseIndependent = true;

    /**
     * Constructor accepting all arguments needed to filter the {@link java.awt.Component}.
     *
     * @param text Regular expression describing the label to be found.
     * @param caseIndependent true if the case should be ignored.
     */
    public LabeledComponentFinder(final String text,
        final boolean caseIndependent) {
      setText(text);
      m_caseIndependent = caseIndependent;
      super.createPatternMatcher(text, caseIndependent);
    }

    /**
     * Set the finder into a case independent mode.
     * @param ignoreCase true if case should be ignored.
     */
    public void setCaseIndependent(final boolean ignoreCase) {
        super.setCaseIndependent(ignoreCase);
        m_caseIndependent = ignoreCase;
        createPatternMatcher(m_text, m_caseIndependent);
    }

    /**
     * Set the text to be matched against the label.
     * @param text String to be matched.
     */
    public final void setText(final String text) {
        m_text = text;
        createPatternMatcher(m_text, m_caseIndependent);
    }

    /**
     * Get the text to be matched against the label.
     * @return String text to be matched.
     */
    public final String getText() {
        return m_text;
    }

    /**
     * Method that returns true if the given {@link java.awt.Component} matches the search
     * criteria.
     *
     * @param comp   The component to test
     * @return true if this component is a match
     */
    public boolean testComponent(final Component comp) {
        if (comp != null && comp instanceof JLabel) {
            JLabel lbl = (JLabel) comp;
            return super.evaluate(getText(), lbl.getText());
        }
        return false;
    }

    /**
     * Find the component in the container at the given index.
     * @param conts Containers to be searched.
     * @param index Index of the component to find.
     * @return Component which was found or null.
     */
    public Component find(final Container[] conts, final int index) {
      Component c = super.find(conts, index);
      if (c != null) {
        c = ((JLabel) c).getLabelFor();
      }
      return c;
    }
}
