package junit.extensions.xml.elements;

import junit.extensions.xml.IXMLTestCase;
import junit.extensions.xml.XMLException;

import junit.framework.Assert;

import org.w3c.dom.Element;


/**
 * This class will handle the processing of &lt;assertenabled&gt; nodes.
 *
 * <H3>Mandatory attributes</H3>
 *    refid Id of the property to be compared.
 *
 * <H3>Optional attributes (default value)</H3>
 *    message Message to be displayed when assertion fails.<br>
 *    enabled true or false. Default value is true.
 *
 * <H3>Examples:</H3>
 *    &lt;assertenabled refid=&quot;MyComponent&quot;
 *       message=&quot;Component is not enabled&quot;
 *       enabled=&quot;true&quot;/&gt;<br>
 *    &lt;assertenabled refid=&quot;MyComponent&quot;
 *       message=&quot;Component is enabled&quot;
 *       enabled=&quot;false&quot;/&gt;<br>
 *    &lt;assertenabled refid=&quot;MyComponent&quot;
 *       message=&quot;Component is not enabled&quot;/&gt;<br>
 *
 * @author <a href="mailto:baylward@nexagent.com">Bruce Aylward : Nexagent Ltd.</a>
 */
public class AssertEnabledTagHandler extends AbstractAssertTagHandler {
    /**
     * Constructor for AssertEnabledTagHandler.
     *
     * @param element     The element to be processed
     * @param testCase    The IXMLTestCase that uses this element
     */
    public AssertEnabledTagHandler(final Element element,
        final IXMLTestCase testCase) {
        super(element, testCase);
    }

    /**
     * This method obtains the refid which is assumed to
     * be a java.awt.Component. The enabled method is tested
     * and compared with test enabed attribute if specified.
     * If enabled is not specified it is assumed to be true.
     *
     * @throws XMLException may be thrown.
     */
    public void processElement() throws XMLException {
        validateElement();

        boolean result = false;
        Object  object = getXMLTestCase().getProperty(getRefId());
        Assert.assertNotNull(
            getMessage(),
            object);

        if (object instanceof java.awt.Component) {
            result = ((java.awt.Component) object).isEnabled();
        }

        Assert.assertTrue(
            getMessage(),
            result == getEnabled());
    }

    /**
     * Validate the element has a refid.
     * @throws XMLException when the refid is not specified.
     */
    public void validateElement() throws XMLException {
        // do the default validations from the super class
        super.validateElement();

        // reqd attribute: refid
        checkRequiredAttribute(REFID);
    }

    /**
     * Returns the value of the ENABLED attribute for this element.
     * @return String  The value of the ENABLED attribute.
     */
    private boolean getEnabled() {
        return getBoolean(ENABLED, true);
    }
}
