package junit.extensions.xml.elements;

import junit.extensions.xml.IXMLTestCase;
import junit.extensions.xml.XMLException;
import junit.extensions.xml.XMLTagResourceBundle;

import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;


/**
 * <H3>Title:</H3>
 * ChooseTagHandler
 * <H3>Description:</H3>
 * <p>The ChooseTagHandler allows for some
 * flow control in a test case. The handler is pattered
 * after the xsl:choose elements.</p>
 * <p>The choose tag processes its child elements which
 * should be either when or otherwise elements. Each when
 * element contains a test attribute which points to a
 * tag handler. If the tag handler does not assert, then the
 * test is assumed to process properly. The when element is
 * passed as the element to process to the test tag handler
 * defined. If the when does not cause a assertion, then the
 * test is assumed to be successful, and the child elements
 * of the when which succeeded will be processed, and the
 * processing of the Choose element will conclude.<p><p>
 *
 * If none of the When elements are tested successful, then
 * the otherwise elements children will be processed.<p><p>
 *
 * <H3>Example:</H3>
 * <pre>
 * &lt;choose&gt;
 *   &lt;when test=&quot;assertnotnull&quot; refid=&quot;ComponentA&quot;&gt;
 *     ...do when...
 *   &lt;/when&gt;
 *   &lt;when test=&quot;assertnotnull&quot; refid=&quot;ComponentB&quot;&gt;
 *     ...do when...
 *   &lt;/when&gt;
 *   &lt;otherwise&gt;
 *     ...do otherwise...
 *   &lt;/otherwise&gt;
 * &lt;/choose&gt;
 * </pre>
 * "choose" element requires no attributes but should contain
 * a ordered list of "when" elements and "otherwise".<p><p>
 *
 * "when" element requires one attribute "test" which defines a tag handler
 * to be processed. Other "when" attributes may be required depending
 * upon the tag handler specified. Children of the when should
 * be specified which are to be conditionaly processed.
 *
 * "otherwise" element requires no attributes. The children elements
 * will conditionally be processed.
 *
 * <p>Copyright: Copyright (c) 2003</p>
 * <p>Company: JFCUnit Sourceforge project</p>
 * @author Kevin Wilson
 * @version 1.0
 */
public class ChooseTagHandler extends AbstractTagHandler {
    /** choose string. */
    private String m_choose;

    /** other string. */
    private String m_other;

    /** when string. */
    private String m_when;

    /**
     * Constructor.
     * @param element Element to be processed.
     * @param testCase Test case processing tag handler.
     */
    public ChooseTagHandler(final Element element, final IXMLTestCase testCase) {
        super(element, testCase);
    }

    /**
     * process the element.
     * @throws XMLException is thrown if the element cannot be understood.
     */
    public void processElement() throws XMLException {
        validateElement();

        NodeList nodes     = getElement().getChildNodes();
        Element  otherwise = null;

        for (int i = 0; i < nodes.getLength(); i++) {
            Node node = nodes.item(i);

            if (node instanceof Element) {
                if (m_when.equals(node.getNodeName())) {
                    boolean result = false;
                    Element e = (Element) node;

                    try {
                        XMLTagResourceBundle.getTagHandler(
                            e,
                            getXMLTestCase(),
                            getString(e, TEST)).processElement();
                        result        = true;
                        otherwise     = null;
                    } catch (XMLException xe) {
                        throw xe;
                    } catch (Throwable t) {
                        result = false;
                    }

                    if (result) {
                        getXMLTestCase().processChildren(e);

                        return;
                    }
                } else if (m_other.equals(node.getNodeName())) {
                    otherwise = (Element) node;
                }
            }
        }

        if (otherwise != null) {
            getXMLTestCase().processChildren(otherwise);

            return;
        }
    }

    /**
     * Validate that the element is correct.
     * @throws XMLException may be thrown.
     */
    public void validateElement() throws XMLException {
        setChoose(CHOOSE);
        setWhen(WHEN);
        setOther(OTHERWISE);

        super.checkElementTagName(m_choose);
    }

    /**
     * Set the choose string.
     * @param choose Choose string to be used.
     */
    protected void setChoose(final String choose) {
        this.m_choose = choose;
    }

    /**
     * Set the other string.
     * @param other string to be used.
     */
    protected void setOther(final String other) {
        this.m_other = other;
    }

    /**
     * Set the when string.
     * @param when When string to be used.
     */
    protected void setWhen(final String when) {
        this.m_when = when;
    }
}
