/*
 * Created on Oct 27, 2003
 *
 * To change the template for this generated file go to
 * Window>Preferences>Java>Code Generation>Code and Comments
 */
package junit.extensions.xml.elements;

import junit.extensions.jfcunit.tools.Operator;

import junit.extensions.xml.IXMLTestCase;
import junit.extensions.xml.XMLException;

import junit.framework.Assert;

import org.w3c.dom.Element;

import javax.swing.JComboBox;
import javax.swing.JList;
import javax.swing.JTabbedPane;
import javax.swing.JTable;
import javax.swing.ListModel;
import javax.swing.table.TableColumn;
import javax.swing.table.TableColumnModel;


/**
 * Provide a mechanism for finding row/column in a table for a given value.
 * It will return throw an assertion if the value is not found.
 *
 * <H3>Tag Name:</H3>
 * indexof
 * <H3>Attributes</H3>
 * <pre>
 * refid   - the object to inspect                [required]
 * id      - the id of the results                [required]
 * value   - the expected value                   [required]
 * type    - the field type of the object         [required]
 *
 *   tablecell - locate a table cell in the
 *               specified row or column. row and column
 *               attributes may be used to narrow the search.
 *
 *   listcell  - locate a item in the list.
 *
 *   tab       - locate a tab.
 *
 *   tablecolumn - locate a table column.
 * </pre>
 * <H3>Optional</H3>
 * <pre>
 * index     - the x'th instance of the value to find [optional]
 * operation - The type of matching operation to perform.
 *             match - Use a regular expression match.
 *             equals - (default) if not specified.
 *             contains - The item contains the value.
 *             startswith - The item starts with the value.
 *             endswith   - The item ends with the value.
 *
 * useRE     - (depricated use operation="match")
 *             allow the value to contain
 *             a regular expression [optional]
 *           default - is false exact string match needed
 *           If set to true partial strings will work.
 * </pre>
 * <H3>Output</H3>
 * <pre>
 * Successful execution will set the following:<p>
 *   tablecell<p>
 *     id     to the value of the cell.
 *     id.row to the row of the tablecell.<p>
 *     id.column to the column of the tablecell.<p>
 *   tablecolumn<p>
 *     id        to the value of the header.
 *     id.index  to the column of the table header.
 *   listitem<p>
 *     id       to the value of the item.
 *     id.index to the index of the list item.<p>
 *   tab<p>
 *     id       to the value of the title.
 *     id.index to the index of the tab.<p>
 * Unsuccessfull id will be set to null and -1 will be set as
 * the index, row and column.<p>
 * </pre>
 * <H3>Examples:</H3>
 *   Return the cell where the value Joe is found in a specific column <p>
 * <pre>
 *     &lt;indexof id="cell" refid="JTable1" type="tablecell" column="1" value="Joe"/&gt;
 * </pre>
 *   Return the cell where the value Joe is found in a specific row
 * <pre>
 *     &lt;indexof id="cell" refid="JTable1" type="tablecell" row="1" value="Joe"/&gt;
 * </pre>
 *   Return the 2nd cell where the value Joe is found in the table
 * <pre>
 *     &lt;indexof id="cell" refid="JTable1" type="tablecell" value="Joe" index="2"/&gt;
 * </pre>
 *   Return the index of the column header containing SearchCode
 * <pre>
 *     &lt;indexof id="table" refid="JTable1" type="tablecolumn" value="SearchCode" index="0"/&gt;
 * </pre>
 *   Return the index of the tab containing the General.
 * <pre>
 *     &lt;indexof id="tab" refid="TabPane1" type="tab" value="General" index="0"/&gt;
 * </pre>
 *   Return the index of the tab containing the Peanut.
 * <pre>
 *     &lt;indexof id="item" refid="JList1" type="listitem" value="Peanut" index="0"/&gt;
 * </pre>
 *   Return the index of the list containing the Watermellon.
 * <pre>
 *     &lt;indexof id="item" refid="JComboBox1" type="listitem"
 *           value="Watermellon" index="0"/&gt;
 * </pre>
 * @author Kevin Wilson
 * @author John Yunker
 */
public class IndexOfTagHandler extends AbstractTagHandler {
    /**
     * Constructor.
     * @param element Element to be processed.
     * @param testcase Containing test case.
     */
    public IndexOfTagHandler(final Element element, final IXMLTestCase testcase) {
        super(element, testcase);
    }

    /**
     * Locate the index or cell matching the value.
     * @throws XMLException exception maybe thrown.
     */
    public void processElement() throws XMLException {
        validateElement();

        boolean debug     = getXMLTestCase().getDebug();
        int     op        = getOperation();
        String  id        = getString(ID);
        int     findcount = getInt(INDEX, 1);

        if (findcount <= 0) {
            findcount = 1;
        }

        String  refid = getString(REFID);
        Object  o = ((IXMLTestCase) getTestCase()).getProperty(refid);

        String  expectedval = getString(VALUE);
        boolean useRegExp   = getBoolean(USERE);

        if (useRegExp) {
            op = Operator.Operation.MATCH;

            if (debug) {
                System.err.println("Index of regular expression:" + expectedval
                    + " in " + o);
            }
        } else if (debug) {
            System.err.println("Index of:" + expectedval + " in " + o);
        }

        Assert.assertNotNull("Error: Unable to locate object id:" + refid, o);

        String type = getString(TYPE);
        Assert.assertNotNull("indexof 'type' is null", type);

        if (type.equals("tablecolumn")) {
            if (o instanceof JTable) {
                JTable           table = (JTable) o;
                TableColumnModel model = table.getColumnModel();

                int              columns = model.getColumnCount();

                for (int i = 0; i < columns; i++) {
                    TableColumn tc    = model.getColumn(i);
                    Object      value = tc.getHeaderValue();

                    if (value == null) {
                        value = tc.getIdentifier();
                    }

                    String s = null;

                    if (value != null) {
                        s = value.toString();
                    }

                    if (evaluate(op, s, expectedval)) {
                        if (debug) {
                            System.err.println("Match." + (findcount - 1)
                                + " more items to find");
                        }

                        if (--findcount == 0) {
                            getXMLTestCase().addProperty(id, value);
                            getXMLTestCase().addProperty(
                                id + ".index",
                                new Integer(i));

                            return;
                        }
                    }
                }

                // Not found
                getXMLTestCase().addProperty(id, null);
                getXMLTestCase().addProperty(
                    id + ".index",
                    new Integer(-1));

                if (debug) {
                    System.err.println("No match found");
                }
            } else {
                Assert.fail("refid must point to a JTable");
            }
        } else if (type.equals("tab")) {
            if (o instanceof JTabbedPane) {
                JTabbedPane tab = (JTabbedPane) o;

                int         count = tab.getTabCount();

                for (int i = 0; i < count; i++) {
                    String title = tab.getTitleAt(i);

                    if (evaluate(op, title, expectedval)) {
                        if (debug) {
                            System.err.println("Match." + (findcount - 1)
                                + " more items to find");
                        }

                        if (--findcount == 0) {
                            getXMLTestCase().addProperty(id, title);
                            getXMLTestCase().addProperty(
                                id + ".index",
                                new Integer(i));

                            return;
                        }
                    }
                }

                // Not found
                getXMLTestCase().addProperty(id, null);
                getXMLTestCase().addProperty(
                    id + ".index",
                    new Integer(-1));

                if (debug) {
                    System.err.println("No match found");
                }
            } else {
                Assert.fail("refid must point to JTabbedPane");
            }
        } else if (type.equals(TABLECELL)) {
            if (o instanceof JTable) {
                JTable table = (JTable) o;

                int    startrow = getInt(ROW, -1);
                int    startcol = getInt(COLUMN, -1);

                int    endrow;
                int    endcol;

                if (startrow == -1) {
                    startrow     = 0;
                    endrow       = table.getRowCount();
                } else {
                    endrow = startrow + 1;
                }

                if (startcol == -1) {
                    startcol     = 0;
                    endcol       = table.getColumnCount();
                } else {
                    endcol = startcol + 1;
                }

                int rowindex = 0;
                int colindex = 0;

                //System.out.println("rc:"+ovt.getRowCount()+" cc:"+ovt.getColumnCount());
                for (rowindex = startrow; rowindex < endrow; rowindex++) {
                    for (colindex = startcol; colindex < endcol; colindex++) {
                        Object cell     = table.getValueAt(rowindex, colindex);
                        String tableval = null;

                        if (cell != null) {
                            tableval = cell.toString();
                        }

                        if (debug) {
                            System.err.println("Evaluate row=" + rowindex
                                + " column=" + colindex + " value=" + tableval);
                        }

                        if (evaluate(op, tableval, expectedval)) {
                            if (debug) {
                                System.err.println("Match " + (findcount - 1)
                                    + "more items to find");
                            }

                            if (--findcount == 0) {
                                getXMLTestCase().addProperty(id, cell);
                                getXMLTestCase().addProperty(
                                    id + ".row",
                                    new Integer(rowindex));
                                getXMLTestCase().addProperty(
                                    id + ".column",
                                    new Integer(colindex));

                                return;
                            }
                        }
                    }
                }

                if (debug) {
                    System.err.println("No match found");
                }

                getXMLTestCase().addProperty(id, null);
                getXMLTestCase().addProperty(
                    id + ".column",
                    new Integer(-1));
                getXMLTestCase().addProperty(
                    id + ".row",
                    new Integer(-1));
            } else {
                Assert.fail("Invalid Table type:" + o.toString());
            }
        } else if (type.equals(LISTITEM)) {
            ListModel model = null;

            if (o instanceof JList) {
                model = ((JList) o).getModel();
            } else if (o instanceof JComboBox) {
                model = ((JComboBox) o).getModel();
            }

            boolean found = false;

            int     rowindex = 0;

            //System.out.println("rc:"+ovt.getRowCount()+" cc:"+ovt.getColumnCount());
            for (rowindex = 0; (rowindex < model.getSize()) && !found;
                    rowindex++) {
                Object item     = model.getElementAt(rowindex);
                String listitem = null;

                if (item != null) {
                    listitem = item.toString();
                }

                if (debug) {
                    System.err.println("Evaluate index=" + rowindex + " value="
                        + listitem);
                }

                if (evaluate(op, listitem, expectedval)) {
                    if (debug) {
                        System.err.println("Match." + (findcount - 1)
                            + " more items to find");
                    }

                    if (--findcount == 0) {
                        getXMLTestCase().addProperty(id, item);
                        getXMLTestCase().addProperty(
                            id + ".index",
                            new Integer(rowindex));

                        return;
                    }
                }
            }

            if (debug) {
                System.err.println("List item match not found.");
            }

            getXMLTestCase().addProperty(id, null);
            getXMLTestCase().addProperty(
                id + ".index",
                new Integer(-1));
        } else {
            Assert.fail("Unknown object type: " + o);
        }
    }

    /**
     * Insure that the id, refid, type and value attributes
     * are specified.
     * @throws XMLException if one or more of the above attributes are missing.
     */
    public void validateElement() throws XMLException {
        super.validateElement();

        // check the element tag name
        // id is a required attribute
        checkRequiredAttribute(ID);
        checkRequiredAttribute(REFID);
        checkRequiredAttribute(TYPE);
        checkRequiredAttribute(VALUE);
    }

    /**
     * Get the case independent attribute.
     * @return boolean true if case should be ignored.
     */
    protected boolean getCaseIndependent() {
        return getBoolean("caseindependent", true);
    }

    /**
     * Get the operation from the string value.
     * @return int representing the operation.
     */
    protected int getOperation() {
        String operation = getString("operation");

        if (operation == null) {
            return Operator.Operation.EQUALS;
        }

        return Operator.Operation.getOperation(operation);
    }

    /**
     * Evaluate the operation.
     * @param operation Operation to be evaluated.
     * @param tableval Actual value.
     * @param expectedval expected value.
     * @return true if a match is found.
     */
    private boolean evaluate(final int operation, final String tableval,
        final String expectedval) {
        return Operator.evaluate(
            tableval,
            expectedval,
            operation,
            getCaseIndependent());
    }
}
